/* ********************************************************************** */
/*                            UPDATE 2.01.03_001                          */
/* ---------------------------------------------------------------------- */

-- This function already exists, but (may) has been changed

/**
 * FUNCTION billing.update_rate
 * 
 * Updates the rate with the specified code, if it exists exactly one time.
 * Otherwise no error is thrown and the function won't update, but raise a notice.
 * 
 * p_Code: The name of the code to update
 * p_OldCoefficient: the value of the old coefficient
 * p_NewCoefficient: the value of the new coefficient
 * p_Applicability: the date, since when the change takes affect
 * p_KeyValueLabel: the label of the codes key value
 */
CREATE OR REPLACE FUNCTION billing.update_rate (p_Code VARCHAR, p_OldCoefficient FLOAT, p_NewCoefficient FLOAT, p_Applicability DATE)
  RETURNS void AS $BODY$
DECLARE
  codeId INTEGER;

BEGIN
  -- check existence of code
  BEGIN
    codeId := id FROM billing.rate WHERE code = p_Code;

    IF codeId IS NULL THEN
      RAISE NOTICE 'Code % not found. Cannot update!', p_Code;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Multiple codes % found. Cannot update!', p_Code;
    RETURN;
  END;

  -- do the update
  UPDATE billing.rate
  SET old_coefficient = p_OldCoefficient,
      new_coefficient = p_NewCoefficient,
      applicability   = p_Applicability
  WHERE id = codeId
  AND applicability < p_Applicability;

  RETURN;
END $BODY$
LANGUAGE plpgsql VOLATILE;


/**
 * FUNCTION billing.add_rate_index
 * 
 * Adds a new rate index, if it doesn't exist, yet.
 * Otherwise no error is thrown, but a notice.
 * 
 * p_IndexLabel: The label of the new rate index
 * p_ParentIndexLabel: The label of the parent rate index
 * p_Description: The description of the new rate index
 */
CREATE OR REPLACE FUNCTION billing.add_rate_index (p_IndexLabel VARCHAR, p_ParentIndexLabel VARCHAR, p_Description VARCHAR)
  RETURNS void AS $BODY$
DECLARE
  indexId INTEGER;
  
BEGIN
  BEGIN
    -- check existence of rate index
    indexId := id FROM billing.rate_index 
        WHERE label = p_IndexLabel;

    IF indexId IS NOT NULL THEN
      RAISE NOTICE 'Rate index % already exists.', p_IndexLabel;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Rate index % already exists (multiple times !!!).', p_IndexLabel;
    RETURN;
  END;
  
  BEGIN
    -- check existence of parent rate index
    indexId := id FROM billing.rate_index 
        WHERE label = p_ParentIndexLabel;

    IF indexId IS NULL THEN
      RAISE NOTICE 'Parent rate index % does NOT exist. Cannot add rate index %', p_ParentIndexLabel, p_IndexLabel;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Parent rate index % does exist multiple times. Cannot add rate index %', p_ParentIndexLabel, p_IndexLabel;
    RETURN;
  END;
  
  -- insert the rate index
  INSERT INTO billing.rate_index
  (label, parent_id, title)
  VALUES
  (p_IndexLabel, indexId, p_Description);
  
  RETURN;
END $BODY$
LANGUAGE plpgsql VOLATILE;


/**
 * FUNCTION billing.add_rate
 * 
 * Adds a rate with the specified parameter, if it doesn't exist, yet.
 * Otherwise no error is thrown and the function won't add a rate, but raise a notice.
 * 
 * p_Code: The name of the code to update
 * p_RateIndexLabel: the name of the rate index
 * p_KeyValueLabel: the label of the codes key value
 * p_Description: the code description
 * p_NewCoefficient: the value of the new coefficient
 * p_Applicability: the date, since when the change takes affect
 */
CREATE OR REPLACE FUNCTION billing.add_rate (p_Code VARCHAR, p_RateIndexLabel VARCHAR, p_KeyValueLabel VARCHAR, p_Description VARCHAR, p_NewCoefficient FLOAT, p_Applicability DATE)
  RETURNS void AS $BODY$
DECLARE
  keyValueId INTEGER;
  rateIndexId INTEGER;
  codeId INTEGER;

BEGIN
  -- check existence of the key value
  BEGIN
    keyValueId := id FROM billing.key_value WHERE LOWER(label) SIMILAR TO p_KeyValueLabel;
    
    IF p_KeyValueLabel IS NULL THEN
      RAISE INFO 'No key value found with label ''%''.', p_KeyValueLabel;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE INFO 'Multiple key values found with label ''%''.', p_KeyValueLabel;
  END;
  
  IF keyValueId IS NULL THEN
    -- try to detect default ...
    IF p_KeyValueLabel = '(ucm|cns) m(e|é|É)decins' THEN
      keyValueId = 1;
      RAISE INFO 'Using default key value ID (1) for rate %.', p_Code;
    ELSIF p_KeyValueLabel = 'Non Conventioné' THEN
      keyValueId = 0;
      RAISE INFO 'Using default key value ID (0) for rate %.', p_Code;
    ELSE
      RAISE NOTICE 'Cannot add rate ''%''!', p_Code;
      RETURN;
    END IF;

    keyValueId := id FROM billing.key_value WHERE id = keyValueId;
    IF keyValueId IS NULL THEN
      RAISE NOTICE 'Cannot add rate ''%''!', p_Code;
      RETURN;
    END IF;
  END IF;
  
  -- check existence of rate index
  BEGIN
    rateIndexId := id FROM billing.rate_index WHERE label = p_RateIndexLabel;

    IF rateIndexId IS NULL THEN
      RAISE NOTICE 'Rate index ''%'' not found. Cannot add rate %!', p_RateIndexLabel, p_Code;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Multiple results found for rate index ''%''. Cannot add rate %!', p_RateIndexLabel, p_Code;
    RETURN;
  END;
  
  -- check if code already exists
  BEGIN
    codeId := id FROM billing.rate WHERE code = p_Code;

    IF codeId IS NOT NULL THEN
      RAISE NOTICE 'Rate % already exists. Cannot add rate!', p_Code;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Rate % already exists multiple times. Cannot add rate!', p_Code;
    RETURN;
  END;
  
  -- do the insert
  INSERT INTO billing.rate
  (key_value_id, index_id, code, label, cat, cac, apcm, acm, old_coefficient, applicability, new_coefficient)
  VALUES
  (keyValueId, rateIndexId, p_Code, p_Description, FALSE, FALSE, FALSE, FALSE, 0.0, p_Applicability, p_NewCoefficient);
  
  RETURN;
END $BODY$
LANGUAGE plpgsql VOLATILE;



-- -----------------------
-- DO THE ADDS AND UPDATES
-- -----------------------

-- Updated rates
SELECT billing.update_rate('V1',  12.48, 14.56, '2015-01-01');

SELECT billing.update_rate('E3',   7.25,  8.00, '2015-01-01');
SELECT billing.update_rate('E4',   7.25,  8.00, '2015-01-01');
SELECT billing.update_rate('E5',   7.25,  8.00, '2015-01-01');
SELECT billing.update_rate('E6',   7.25,  8.00, '2015-01-01');
SELECT billing.update_rate('E7',   7.25,  8.00, '2015-01-01');


-- Added rate indices
SELECT billing.add_rate_index ('G_2_1_3', 'G_2_1', 'Médecins généralistes et médecins spécialistes en gériatrie en cas de visite auprès d’un patient relevant de l’assurance dépendance ou déclaré «soins palliatifs» ou en cas de certificat de décès');
SELECT billing.add_rate_index ('G_10',    'G',     'Consultations dans le cadre de réunion de concertation pluridisciplinaire en cancérologie');

-- Added rates
SELECT billing.add_rate('C40',  'G_1_2',   '(ucm|cns) m(e|é|É)decins', 'Consultation majorée du médecin spécialiste en dermatologie',                                                                                                   15.34, '2015-01-01');

SELECT billing.add_rate('V101', 'G_2_1_3', '(ucm|cns) m(e|é|É)decins', 'Visite du médecin généraliste et du médecin spécialiste en gériatrie',                                                                                          18.72, '2015-01-01');
SELECT billing.add_rate('V301', 'G_2_1_3', '(ucm|cns) m(e|é|É)decins', 'Visite urgente',                                                                                                                                                18.72, '2015-01-01');
SELECT billing.add_rate('V401', 'G_2_1_3', '(ucm|cns) m(e|é|É)decins', 'Visite du soir demandée et faite entre 18 et 22 heures',                                                                                                        18.72, '2015-01-01');
SELECT billing.add_rate('V501', 'G_2_1_3', '(ucm|cns) m(e|é|É)decins', 'Visite demandée et faite le samedi après 12 heures',                                                                                                            18.72, '2015-01-01');
SELECT billing.add_rate('V601', 'G_2_1_3', '(ucm|cns) m(e|é|É)decins', 'Visite demandée et faite le dimanche ou un jour férié légal',                                                                                                   18.72, '2015-01-01');
SELECT billing.add_rate('V701', 'G_2_1_3', '(ucm|cns) m(e|é|É)decins', 'Visite de nuit demandée et faite entre 22 et 7 heures',                                                                                                         25.32, '2015-01-01');

SELECT billing.add_rate('F111', 'G_4_1',   '(ucm|cns) m(e|é|É)decins', '1er jour d’hospitalisation, le dimanche ou un jour férié légal',                                                                                                 6.86, '2015-01-01');
SELECT billing.add_rate('F121', 'G_4_1',   '(ucm|cns) m(e|é|É)decins', '2e au 14e jour d’hospitalisation, le dimanche ou un jour férié légal',                                                                                           6.86, '2015-01-01');
SELECT billing.add_rate('F201', 'G_4_2',   '(ucm|cns) m(e|é|É)decins', '1er jour d’hospitalisation en cas de malade transféré à un médecin spécialiste, le dimanche ou un jour férié légal',                                            40.92, '2015-01-01');
SELECT billing.add_rate('F211', 'G_4_2',   '(ucm|cns) m(e|é|É)decins', '1er jour d’hospitalisation (malade non transféré), le dimanche ou un jour férié légal',                                                                         14.54, '2015-01-01');
SELECT billing.add_rate('F221', 'G_4_2',   '(ucm|cns) m(e|é|É)decins', '2e au 14e jour d’hospitalisation, le dimanche ou un jour férié légal',                                                                                          11.03, '2015-01-01');
SELECT billing.add_rate('F251', 'G_4_2',   '(ucm|cns) m(e|é|É)decins', '1er jour d’hospitalisation d’un malade transféré à un médecin spécialiste en médecine interne ou en oncologie, le dimanche ou un jour férié légal',             51.21, '2015-01-01');
SELECT billing.add_rate('F261', 'G_4_2',   '(ucm|cns) m(e|é|É)decins', '1er jour d’hospitalisation par un médecin spécialiste en médecine interne ou en oncologie (malade non transféré), le dimanche ou un jour férié légal',          25.60, '2015-01-01');
SELECT billing.add_rate('F271', 'G_4_2',   '(ucm|cns) m(e|é|É)decins', '1er jour d’hospitalisation d’un enfant de moins de 14 ans transféré à un médecin spécialiste en pédiatrie, le dimanche ou un jour férié légal',                 40.92, '2015-01-01');
SELECT billing.add_rate('F281', 'G_4_2',   '(ucm|cns) m(e|é|É)decins', '1er jour d’hospitalisation d’un enfant de moins de 14 ans par un médecin spécialiste en pédiatrie (malade non transféré), le dimanche ou un jour férié légal',  14.54, '2015-01-01');
SELECT billing.add_rate('F311', 'G_4_3',   '(ucm|cns) m(e|é|É)decins', '1er au 7e jour d’hospitalisation, le dimanche ou un jour férié légal',                                                                                           5.85, '2015-01-01');
SELECT billing.add_rate('F321', 'G_4_3',   '(ucm|cns) m(e|é|É)decins', '8e au 14e jour d’hospitalisation, le dimanche ou un jour férié légal',                                                                                           2.91, '2015-01-01');
SELECT billing.add_rate('F511', 'G_4_5',   '(ucm|cns) m(e|é|É)decins', '1er et 2e jour de soins intensifs, le dimanche ou un jour férié légal',                                                                                         69.15, '2015-01-01');
SELECT billing.add_rate('F521', 'G_4_5',   '(ucm|cns) m(e|é|É)decins', '3e au 6e jour de soins intensifs, le dimanche ou un jour férié légal',                                                                                          35.07, '2015-01-01');
SELECT billing.add_rate('F611', 'G_4_6_1', '(ucm|cns) m(e|é|É)decins', '1er et 2e jour de soins intensifs, le dimanche ou un jour férié légal',                                                                                         74.82, '2015-01-01');
SELECT billing.add_rate('F621', 'G_4_6_1', '(ucm|cns) m(e|é|É)decins', 'A partir du 3e jour de soins intensifs, le dimanche ou un jour férié légal',                                                                                    29.16, '2015-01-01');
SELECT billing.add_rate('F651', 'G_4_6_2', '(ucm|cns) m(e|é|É)decins', '1er et 2e jour de soins intensifs post-opératoires, le dimanche ou un jour férié légal',                                                                        44.85, '2015-01-01');
SELECT billing.add_rate('F661', 'G_4_6_2', '(ucm|cns) m(e|é|É)decins', 'A partir du 3e jour de soins intensifs post-opératoires, le dimanche ou un jour férié légal',                                                                   44.85, '2015-01-01');
SELECT billing.add_rate('F681', 'G_4_6_3', '(ucm|cns) m(e|é|É)decins', 'Traitement par anesthésie continue, le dimanche ou un jour férié légal',                                                                                        29.16, '2015-01-01');
SELECT billing.add_rate('F691', 'G_4_6_4', '(ucm|cns) m(e|é|É)decins', 'Traitement de la douleur aiguë post-opératoire, le dimanche ou un jour férié légal',                                                                            29.16, '2015-01-01');
SELECT billing.add_rate('F711', 'G_4_7_1', '(ucm|cns) m(e|é|É)decins', '1er et 2e jour de réanimation, le dimanche ou un jour férié légal',                                                                                            170.43, '2015-01-01');
SELECT billing.add_rate('F721', 'G_4_7_1', '(ucm|cns) m(e|é|É)decins', '3e et 4e jour de réanimation, le dimanche ou un jour férié légal',                                                                                              85.22, '2015-01-01');
SELECT billing.add_rate('F731', 'G_4_7_1', '(ucm|cns) m(e|é|É)decins', 'A partir du 5e jour de réanimation, le dimanche ou un jour férié légal',                                                                                        51.05, '2015-01-01');
SELECT billing.add_rate('F751', 'G_4_7_2', '(ucm|cns) m(e|é|É)decins', '1er et 2e jour de réanimation post-opératoire, le dimanche ou un jour férié légal',                                                                             86.25, '2015-01-01');
SELECT billing.add_rate('F761', 'G_4_7_2', '(ucm|cns) m(e|é|É)decins', '3e et 4e jour de réanimation post-opératoire, le dimanche ou un jour férié légal',                                                                              86.25, '2015-01-01');
SELECT billing.add_rate('F771', 'G_4_7_2', '(ucm|cns) m(e|é|É)decins', 'A partir du 5e jour de réanimation post-opératoire, le dimanche ou un jour férié légal',                                                                        86.25, '2015-01-01');
SELECT billing.add_rate('F801', 'G_4_8',   '(ucm|cns) m(e|é|É)decins', 'Forfait le dimanche ou un jour férié légal',                                                                                                                    87.09, '2015-01-01');
SELECT billing.add_rate('F851', 'G_4_9',   '(ucm|cns) m(e|é|É)decins', 'Forfait le dimanche ou un jour férié légal',                                                                                                                    26.85, '2015-01-01');

SELECT billing.add_rate('J2',   'G_8',     '(ucm|cns) m(e|é|É)decins', 'Forfait par demi-journée pour un enfant présent au centre de jour du service national de psychiatrie juvénile',                                                 12.28, '2015-01-01');

SELECT billing.add_rate('P1',   'G_10',    '(ucm|cns) m(e|é|É)decins', 'Consultation du médecin spécialiste rapporteur participant à la réunion de concertation pluridisciplinaire en cancérologie',                                    35.00, '2011-01-01');
SELECT billing.add_rate('P2',   'G_10',    '(ucm|cns) m(e|é|É)decins', 'Consultation du médecin spécialiste participant à la réunion de concertation pluridisciplinaire en cancérologie',                                               26.00, '2011-01-01');
SELECT billing.add_rate('P3',   'G_10',    '(ucm|cns) m(e|é|É)decins', 'Consultation du médecin généraliste participant à la réunion de concertation pluridisciplinaire en cancérologie',                                               30.00, '2011-01-01');

SELECT billing.add_rate('7A96', 'T_7_5',   '(ucm|cns) m(e|é|É)decins', 'Anesthésie péridurale pour accouchement de nuit, de dimanche, de jour férié légal',                                                                            111.81, '2009-03-01');


-- Rates removed
SELECT billing.update_rate('J7',   2.58,  0.00, '2011-01-01');
SELECT billing.update_rate('J8',   5.15,  0.00, '2011-01-01');

/**** UPDATE THE SCRIPTNAME BELOW TO THE FILENAME OF THIS FILE !!!!!!!! *****/
INSERT INTO core.info (date,key,value) VALUES ('now', 'LAST_UPDATE', 'db_update_2.01.03_001.sql');