/* ********************************************************************** */
/*                            UPDATE 2.00.07_002                          */
/* ---------------------------------------------------------------------- */

-- Add and update billing rates


/**
 * FUNCTION billing.update_rate
 * 
 * Updates the rate with the specified code, if it exists exactly one time.
 * Otherwise no error is thrown and the function won't update, but raise a notice.
 * 
 * p_Code: The name of the code to update
 * p_OldCoefficient: the value of the old coefficient
 * p_NewCoefficient: the value of the new coefficient
 * p_Applicability: the date, since when the change takes affect
 * p_KeyValueLabel: the label of the codes key value
 */
CREATE OR REPLACE FUNCTION billing.update_rate (p_Code VARCHAR, p_OldCoefficient FLOAT, p_NewCoefficient FLOAT, p_Applicability DATE)
  RETURNS void AS $BODY$
DECLARE
  codeId INTEGER;

BEGIN
  -- check existence of code
  BEGIN
    codeId := id FROM billing.rate WHERE code = p_Code;

    IF codeId IS NULL THEN
      RAISE NOTICE 'Code % not found. Cannot update!', p_Code;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Multiple codes % found. Cannot update!', p_Code;
    RETURN;
  END;

  -- do the update
  UPDATE billing.rate
  SET old_coefficient = p_OldCoefficient,
      new_coefficient = p_NewCoefficient,
      applicability   = p_Applicability
  WHERE id = codeId
  AND applicability < p_Applicability;

  RETURN;
END $BODY$
LANGUAGE plpgsql VOLATILE;

/**
 * FUNCTION billing.add_rate
 * 
 * Adds a rate with the specified parameter, if it doesn't exist, yet.
 * Otherwise no error is thrown and the function won't add a rate, but raise a notice.
 * 
 * p_Code: The name of the code to update
 * p_RateIndexLabel: the name of the rate index
 * p_KeyValueLabel: the label of the codes key value
 * p_Description: the code description
 * p_NewCoefficient: the value of the new coefficient
 * p_Applicability: the date, since when the change takes affect
 */
CREATE OR REPLACE FUNCTION billing.add_rate (p_Code VARCHAR, p_RateIndexLabel VARCHAR, p_KeyValueLabel VARCHAR, p_Description VARCHAR, p_NewCoefficient FLOAT, p_Applicability DATE)
  RETURNS void AS $BODY$
DECLARE
  keyValueId INTEGER;
  rateIndexId INTEGER;
  codeId INTEGER;

BEGIN
  -- check existence of the key value
  BEGIN
    keyValueId := id FROM billing.key_value WHERE label = p_KeyValueLabel;
    
    IF p_KeyValueLabel IS NULL THEN
      RAISE INFO 'No key value found with label ''%''.', p_KeyValueLabel;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE INFO 'Multiple key values found with label ''%''.', p_KeyValueLabel;
  END;
  
  IF keyValueId IS NULL THEN
    -- try to detect default ...
    IF p_KeyValueLabel = 'UCM Médecins' THEN
      keyValueId = 1;
      RAISE INFO 'Using default key value ID (1) for rate %.', p_Code;
    ELSIF p_KeyValueLabel = 'Non Conventioné' THEN
      keyValueId = 0;
      RAISE INFO 'Using default key value ID (0) for rate %.', p_Code;
    ELSE
      RAISE NOTICE 'Cannot add rate ''%''!', p_Code;
      RETURN;
    END IF;

    keyValueId := id FROM billing.key_value WHERE id = keyValueId;
    IF keyValueId IS NULL THEN
      RAISE NOTICE 'Cannot add rate ''%''!', p_Code;
      RETURN;
    END IF;
  END IF;
  
  -- check existence of rate index
  BEGIN
    rateIndexId := id FROM billing.rate_index WHERE label = p_RateIndexLabel;

    IF rateIndexId IS NULL THEN
      RAISE NOTICE 'Rate index ''%'' not found. Cannot add rate %!', p_RateIndexLabel, p_Code;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Multiple results found for rate index ''%''. Cannot add rate %!', p_RateIndexLabel, p_Code;
    RETURN;
  END;
  
  -- check if code already exists
  BEGIN
    codeId := id FROM billing.rate WHERE code = p_Code;

    IF codeId IS NOT NULL THEN
      RAISE NOTICE 'Rate % already exists. Cannot add rate!', p_Code;
      RETURN;
    END IF;
  EXCEPTION WHEN CARDINALITY_VIOLATION THEN
    RAISE NOTICE 'Rate % already exists multiple times. Cannot add rate!', p_Code;
    RETURN;
  END;
  
  -- do the insert
  INSERT INTO billing.rate
  (key_value_id, index_id, code, label, cat, cac, apcm, acm, old_coefficient, applicability, new_coefficient)
  VALUES
  (keyValueId, rateIndexId, p_Code, p_Description, FALSE, FALSE, FALSE, FALSE, 0.0, p_Applicability, p_NewCoefficient);
  
  RETURN;
END $BODY$
LANGUAGE plpgsql VOLATILE;



-- -----------------------
-- DO THE ADDS AND UPDATES
-- -----------------------

-- update the rate E8 - E13
SELECT billing.update_rate('E8',  10.25, 15.08, '2014-06-01');
SELECT billing.update_rate('E9',  10.25, 15.08, '2014-06-01');
SELECT billing.update_rate('E10', 10.25, 15.08, '2014-06-01');
SELECT billing.update_rate('E11', 10.25, 15.08, '2014-06-01');
SELECT billing.update_rate('E12', 10.25, 15.08, '2014-06-01');
SELECT billing.update_rate('E13', 10.25, 15.08, '2014-06-01');
-- update the rate E18 & E19
SELECT billing.update_rate('E18',  9.03, 15.08, '2014-06-01');
SELECT billing.update_rate('E19',  9.03, 15.08, '2014-06-01');
-- remove rate E30, because it doesn't exist anymore
SELECT billing.update_rate('E30',  8.70,  0.0, '2014-06-01');
-- add the new rate E60
SELECT billing.add_rate('E60', 'G_6_4', 'UCM Médecins', 'Consultation suivie de l''établissement de la fiche de prévention validée par la direction de la santé', 21.0, '2014-06-01');

-- add the convenances personnelles rates
SELECT billing.add_rate('CP1', 'N_C', 'Non Conventioné', 'Rendez-vous fixé à la demande expresse du patient à un jour et une heure précis à condition que le rendez-vous ait été respecté par le médecin.', 1.0, '2010-01-01');
SELECT billing.add_rate('CP2', 'N_C', 'Non Conventioné', 'Un rendez-vous fixé à la demande expresse du patient un samedi-matin et donné par un médecin qui travaille du lundi au vendredi.', 1.0, '2010-01-01');
SELECT billing.add_rate('CP3', 'N_C', 'Non Conventioné', 'Fait que le patient vient trop tard à son rendez-vous sans fournir d''excuse valable.', 1.0, '2010-01-01');
SELECT billing.add_rate('CP4', 'N_C', 'Non Conventioné', 'Rendez-vous fixé à la demande expresse du patient à un jour et une heure précis après que deux propositions faites par le médecin n''ont pas été acceptées.', 1.0, '2010-01-01');
SELECT billing.add_rate('CP5', 'N_C', 'Non Conventioné', 'L''examen immédiat du patient sans qu''il ait été fixé de rendez-vous préalable, sauf en cas d''urgence (l''indemnité pouvant être réclamée est égale à la différence entre le tarif de la consultation ou la visite normale et celui de la consultation ou la visite d''urgence. En cas de visite, les frais de déplacement éventuels sont inclus dans l''indemnité.', 1.0, '2010-01-01');
SELECT billing.add_rate('CP6', 'N_C', 'Non Conventioné', 'Fait par le patient de se faire attribuer des soins lors d''une consultation ou visite d''urgence, telles que celles-ci sont définies dans la nomenclature des médecins et des médecins-dentistes, sans que le caractère urgent n''ait été reconnu par le médecin.', 1.0, '2010-01-01');
SELECT billing.add_rate('CP7', 'N_C', 'Non Conventioné', 'Fait par le patient de solliciter les conseils du médecin par téléphone, si la durée de l''entretien dépasse dix minutes au moins.', 1.0, '2010-01-01');


/**** UPDATE THE SCRIPTNAME BELOW TO THE FILENAME OF THIS FILE !!!!!!!! *****/
INSERT INTO "core"."info" (date,key,value) VALUES ('now','LAST_UPDATE' ,'db_update_2.00.07_002.sql');