package lu.tudor.santec.org.fife.ui.autocomplete;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.KeyStroke;
import javax.swing.text.JTextComponent;

import com.jgoodies.forms.layout.CellConstraints;

/**
 * @author ferring
 *
 */
public class AutoCompletionTextComponent extends JComponent
{
	/* ======================================== */
	// 		PRIVATE FINAL MEMBERS
	/* ======================================== */
	
	private static final long serialVersionUID = 1L;
	
	public static final Color 	SUCCESS_HIGHLIGHTING_COLOR 	= Color.GREEN;
	public static final Color 	FAILED_HIGHLIGHTING_COLOR 	= Color.RED;
	public static final long 	DEFAULT_HIGHLIGHTING_TIME 	= 100;
	
	public static boolean QUICK_ADD_ENABLED_BY_DEFAULT 		 = true;
	public static boolean SAVE_COMPLETION_ENABLED_BY_DEFAULT = true;
	public static boolean EDIT_COMPLETION_ENABLED_BY_DEFAULT = true;
	public static boolean EDIT_HOT_KEYS_ENABLED_BY_DEFAULT 	 = true;
//	public static boolean AUTO_FILL_ENABLED_BY_DEFAULT 		 = true;
	
	/* ======================================== */
	// 		PRIVATE MEMBERS
	/* ======================================== */
	
	protected final JTextComponent 	textComponent;
	protected JLabel				menuLabel;
	
	protected final JScrollPane		scrollPane 				= createScrollPane();
	
	protected MWCompletionProvider	provider;
	protected MWAutoCompletion		autoCompletion;
	
	protected MWACHandler 			handler;
	protected MWACCHotKeyHandler 	hotKeyHandler;
	protected MWACMenuHandler 		menuHandler;
	
	protected Color 				selectionColor;
	
	protected CellConstraints		cc 						= new CellConstraints();
	
	protected boolean 				quickAddEnabled 		= QUICK_ADD_ENABLED_BY_DEFAULT;
	protected boolean 				saveCompletionEnabled 	= SAVE_COMPLETION_ENABLED_BY_DEFAULT;
	protected boolean 				editCompletionEnabled 	= EDIT_COMPLETION_ENABLED_BY_DEFAULT;
	protected boolean				editHotKeysEnabled 		= EDIT_HOT_KEYS_ENABLED_BY_DEFAULT;
//	private boolean					autoFillEnabled			= true;

	private static MWACListener 	defaultAutoCompletionListener;
	
	private List<MWHotKeyListener> 	hotKeyListeners 		= new LinkedList<MWHotKeyListener>();
	
	
	
	
	/* ======================================== */
	// 		CONSTRUCTORS
	/* ======================================== */
	
	/**
	 * Creates a JComponent, you hand over a JTextComponent, that is placed into the center of this
	 * component. You will be able to use AutoCompletion not only on single words, but additionally
	 * on longer expressions, starting from the last carriage return. 
	 * 
	 * @param textComponent The text component
	 * @param completions a list of text, that is added to the provider as completions
	 */
	public AutoCompletionTextComponent (JTextComponent textComponent, String[] words)
	{
		this (textComponent, new MWCompletionProvider(null, words), null);
	}

	/**
	 * Creates a JComponent, you hand over a JTextComponent, that is placed into the center of this
	 * component. You will be able to use AutoCompletion not only on single words, but additionally
	 * on longer expressions, starting from the last carriage return. 
	 * 
	 * @param textComponent The text component
	 */
	public AutoCompletionTextComponent (JTextComponent textComponent)
	{
		this(textComponent, null, null);
	}

	/**
	 * Creates a JComponent, you hand over a JTextComponent, that is placed into the center of this
	 * component. You will be able to use AutoCompletion not only on single words, but additionally
	 * on longer expressions, starting from the last carriage return. 
	 * 
	 * @param textComponent The text component
	 * @param provider The completion provider, that contains the completions
	 */
	public AutoCompletionTextComponent (JTextComponent textComponent, MWCompletionProvider provider)
	{
		this(textComponent, provider, null);
	}

	/**
	 * Creates a JComponent, you hand over a JTextComponent, that is placed into the center of this
	 * component. You will be able to use AutoCompletion not only on single words, but additionally
	 * on longer expressions, starting from the last carriage return. 
	 * 
	 * @param textComponent The text component
	 * @param context According to the context, the provider will be chosen. Having the same context
	 * means having the same provider and therefore also the same completions.
	 */
	public AutoCompletionTextComponent (JTextComponent textComponent, String context)
	{
		this(textComponent, 
				defaultAutoCompletionListener != null 
					? defaultAutoCompletionListener.getProviderByContext(context)
					: null, 
				null);
	}
	
	/**
	 * Creates a JComponent, you hand over a JTextComponent, that is placed into the center of this
	 * component. You will be able to use AutoCompletion not only on single words, but additionally
	 * on longer expressions, starting from the last carriage return. 
	 * 
	 * @param textComponent The text component
	 * @param provider The completion provider, that contains the completions
	 * @param completions a list of completions, that is added to the provider
	 */
	public AutoCompletionTextComponent (JTextComponent textComponent, MWCompletionProvider provider, List<Completion> completions)
	{
		if (textComponent == null)
			throw new NullPointerException("The text component must not be null!");
		
		this.textComponent = textComponent;
		
		// initialize the provider
		if (provider == null)
			this.provider = new MWCompletionProvider(null);
		else
			this.provider = provider;
		
		if (completions != null)
			this.provider.addCompletions(completions);
		
		initialze(textComponent);
	}
	
	
	
	/* ======================================== */
	// 		HELPER
	/* ======================================== */
	
	/**
	 * Initializes this component. Is supposed to be called by the constructor only.
	 * 
	 * @param component the text component
	 */
	protected void initialze (JTextComponent component)
	{
		Icon menuIcon = IconManager.getIcon(IconManager.MENU_ICON_NAME);
		
		// define the layout of the component using a FormLayout
		this.setLayout(new BorderLayout());
		
		// create a scroll pane around the component, if needed 
		// => JTextFields normally doesn't need a scroll pane
		this.enableScrollPane(textComponent instanceof JTextArea
				|| textComponent instanceof JEditorPane);
		
		selectionColor = textComponent.getSelectionColor();
		
		// initialize auto completion
		installAutoCompletion();
		
		menuLabel 		= new JLabel ();
		menuLabel.setBackground(Color.LIGHT_GRAY);
		menuLabel.setIcon(menuIcon);
		this.add(menuLabel, BorderLayout.EAST);
		
		// initialize the controller
		handler = new MWACHandler(this);
		
		hotKeyHandler 	= new MWACCHotKeyHandler(this, handler);
		textComponent.addKeyListener(hotKeyHandler);
		
		menuHandler 	= new MWACMenuHandler(this, handler);
		menuLabel.addMouseListener(menuHandler);
	}
	
	protected void installAutoCompletion ()
	{
		if (provider == null 
				|| provider.getContext() == null 
				|| provider.getContext().equals(""))
			return;
		
		if (autoCompletion != null)
			autoCompletion.uninstall();
		autoCompletion = new MWAutoCompletion(provider);
		autoCompletion.install(textComponent);
	}
	
	
	public void uninstallAutoCompletion ()
	{
		if (autoCompletion != null)
			autoCompletion.uninstall();
	}
	
	
	/* ======================================== */
	// 		GETTER & SETTER / ADDER & REMOVER
	/* ======================================== */
	
	/**
	 * Sets the selection color of the component.
	 * 
	 * @param c The new selection color 
	 */
	public void setSelectionColor(Color c)
	{
		this.selectionColor = c;
		textComponent.setSelectionColor(c);
	}
	
	
	
	/**
	 * @return The provider that contains the completions
	 */
	public MWCompletionProvider getProvider ()
	{
		return provider;
	}
	
	/**
	 * Searches a provider with the given context by delegating 
	 * it to the default MWACListener. Afterwards the autocompletion
	 * with this provider is installed. 
	 * @param context the context of the new provider
	 */
	public void changeProvider (String context)
	{
		uninstallAutoCompletion();
		this.provider = defaultAutoCompletionListener.getProviderByContext(context);
		installAutoCompletion();
	}
	
	
	/**
	 * Adds a completion to the belonging provider.
	 * 
	 * @param replacementText The text that will inserted
	 * @return <code>true</code> if the completion was added successfully, else <code>false</code>
	 */
	public boolean addCompletion (String replacementText)
	{
		return addCompletion(new MultiWordCompletion(provider, replacementText));
	}

	/**
	 * Adds a completion to the belonging provider.
	 * 
	 * @param shortCut The text that needs to be entered, to show the completion
	 * @param replacementText The text that will inserted
	 * @return <code>true</code> if the completion was added successfully, else <code>false</code>
	 */
	public boolean addCompletion (String shortCut, String replacementText)
	{
		return addCompletion(new MultiWordCompletion(provider, shortCut, replacementText));
	}
	
	
	/**
	 * Adds a completion to the belonging provider.
	 * 
	 * @param completion The completion to add
	 * @return <code>true</code> if the completion was added successfully, else <code>false</code>
	 */
	public boolean addCompletion (Completion completion)
	{
		return provider.addCompletionGetFeedback(completion);
	}
	
	/**
	 * Removes a completion from the belonging provider.
	 * 
	 * @param completion
	 * @return <code>true</code> if the completion was found and removed, else <code>false</code>
	 */
	public boolean removeCompletion (Completion completion)
	{
		return provider.removeCompletion(completion);
	}
	
	/**
	 * Changes the completion and replaces it with the given input and replacement text
	 * 
	 * @param oldCompletion
	 * @param newInputText
	 * @param newReplacementText
	 * @return <code>true</code> if the completion was found and changed, else <code>false</code>
	 */
	public boolean changeCompletion (Completion oldCompletion, String newInputText, String newReplacementText)
	{
		MultiWordCompletion newCompletion;
		if (oldCompletion instanceof MultiWordCompletion)
		{
			provider.changeCompletion((MultiWordCompletion) oldCompletion, newInputText, newReplacementText);
			return true;
		}
		else 
		{
			newCompletion = new MultiWordCompletion(provider, newInputText, newReplacementText);
			return provider.changeCompletion(oldCompletion, newCompletion);
		}
	}
	
	/**
	 * Adds a listener that notifies whenever a completion is added, removed, 
	 * changed or the provider needs to be initialized.
	 * 
	 * @param listener The auto completion listener
	 */
	public void addAutoCompletionListener (MWACListener listener)
	{
		provider.addAutoCompletionListener(listener);
	}
	
	/**
	 * Removes a listener that notifies whenever a completion is added, removed, 
	 * changed or the provider needs to be initialized.
	 * 
	 * @param listener The auto completion listener
	 * @return <code>true</code> if the listener was found and removed, else <code>false</code>
	 */
	public boolean removeAutoCompletionListener (MWACListener listener)
	{
		return provider.removeAutoCompletionListener(listener);
	}
	
	

	/**
	 * Highlights the selected text in the specified color for the specified time.
	 * 
	 * @param color The color in which the text is highlighted
	 * @param millis The milliseconds for which the text is highlighted
	 */
	public void highlightSelectedText (final Color color, final long millis)
	{
		new Thread () {
			@Override
			public void run()
			{
				try
				{
					// set the highlighting color as selection color
					textComponent.setSelectionColor(color);
					// make sure it is painted
					repaint();
					
					// wait the given time
					sleep(millis);
					
				} catch (Exception e)
				{
					e.printStackTrace();
				} finally
				{
					// set the normal selection color as selection color again
					textComponent.setSelectionColor(selectionColor);
					repaint();
				}
			}
		}.start();
	}

	
	/**
	 * @return The hot key to save a new completion
	 */
	public KeyStroke getSaveCompletionKey ()
	{
		return hotKeyHandler.getSaveCompletionKey();
	}
	
	/**
	 * @param key The hot key to save a new completion
	 */
	public void setSaveCompletionKey (KeyStroke key)
	{
		KeyStroke oldKey = hotKeyHandler.getSaveCompletionKey();
		hotKeyHandler.setSaveCompletionKey(key);
		callHotKeyListener(oldKey, key, MWHKEvent.SAVE_COMPLETION_ACTION);
	}
	
	/**
	 * @return The hot key to add the selected text as a completion
	 */
	public KeyStroke getQuickAddCompletionKey()
	{
		return hotKeyHandler.getQuickAddCompletionKey();
	}

	/**
	 * @param key The hot key to add the selected text as a completion
	 */
	public void setQuickAddCompletionKey(KeyStroke key)
	{
		KeyStroke oldKey = hotKeyHandler.getQuickAddCompletionKey();
		hotKeyHandler.setQuickAddCompletionKey(key);
		callHotKeyListener(oldKey, key, MWHKEvent.QUICK_ADD_ACTION);
	}
	
	/**
	 * @return The hot key to open the edit dialog
	 */
	public KeyStroke getEditCompletionKey()
	{
		return hotKeyHandler.getEditCompletionKey();
	}

	/**
	 * @param key The hot key to open the edit dialog
	 */
	public void setEditCompletionKey(KeyStroke key)
	{
		KeyStroke oldKey = hotKeyHandler.getEditCompletionKey();
		hotKeyHandler.setEditCompletionKey(key);
		callHotKeyListener(oldKey, key, MWHKEvent.EDIT_COMPLETION_ACTION);
	}
	
	/**
	 * @param key The hot key to start the auto completion
	 */
	public void setTriggerKey (KeyStroke key) 
	{
		KeyStroke oldKey = getTriggerKey();
		autoCompletion.setTriggerKey(key);
		callHotKeyListener(oldKey, key, MWHKEvent.START_AUTOCOMPLETION_ACTION);
	}
	
	/**
	 * @return The hot key to start the auto completion
	 */
	public KeyStroke getTriggerKey ()
	{
		return autoCompletion.getTriggerKey();
	}
	
	
	/**
	 * Adds a listener that notifies, whenever a hot key changes.
	 * 
	 * @param l The hot key listener to add
	 */
	public void addHotKeyListener (MWHotKeyListener l)
	{
		hotKeyListeners.add(l);
	}
	
	/**
	 * Removes a listener that notifies, whenever a hot key changes.
	 * 
	 * @param l The hot key listener to remove
	 */
	public boolean removeHotKeyListener (MWHotKeyListener l)
	{
		return hotKeyListeners.remove(l);
	}
	
	/**
	 * Calls all hot key listener that a hot key has changed.
	 * 
	 * @param oldHotKey The former hot key
	 * @param newHotKey The new hot key
	 * @param actionForHotKey The type of action, that is executed, when the hot key is pressed,
	 * specified in MWHKEvent
	 */
	private void callHotKeyListener (KeyStroke oldHotKey, KeyStroke newHotKey, int actionForHotKey)
	{
		if (hotKeyListeners.isEmpty())
			return;
		
		MWHKEvent event = new MWHKEvent(this, System.currentTimeMillis(), oldHotKey, newHotKey, actionForHotKey);
		
		for (MWHotKeyListener l : hotKeyListeners)
		{
			l.hotkeyChanged(event);
		}
	}
	
	
	/**
	 * Enables or disables the scroll pane around the text component
	 * 
	 * @param enable if <code>true</code> the scroll pane is enabled
	 * else it will be disabled
	 */
	public void enableScrollPane (boolean enable)
	{
		if (enable)
		{
			this.remove(textComponent);
			scrollPane.setViewportView(textComponent);
//			this.add(scrollPane, cc.xy(1, 1));
			this.add(scrollPane, BorderLayout.CENTER);
		}
		else
		{
			this.remove(scrollPane);
//			this.add(textComponent, cc.xy(1, 1));
			this.add(textComponent, BorderLayout.CENTER);
		}
	}
	
	/**
	 * @return The scroll pane around the text component
	 */
	public JScrollPane getScrollPane ()
	{
		return scrollPane;
	}
	
	
	/**
	 * @param l The default auto completion listener
	 */
	public static void setDefaultAutoCompletionListener (MWACListener l)
	{
		defaultAutoCompletionListener = l;
	}
	
	
	/**
	 * @return The default auto completion listener
	 */
	public static MWACListener getDefaultAutoCompletionListener ()
	{
		return defaultAutoCompletionListener;
	}
	
	
	/**
	 * Sets the translations for the dialog. See <code>Translator</code> for the keys.
	 * 
	 * @param translations A map containing the translations
	 */
	public static void setTranslations (Map<String, String> translations)
	{
		Translator.setTranslations(translations);
	}
	
	/**
	 * @return <code>true</code> if quick add is enabled, else <code>false</code>
	 */
	public boolean isQuickAddEnabled()
	{
		return quickAddEnabled;
	}

	/**
	 * @param enable <code>true</code> to enable the quick add function, 
	 * <code>false</code> to disable it.
	 */
	public void setQuickAddEnabled(boolean enable)
	{
		this.quickAddEnabled = enable;
		menuHandler.showQuickAdd(enable);
	}

	/**
	 * @return <code>true</code> if save completion is enabled, else <code>false</code>
	 */
	public boolean isSaveCompletionEnabled()
	{
		return saveCompletionEnabled;
	}

	/**
	 * @param enable <code>true</code> to enable the save completion function, 
	 * <code>false</code> to disable it.
	 */
	public void setSaveCompletionEnabled(boolean enable)
	{
		this.saveCompletionEnabled = enable;
		menuHandler.showSaveCompletion(enable);
	}

	/**
	 * @return <code>true</code> if edit completions is enabled, else <code>false</code>
	 */
	public boolean isEditCompletionEnabled()
	{
		return editCompletionEnabled;
	}

	/**
	 * @param enable <code>true</code> to enable the edit completions function, 
	 * <code>false</code> to disable it.
	 */
	public void setEditCompletionEnabled(boolean enable)
	{
		this.editCompletionEnabled = enable;
		menuHandler.showEditCompletion(enable);
	}

	/**
	 * @return <code>true</code> if the hot key edit dialog is enabled, else <code>false</code>
	 */
	public boolean isEditHotKeysEnabled()
	{
		return editHotKeysEnabled;
	}

	/**
	 * @param enable <code>true</code> to enable hot key edit dialog function, 
	 * <code>false</code> to disable it.
	 */
	public void setEditHotKeysEnabled(boolean enable)
	{
		this.editHotKeysEnabled = enable;
		menuHandler.showEditHotKeys(enable);
	}

	/**
	 * @return <code>true</code> if the menu is enabled, else <code>false</code>
	 */
	public boolean isMenuEnabled()
	{
		return menuLabel.isVisible();
	}

	/**
	 * @param enable <code>true</code> to enable the menu, 
	 * <code>false</code> to disable it.
	 */ 
	public void setMenuEnabled(boolean menuEnabled)
	{
		menuLabel.setVisible(menuEnabled);
	}

	/**
	 * @return the AutoCompletion used by this component
	 */
	public MWAutoCompletion getAutoCompletion()
	{
		return autoCompletion;
	}

	/**
	 * @return The handler which is used by the key and menu handler
	 * to open the dialogs and execute the operations.
	 */
	public MWACHandler getHandler()
	{
		return handler;
	}

	/**
	 * @return The handler that is used to handle the key events.
	 */
	public MWACCHotKeyHandler getHotKeyHandler()
	{
		return hotKeyHandler;
	}

	/**
	 * @return The handler that is called, when clicking an action in the menu.
	 */
	public MWACMenuHandler getMenuHandler()
	{
		return menuHandler;
	}
	
	/**
	 * @param searchMode see MWCompletionProvider
	 */
	public void setSearchMode (int searchMode)
	{
		provider.setSearchMode(searchMode);
	}
	
	public int getSearchMode ()
	{
		return provider.getSearchMode();
	}
	
	/* ---------------------------------------- */
	// 		GETTER & SETTER OF THE TEXT COMPONENT
	/* ---------------------------------------- */
	
	private JScrollPane createScrollPane ()
	{
		JScrollPane scrollPane = new JScrollPane()
		{
			private static final long serialVersionUID = 1L;

			public void paint(java.awt.Graphics g) 
			{
				int iconWidth = menuLabel == null ? 0 : menuLabel.getWidth();
				setMaximumSize(new Dimension(
						AutoCompletionTextComponent.this.getWidth() - iconWidth,
						AutoCompletionTextComponent.this.getHeight()));
				
				super.paint(g);
			}
		};
		
		scrollPane.setOpaque(false);
		scrollPane.getViewport().setOpaque(false);
		
		return scrollPane;
	}
	
	/**
	 * @return the text, selected by the text component
	 */
	public String getSelectedText ()
	{
		return textComponent.getSelectedText();
	}
	
	/**
	 * @return The text component
	 */
	public JTextComponent getTextComponent ()
	{
		return textComponent;
	}
	
//	@Override
//	public Dimension getPreferredSize()
//	{
//		Dimension size = textComponent.getPreferredSize();
//		size.width += menuLabel.getPreferredSize().width;
//		return size;
//	}
//	
//	@Override
//	public int getWidth()
//	{
//		return textComponent.getWidth() + menuLabel.getWidth();
//	}
//	
//	@Override
//	public Dimension getSize()
//	{
//		Dimension size = textComponent.getSize();
//		size.width += menuLabel.getSize().width;
//		return size;
//	}
}
