package lu.tudor.santec.org.fife.ui.autocomplete;

import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.KeyStroke;

/**
 * @author ferring
 * 
 * This is used to listen to and handle the hot key events.
 */
public class MWACCHotKeyHandler implements KeyListener
{
	/* ======================================== */
	// 		PUBLIC STATIC MEMBERS
	/* ======================================== */
	
	public static final int 	  DEFAULT_KEY_MODIFIER_MASK			= KeyEvent.CTRL_DOWN_MASK | KeyEvent.ALT_DOWN_MASK;
	public static final KeyStroke DEFAULT_KEY_ADD_COMPLETION 		= KeyStroke.getKeyStroke(KeyEvent.VK_S, DEFAULT_KEY_MODIFIER_MASK); // save
	public static final KeyStroke DEFAULT_KEY_QUICK_ADD_COMPLETION	= KeyStroke.getKeyStroke(KeyEvent.VK_A, DEFAULT_KEY_MODIFIER_MASK); // quick add
	public static final KeyStroke DEFAULT_KEY_EDIT_COMPLETION 		= KeyStroke.getKeyStroke(KeyEvent.VK_C, DEFAULT_KEY_MODIFIER_MASK); // change
	
	
	
	/* ======================================== */
	// 		PRIVATE MEMBERS
	/* ======================================== */
	
	private final AutoCompletionTextComponent 	view;
	
	private MWACHandler							handler;
	
	
	/* ---------------------------------------- */
	// 		Keys & Modifiers
	/* ---------------------------------------- */
	private KeyStroke saveCompletionKey 	= DEFAULT_KEY_ADD_COMPLETION;

	private KeyStroke quickAddCompletionKey = DEFAULT_KEY_QUICK_ADD_COMPLETION;
	
	private KeyStroke editCompletionKey 	= DEFAULT_KEY_EDIT_COMPLETION;
	
//	private int changeCompletionKey 		= DEFAULT_KEY_CHANGE_COMPLETION;
//	private int changeCompletionModifiers 	= DEFAULT_KEY_MODIFIER_MASK;
	
	private int actionForThisKeyPerformed 	= 0;
	
	
	/* ======================================== */
	// 		CONSTRUCTOR
	/* ======================================== */
	
	/**
	 * @param textComponent the AutoCompletionTextComponent
	 * @param handler The handler most events are delegated to
	 */
	public MWACCHotKeyHandler(AutoCompletionTextComponent view, MWACHandler handler)
	{
		this.view 		= view;
		this.handler 	= handler;
		
		setSaveCompletionKey(DEFAULT_KEY_ADD_COMPLETION);
		setQuickAddCompletionKey(DEFAULT_KEY_QUICK_ADD_COMPLETION);
		setEditCompletionKey(DEFAULT_KEY_EDIT_COMPLETION);
	}
	
	
	
	/* ======================================== */
	// 		GETTER & SETTER
	/* ======================================== */
	
	/**
	 * @return The text component
	 */
	public AutoCompletionTextComponent getView()
	{
		return view;
	}
	
	/**
	 * @return The KeyStroke to call the save completion dialog
	 */
	public KeyStroke getSaveCompletionKey ()
	{
		return saveCompletionKey;
	}
	
	/**
	 * @param key The KeyStroke to call the save completion dialog
	 */
	public void setSaveCompletionKey(KeyStroke key)
	{
		this.saveCompletionKey = KeyStroke.getKeyStroke(key.getKeyCode(), modify(key.getModifiers()));
	}
	
	/**
	 * @return The KeyStroke to call the quick add dialog
	 */
	public KeyStroke getQuickAddCompletionKey()
	{
		return quickAddCompletionKey;
	}

	/**
	 * @param key The KeyStroke to call the quick add dialog
	 */
	public void setQuickAddCompletionKey(KeyStroke key)
	{
		this.quickAddCompletionKey = KeyStroke.getKeyStroke(key.getKeyCode(), modify(key.getModifiers()));
	}
	
	/**
	 * @return The KeyStroke to call the edit completions dialog
	 */
	public KeyStroke getEditCompletionKey()
	{
		return editCompletionKey;
	}

	/**
	 * @param key The KeyStroke to call the edit completions dialog
	 */
	public void setEditCompletionKey(KeyStroke key)
	{
		this.editCompletionKey = KeyStroke.getKeyStroke(key.getKeyCode(), modify(key.getModifiers()));
	}
	
	/**
	 * @return the handler that handles the events
	 */
	public MWACHandler getHandler ()
	{
		return handler;
	}
	
	/**
	 * @param handler the new handler to handle the events
	 */
	public void setHandler (MWACHandler handler)
	{
		this.handler = handler;
	}
	
	
	
	/* ======================================== */
	// 		HELPER METHODS
	/* ======================================== */

	/**
	 * Changes the modifier so that it contains the extended modifiers and the normal modifiers
	 * 
	 * @param modifiers The modifier to modify
	 * @return The new modifier
	 */
	private int modify (int modifiers)
	{
		if ((modifiers & KeyEvent.SHIFT_DOWN_MASK) == KeyEvent.SHIFT_DOWN_MASK
				|| (modifiers & KeyEvent.SHIFT_MASK) == KeyEvent.SHIFT_MASK)
			// the key modifiers contain SHIFT
			modifiers = modifiers | KeyEvent.SHIFT_MASK | KeyEvent.SHIFT_DOWN_MASK;
		
		if ((modifiers & KeyEvent.CTRL_DOWN_MASK) == KeyEvent.CTRL_DOWN_MASK
				|| (modifiers & KeyEvent.CTRL_MASK) == KeyEvent.CTRL_MASK)
			// the key modifiers contain CTRL
			modifiers = modifiers | KeyEvent.CTRL_MASK | KeyEvent.CTRL_DOWN_MASK;
		
		if ((modifiers & KeyEvent.META_DOWN_MASK) == KeyEvent.META_DOWN_MASK
				|| (modifiers & KeyEvent.META_MASK) == KeyEvent.META_MASK)
			// the key modifiers contain META
			modifiers = modifiers | KeyEvent.META_MASK | KeyEvent.META_DOWN_MASK;
		
		if ((modifiers & KeyEvent.ALT_DOWN_MASK) == KeyEvent.ALT_DOWN_MASK
				|| (modifiers & KeyEvent.ALT_MASK) == KeyEvent.ALT_MASK)
			// the key modifiers contain ALT
			modifiers = modifiers | KeyEvent.ALT_MASK | KeyEvent.ALT_DOWN_MASK;
		
		if ((modifiers & KeyEvent.ALT_GRAPH_DOWN_MASK) == KeyEvent.ALT_GRAPH_DOWN_MASK
				|| (modifiers & KeyEvent.ALT_GRAPH_MASK) == KeyEvent.ALT_GRAPH_MASK)
			// the key modifiers contain ALT GR
			modifiers = modifiers | KeyEvent.ALT_GRAPH_MASK | KeyEvent.ALT_GRAPH_DOWN_MASK;
		
		return modifiers;
	}
	
	

	/* ======================================== */
	// 		LISTENER EVENTS
	/* ======================================== */
	
	public void keyPressed(KeyEvent e) 
	{
		int key = e.getKeyCode();
		if (actionForThisKeyPerformed == key)
			return;
		
		actionForThisKeyPerformed = key;
		
		int modifiers 	= e.getModifiers() | e.getModifiersEx();
		
		if (view.isSaveCompletionEnabled()
				&& key == saveCompletionKey.getKeyCode()
				&& modifiers == saveCompletionKey.getModifiers())
		{
			// add a completion
			handler.showSaveCompletionDialog();
		}
		else if (view.isQuickAddEnabled()
				&& key == quickAddCompletionKey.getKeyCode()
				&& modifiers == quickAddCompletionKey.getModifiers())
		{
			// add the selected text
			handler.quickAdd();
		}
		else if (view.isEditCompletionEnabled()
				&& key == editCompletionKey.getKeyCode()
				&& modifiers == editCompletionKey.getModifiers())
		{
			// remove the completion
			handler.showEditCompletionDialog();
		}
	}
	
	
	public void keyReleased(KeyEvent e)
	{
		actionForThisKeyPerformed = 0;
	}

	/* ---------------------------------------- */
	// 		UNNEEDED EVENTS
	/* ---------------------------------------- */
	public void keyTyped(KeyEvent e) {}
}
