package lu.tudor.santec.org.fife.ui.autocomplete;

import java.awt.Window;
import java.util.List;

import javax.swing.InputMap;
import javax.swing.KeyStroke;
import javax.swing.text.BadLocationException;
import javax.swing.text.Caret;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;

public class MWAutoCompletion extends AutoCompletion
{
	public static final String AUTO_FILL_KEY = "key.autofill";
	
	
	/* ======================================== */
	// 		MEMBERS
	/* ======================================== */
	
	private boolean autoAddLettersWhereUnambiguous = true;
	
	
	
	/* ======================================== */
	// 		CONSTRUCTOR
	/* ======================================== */
	
	/**
	 * @param provider The completion provider
	 */
	public MWAutoCompletion(MWCompletionProvider provider)
	{
		super(provider);
	}
	
	
	
	/* ======================================== */
	// 		GETTER & SETTER
	/* ======================================== */
	
	/**
	 * Sets the key to complete the word as far as the auto completion 
	 * is unambiguous. The default key for that is the Tabulator key.
	 * 
	 * @param ks The new key
	 */
	public void setTabKey (KeyStroke ks)
	{
		JTextComponent 	tc = getTextComponent();
		InputMap 		im = tc.getInputMap();
		im.put(ks, AUTO_FILL_KEY);
	}
	
	/**
	 * @return <code>true</code> if the auto-add-letter-where-unambiguous function 
	 * is enabled.
	 */
	public boolean isAutoAddLettersWhereUnambiguous ()
	{
		return autoAddLettersWhereUnambiguous;
	}
	
	/**
	 * @param b <code>true</code> to enable the auto-add-letter-where-unambiguous
	 * function, <code>false</code> to disable it.
	 */
	public void setAutoAddLettersWhereUnambiguous (boolean b)
	{
		autoAddLettersWhereUnambiguous = b;
	}
	
	/**
	 * Adds the text as far as it is unambiguous.
	 */
	public synchronized void autoAddLettersWhereObvious ()
	{
		if (!autoAddLettersWhereUnambiguous)
			return;
		
		MWCompletionProvider provider 			= getCompletionProvider();
		JTextComponent 		 textComponent 		= getTextComponent();
		String 				 enteredWord		= provider.getAlreadyEnteredText(textComponent, false);
		int 				 enteredWordSize 	= enteredWord 		== null ? 0 : enteredWord.length();
		String 				 enteredMultiWord	= provider.getAlreadyEnteredText(textComponent, true);
		int 				 enteredMWSize		= enteredMultiWord 	== null ? 0 : enteredMultiWord.length();
		@SuppressWarnings("unchecked")
		List<Completion> 	completions 		= provider.getCompletions(textComponent);
		
		String obviousLetters = null;
		String inputText;
		for (Completion completion : completions)
		{
			inputText = completion.getInputText();
			// find out, whether the text is a multi-word text or not
			if (enteredMultiWord != null && inputText.toUpperCase().startsWith(enteredMultiWord.toUpperCase()))
				inputText 		= inputText.substring(enteredMWSize);
			else if (enteredWord != null)
				inputText 		= inputText.substring(enteredWordSize);
			
			if (obviousLetters == null)
				// this is the first iteration and the string obviousLetters is not set
				obviousLetters 	= inputText;
			else
				// compare the minimum of possible added letters with the current text
				obviousLetters 	= getAsFarAsEqual(obviousLetters, inputText);
			
			if (obviousLetters.length() <= 0)
				// no letters are obvious to complete
				return;
		}
		
		if (obviousLetters == null) 
			// the list of completions was empty
			return;
		
		// complete the text in the text component with the obvious letters
		Caret 	 caret 			= textComponent.getCaret();
		int 	 dot 			= caret.getDot();
		int 	 mark 			= caret.getMark();
		int 	 offset 		= Math.max(dot, mark);
		Document doc 			= textComponent.getDocument();
		
		try
		{
			// insert the letters, that are obvious to add
			doc.insertString(offset, obviousLetters, null);
			
			if (dot != mark)
			{
				// update the selection
				int start 		= Math.min(dot, mark);
				int end 		= offset + obviousLetters.length();
				
				// set the selection the right way (LTR or RTL)
				if (dot > mark)
				{
					caret.setDot(start);
					caret.moveDot(end);
				}
				else
				{
					caret.setDot(end);
					caret.moveDot(start);
				}
			}
		}
		catch (BadLocationException e)
		{
			e.printStackTrace();
		}
	}
	
	
	
	/* ======================================== */
	// 		OVERRIDDEN METHODS
	/* ======================================== */
	
	@Override
	protected AutoCompletePopupWindow createAutoCompletionPopupWindow(
			Window parentWindow)
	{
		return new MWACPopupWindow(parentWindow, this);
	}
	
	@Override
	public MWCompletionProvider getCompletionProvider()
	{
		return (MWCompletionProvider)super.getCompletionProvider();
	}
	
	@Override
	void insertCompletion(Completion c)
	{
		JTextComponent textComp = getTextComponent();
		Caret caret 			= textComp.getCaret();
		int dot 				= caret.getDot();
		int mark				= caret.getMark();
		int startOffset;
		
		MWCompletionProvider provider = getCompletionProvider();
		String inputText 		= c.getInputText();
		String replacementText 	= c.getReplacementText();
		if (inputText == null)
			inputText 			= replacementText;
		
		if (inputText == null || inputText.equals(""))
			return;
		
		if (dot != mark)
		{
			// something is selected => remove the selection
			startOffset			= Math.min(dot, mark);
			dot					= Math.max(dot, mark);
		}
		else if (provider.searchByLine() && isMultiWord(inputText))
		{
			// the text of entered from the start of this current line to the cursor position
			String enteredText 	= provider.getAlreadyEnteredText(textComp, true);
			
			// the length of the entered text
			int enteredTextSize	= enteredText.length();
			// the length of the input text of the chosen completion
			int inputTextSize	= inputText.length();
			// the index telling you, where to start comparing, 
			// respectively (later on) where the replaced word starts
			int startIndex		= enteredTextSize;
			
			for (int i = 0; i < enteredTextSize && startIndex == enteredTextSize; i++)
				for (int j = 0; j 	  < inputTextSize
							&& 	i + j < enteredTextSize
							// are the chars equal (ignore case)
							&& 	areCharsEqual(enteredText.charAt(i + j), inputText.charAt(j));
						j++)
				{
					if (j + i == enteredTextSize - 1)
						startIndex = i;
				}
			
			int len 			= enteredTextSize - startIndex;
			startOffset			= dot - len;
		}
		else if (provider.searchByWord())
		{
			// the text contains only digits, letters and hyphens
			String enteredText 	= provider.getAlreadyEnteredText(textComp, false);
			
			startOffset			= dot - enteredText.length();
		}
		else
			startOffset = Math.max(dot, mark);
		
		caret.setDot(startOffset);
		caret.moveDot(dot);
		hidePopupWindow();
		
		textComp.replaceSelection(replacementText);
		
		if (isParameterAssistanceEnabled() &&
				(c instanceof ParameterizedCompletion)) {
			ParameterizedCompletion pc = (ParameterizedCompletion)c;
			displayDescriptionToolTip(pc, true);
		}
	}
	
	
	
	/* ======================================== */
	// 		HELPER METHODS
	/* ======================================== */
	
	/**
	 * @param text The text to check
	 * @return Checks if the text contains several words or just one
	 */
	private boolean isMultiWord (String text)
	{
		for (int i = text.length() - 1; i >= 0; i--)
			if (!Character.isLetterOrDigit(text.charAt(i)))
				return true;
		
		return false;
	}
	
	/**
	 * @param a A character
	 * @param b Another character
	 * @return <code>true</code> if the chars are equal (ignore case), else <code>false</code>.
	 */
	private boolean areCharsEqual (char a, char b)
	{
		boolean value;
		
		if (Character.toUpperCase(a) == Character.toUpperCase(b))
			value = true;
		else
			value = false;
		
		return value;
	}
	
	/**
	 * @param text1 a text
	 * @param text2 another text
	 * @return A String as far as the given strings are equal (ignore case)
	 */
	private String getAsFarAsEqual (String text1, String text2)
	{
		if (text1 == null || text2 == null)
			return "";
		
		// get the minimum length of the 2 strings
		int minLength = Math.min(text1.length(), text2.length());
		
		int index = 0;
		while (index < minLength)
		{
			if (areCharsEqual(text1.charAt(index), text2.charAt(index)))
				// the chars are equal (ignore case) => go on
				index++;
			else
				// the chars aren't equal from here on => stop
				break;
		}
		
		return text1.substring(0, index);
	}
}
