package lu.tudor.santec.org.fife.ui.autocomplete;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.text.BadLocationException;
import javax.swing.text.Caret;
import javax.swing.text.Document;
import javax.swing.text.Element;
import javax.swing.text.JTextComponent;

/**
 * @author ferring
 * 
 * A special completion provider that provides special functions and 
 * replaces those of the DefaultCompletionProvider.
 */
public class MWCompletionProvider extends DefaultCompletionProvider
{
	/* ======================================== */
	// 		FINAL MEMBERS
	/* ======================================== */
	
	private static final int COMPLETION_ADDED 	= 0;
	private static final int COMPLETION_REMOVED = 1;
	private static final int COMPLETION_CHANGED = 2;
	private static final int INIT_COMPLETION 	= 3;
	
	public static final int SEARCH_MODE_DEACTIVATED = 0;
	public static final int SEARCH_MODE_WORD_ONLY 	= 1;
	public static final int SEARCH_MODE_LINE_ONLY	= 2;
	public static final int SEARCH_MODE_BOTH		= 3;
	
	
	/* ======================================== */
	// 		MEMBERS
	/* ======================================== */
	
	private int searchMode = SEARCH_MODE_BOTH;
	
	private LinkedList<MWACListener> listener = new LinkedList<MWACListener>();
	
	private String context;
	
	
	
	/* ======================================== */
	// 		CONSTRUCTORS & INITIALIZATION
	/* ======================================== */
	
	/**
	 * @param context The context of this provider.
	 */
	public MWCompletionProvider (String context)
	{
		initialize(context);
	}
	
	/**
	 * @param context The context of this provider.
	 * @param words Several Strings to create completions from, to initialize the provider.
	 */
	public MWCompletionProvider (String context, String[] words)
	{
		List<Completion> completions = new LinkedList<Completion>();
		for (String word : words)
		{
			completions.add(new MultiWordCompletion(this, word));
		}
		super.addCompletions(completions);
		
		initialize(context);
	}
	
	/**
	 * Initializes the 
	 * 
	 * @param context 
	 */
	private void initialize (String context)
	{
		MWACListener defaultListener = AutoCompletionTextComponent.getDefaultAutoCompletionListener();
		if (defaultListener != null
				&& !listener.contains(defaultListener))
			listener.add(defaultListener);
		
		initializeContext(context);
	}
	
	/**
	 * Same as addCompletions, but this method will not call the listeners.
	 * 
	 * @param completions a list of Completions, to add
	 */
	public void initializeProviderCompletions (List<? extends Completion> completions)
	{
		this.addCompletions(completions);
	}
	
	
	
	/* ======================================== */
	// 		GETTER & SETTER
	/* ======================================== */
	
	/**
	 * @return The context of this provider
	 */
	public String getContext ()
	{
		return context;
	}
	
//	/**
//	 * @param context The new context of this provider
//	 */
//	public void setContext (String context)
//	{
//		if (this.context == null
//				|| !this.context.equals(context))
//			initializeContext(context);
//	}
	
	
	/**
	 * @return returns all completions of the provider
	 */
	@SuppressWarnings("unchecked")
	public LinkedList<? extends Completion> getAllCompletions ()
	{
		LinkedList<Completion> retVal = new LinkedList<Completion>();
		String text = "";

		int index = Collections.binarySearch(completions, text, comparator);
		if (index<0) 
			// No exact match
			index = -index - 1;
		else 
		{
			// If there are several overloads for the function being
			// completed, Collections.binarySearch() will return the index
			// of one of those overloads, but we must return all of them,
			// so search backward until we find the first one.
			int pos = index - 1;
			while (pos>0 &&
					comparator.compare(completions.get(pos), text)==0) 
				retVal.add((Completion)completions.get(pos--));
		}

		while (index<completions.size()) {
			Completion c = (Completion)completions.get(index);
			if (startsWithIgnoreCase(c.getInputText(), text)) 
			{
				retVal.add(c);
				index++;
			}
			else 
				break;
		}

		return retVal;
	}
	
	
	/**
	 * @param c The completion to add
	 * @return <code>true</code> if the completion was added successfully, else <code>false</code>
	 */
	public boolean addCompletionGetFeedback (Completion c)
	{
		String inputText = c.getInputText();
		List<Completion> sameCompletions;
		
		if (inputText == null || inputText.equals(""))
			inputText = c.getReplacementText();
		
		sameCompletions = getCompletionByInputText(inputText);
		
		if (sameCompletions != null && !sameCompletions.isEmpty())
			for (Completion completion : sameCompletions)
				if (	// ignore completions, that are null
						completion != null
						// if replacement text is equal, the completion is equal
						&& (completion.getReplacementText() != null
								|| c.getReplacementText() == null)
						&& completion.getReplacementText().equals(c.getReplacementText()))
				{
					// cancel the 
					return false;
				}
		
		// no duplicate entry exists -> add the completion
		super.addCompletion(c);
		callListener(c, COMPLETION_ADDED);
		
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.fife.ui.autocomplete.AbstractCompletionProvider#addCompletion(org.fife.ui.autocomplete.Completion)
	 */
	@Override
	public void addCompletion(Completion c)
	{
		addCompletionGetFeedback(c);
	}
	
	/* (non-Javadoc)
	 * @see org.fife.ui.autocomplete.AbstractCompletionProvider#addCompletions(java.util.List)
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public void addCompletions(List completions)
	{
		List<Completion> mwCompletions = new LinkedList<Completion>();
		for (Object c : completions)
		{
			mwCompletions.add((Completion)c);
		}
		
		if (!mwCompletions.isEmpty())
		{
			super.addCompletions(mwCompletions);
			callListener(completions, COMPLETION_ADDED);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.fife.ui.autocomplete.AbstractCompletionProvider#removeCompletion(org.fife.ui.autocomplete.Completion)
	 */
	@Override
	public boolean removeCompletion(Completion c)
	{
		boolean success = super.removeCompletion(c);
		callListener(c, COMPLETION_REMOVED);
		return success;
	}
	
	
	/**
	 * @param oldCompletion The old completion that shall be replaced
	 * @param newCompletion The new completion that shall replace the old one
	 * @return <code>true</code> if the completion was replaced, else <code>false</code>
	 */
	public boolean changeCompletion (Completion oldCompletion, Completion newCompletion)
	{
		removeCompletion(oldCompletion);
		boolean success = addCompletionGetFeedback(newCompletion);
		
		// call the listeners
		if (success)
			callListener(oldCompletion, newCompletion, COMPLETION_CHANGED);
		else
			callListener(oldCompletion, COMPLETION_REMOVED);
		
		return success;
	}
	
	public void changeCompletion (MultiWordCompletion oldCompletion, String inputText, String replacementText)
	{
		MultiWordCompletion newCompletion = new MultiWordCompletion(
				this, inputText, replacementText);
		changeCompletion(oldCompletion, newCompletion);
	}
	
	/* (non-Javadoc)
	 * @see org.fife.ui.autocomplete.DefaultCompletionProvider#getAlreadyEnteredText(javax.swing.text.JTextComponent)
	 */
	@Override
	public String getAlreadyEnteredText(JTextComponent comp)
	{
		String text = null;
		
		if (searchByLine())
			text = getAlreadyEnteredText(comp, true);
		if (text == null && searchByWord())
			text = getAlreadyEnteredText(comp, false);
		
		return text;
	}
	
	/**
	 * @param comp The entered text to be considered by the auto completion
	 * @param considerWholeLine <code>true</code> if the whole line (from cursor back to the last carriage return)
	 * shall be considered, <code>false</code> if only the last word shall be considered.
	 * @return The entered text, starting from the cursor backwards.
	 */
	public String getAlreadyEnteredText(JTextComponent comp, boolean considerWholeLine)
	{
		String searchText;
		
		Caret caret = comp.getCaret();
		int dot = caret.getDot();
		int mark = caret.getMark();
		
		if (dot != mark)
		{
			searchText = comp.getSelectedText();
		}
		else if (!considerWholeLine)
		{
			if (!searchByWord())
				return null;
			
			searchText = super.getAlreadyEnteredText(comp);
		}
		else
		{
			if (!searchByLine())
				return null;
			
			// get the text, starting from the last line break ...
			Document doc 	= comp.getDocument();
			Element root 	= doc.getDefaultRootElement();
			int index 		= root.getElementIndex(dot);
			Element elem 	= root.getElement(index);
			int startOffset = elem.getStartOffset();
			
			try
			{
				searchText = doc.getText(startOffset, dot - startOffset);
			} catch (BadLocationException e1)
			{
				e1.printStackTrace();
				searchText = super.getAlreadyEnteredText(comp);
			}
		}
			
		return searchText;
	}
	
	/* (non-Javadoc)
	 * @see org.fife.ui.autocomplete.AbstractCompletionProvider#getCompletionsImpl(javax.swing.text.JTextComponent)
	 */
	@Override
	public List<Completion> getCompletionsImpl(JTextComponent comp) 
	{
		List<Completion> resultList = new ArrayList<Completion>();
		
		if (comp.getSelectionStart() != comp.getSelectionEnd())
		{
			// a selected text exists
			// => return a list of all completions that fit to the selected text
			resultList = getCompletions(comp.getSelectedText());
		}
		else
		{
			// get the last word before the caret 
			// => the text from the beginning of the word to the caret position
			// => if it shouldn't be searched for the whole line, set it to NULL
			String word = getAlreadyEnteredText(comp, false);
			
			// get the line of the caret 
			// => the text from the beginning of the line to the caret position
			// => if it shouldn't be searched for a single word, set it to NULL
			String line = getAlreadyEnteredText(comp, true);
			
			if (line != null)
				// the results of the line are added first, so they are on top
				resultList = getCompletions(line);
			
			if (word != null && !word.equals(line))
			{
				// remove duplicate entries
				List<Completion> wordCompletions = getCompletions(word);
				for (Completion completion : resultList)
				{
					wordCompletions.remove(completion);
				}
				
				// add the left completions to the result
				resultList.addAll(wordCompletions);
			}
		}
		
		return resultList;
	}
	
	/**
	 * @param text The text the completions shall be filtered by
	 * @return The completions that fit to the text
	 */
	@SuppressWarnings("unchecked")
	public List<Completion> getCompletions (String text)
	{
		List<Completion> resultList = new ArrayList<Completion>();
		
		if (text!=null) 
		{
			int index = Collections.binarySearch(completions, text, comparator);
			if (index<0) 
				// No exact match
				index = -index - 1;
			else 
			{
				// If there are several overloads for the function being
				// completed, Collections.binarySearch() will return the index
				// of one of those overloads, but we must return all of them,
				// so search backward until we find the first one.
				int pos = index - 1;
				while (pos > 0 && comparator.compare(completions.get(pos), text) == 0) 
					resultList.add((Completion)completions.get(pos--));
			}

			while (index<completions.size()) {
				Completion c = (Completion)completions.get(index);
				if (startsWithIgnoreCase(c.getInputText(), text))
					resultList.add(c);
				else 
					break;
				index++;
			}

		}

		return resultList;
	}
	
	
	public void setSearchMode (int searchMode)
	{
		this.searchMode = searchMode;
	}
	
	public int getSearchMode ()
	{
		return this.searchMode;
	}
	
	public boolean searchByWord ()
	{
		return (this.searchMode & SEARCH_MODE_WORD_ONLY) == SEARCH_MODE_WORD_ONLY;
	}
	
	public boolean searchByLine ()
	{
		return (this.searchMode & SEARCH_MODE_LINE_ONLY) == SEARCH_MODE_LINE_ONLY;
	}
	
	
	
	/* ======================================== */
	// 		HELPER
	/* ======================================== */
	
	/**
	 * Reinitializes the provider with a new Context.
	 * 
	 * @param newContext The new context
	 */
	private void initializeContext (String newContext)
	{
		this.context = newContext;

		this.completions.clear();
		
		callListener((Completion)null, INIT_COMPLETION);
	}
	
	
	/* ======================================== */
	// 		LISTENER MANAGING
	/* ======================================== */
	
	/**
	 * @param listener The AutoCompletionListener to add
	 */
	public void addAutoCompletionListener (MWACListener listener)
	{
		this.listener.add(listener);
		callListener((Completion)null, INIT_COMPLETION);
	}
	
	
	/**
	 * @param listener The AutoCompletionListener to remove
	 * @return <code>true</code> if the listener was removed successfully.
	 */
	public boolean removeAutoCompletionListener (MWACListener listener)
	{
		return this.listener.remove(listener);
	}
	
	
	private void callListener (List<Completion> c, int callCase)
	{
		MWACEvent event = new MWACEvent(
				this, System.currentTimeMillis(), c);
		callListener(event, callCase);
	}
	
	private void callListener (Completion c, int callCase)
	{
		this.callListener(null, c, callCase);
	}
	
	private void callListener (Completion oldCompletion, Completion newCompletion, int callCase)
	{
		MWACEvent event = new MWACEvent(
				this, System.currentTimeMillis(), newCompletion, oldCompletion);
		callListener(event, callCase);
	}
	
	
	protected void callListener (MWACEvent event, int callCase)
	{
		for (MWACListener l : this.listener)
		{
			switch (callCase)
			{
			case COMPLETION_ADDED:
				l.completionAdded(event);
				break;
				
			case COMPLETION_REMOVED:
				l.completionRemoved(event);
				break;
				
			case COMPLETION_CHANGED:
				l.completionChanged(event);
				break;
				
			case INIT_COMPLETION:
				l.initializeCompletion(event);
				break;
			}
		}
	}
}
