package lu.tudor.santec.org.fife.ui.autocomplete.dialog;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.plaf.basic.BasicComboBoxRenderer;

import lu.tudor.santec.org.fife.ui.autocomplete.AutoCompletionTextComponent;
import lu.tudor.santec.org.fife.ui.autocomplete.Completion;
import lu.tudor.santec.org.fife.ui.autocomplete.IconManager;
import lu.tudor.santec.org.fife.ui.autocomplete.MWCompletionProvider;
import lu.tudor.santec.org.fife.ui.autocomplete.Translator;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * @author ferring
 * 
 * This dialog provides the option to edit completions.
 * Choose one and set the new values.
 */
public class EditCompletionDialog extends BaseDialogImpl 
{
	private static final long serialVersionUID = 1L;
	
	protected AutoCompletionTextComponent view;
	
	protected JTextField searchField;
	
	protected JList resultList;
	
	
	
	/**
	 * Creates a dialog that provides the option to edit completions.
	 * Choose one and set the new values.
	 * 
	 * @param textArea the AutoCompletionTextComponent, to edit the completions of.
	 */
	public EditCompletionDialog(JDialog owner, AutoCompletionTextComponent textArea)
	{
		super(owner, 
				Translator.translate(Translator.EDIT_COMPLETIONS), 
				BaseDialog.OK_BUTTON_MODE, 
				new JPanel());
		
		view = textArea;
		
		init();
	}
	
	
	/**
	 * Creates a dialog that provides the option to edit completions.
	 * Choose one and set the new values.
	 * 
	 * @param textArea the AutoCompletionTextComponent, to edit the completions of.
	 */
	public EditCompletionDialog(JFrame owner, AutoCompletionTextComponent textArea)
	{
		super(owner,
				Translator.translate(Translator.EDIT_COMPLETIONS), 
				BaseDialog.OK_BUTTON_MODE, 
				new JPanel());
		
		view = textArea;
		
		init();
	}
	
	/**
	 * initialises the dialog
	 */
	protected void init ()
	{
		// THE BUTTONS ...
		// make the OK button an exit button
		okButton.setText(Translator.translate(Translator.CLOSE));
		okButton.setIcon(IconManager.getIcon(IconManager.CLOSE_BUTTON_ICON_NAME));
		enableOkKeyMapping(false);
		
		// add button
		final JButton addButton = new JButton(
				IconManager.getIcon(IconManager.ADD_ICON_NAME));
		addButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				add();
			}
		});
		addButton.setEnabled(true);
		super.addButton(addButton);
		
		// remove button
		final JButton removeButton = new JButton(
//				Translatrix.getTranslationString("core.remove"), 
				IconManager.getIcon(IconManager.REMOVE_ICON_NAME));
		removeButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				remove();
			}
		});
		removeButton.setEnabled(false);
		super.addButton(removeButton);
		
		// change button
		final JButton editButton = new JButton(
				IconManager.getIcon(IconManager.EDIT_ICON_NAME));
		editButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				edit();
			}
		});
		editButton.setEnabled(false);
		super.addButton(editButton);
		
		
		
		searchField = new JTextField(view.getSelectedText());
		searchField.addKeyListener(new KeyListener()
		{
			public void keyTyped(KeyEvent e) {}
			public void keyPressed(KeyEvent e) {}
			
			public void keyReleased(KeyEvent e) 
			{
				search();
			}
		});
		
		// the list, that shows the results
		resultList = new JList((ListModel) new ResultListModel());
		resultList.setCellRenderer(new ToolTipListRenderer());
		resultList.addListSelectionListener(new ListSelectionListener()
		{
			public void valueChanged(ListSelectionEvent e)
			{
				if (!e.getValueIsAdjusting())
				{
					// is something selected?
					boolean enabled = resultList.getSelectedIndex() >= 0;
					removeButton.setEnabled(enabled);
					editButton.setEnabled(enabled);
				}
			}
		});
		JScrollPane tableScroller = new JScrollPane(resultList);
		
		CellConstraints cc = new CellConstraints();
		
		mainPanel.setLayout(new FormLayout(
				"5px, f:min(p;350px):g, 5px", 
				"5px, f:p, 5px, f:250px:g, 5px"));
		
		mainPanel.add(searchField, 	cc.xy(2, 2));
		mainPanel.add(tableScroller,cc.xy(2, 4));
		
		this.pack();
		this.setLocationRelativeTo(view);
		
		search();
	}

	
	
	/**
	 * Searches for the in the searchBox selected text and shows the 
	 * results in the resultList.
	 */
	protected void search ()
	{
		String searchText 				= (String)searchField.getText();
		if (searchText == null)
			searchText 					= "";
		
		ResultListModel model 			= (ResultListModel)resultList.getModel();
		MWCompletionProvider provider 	= view.getProvider();
		List<Completion> searchResults	= provider.getCompletions(searchText);
		
		model.clear();
		if (searchResults != null)
			model.addAll((List<Completion>) searchResults);
		
		resultList.removeSelectionInterval(0, model.getSize());
	}
	
	
	/**
	 * opens the dialog to add, resp. save a new completion
	 */
	protected void add ()
	{
		SaveCompletionDialog dialog = new SaveCompletionDialog(this, view);
		dialog.setReplacementText("");
		dialog.setVisible(true);
		
		search();
	}
	
	
	/**
	 * Removes the selected completion.
	 */
	protected void remove ()
	{
		int selectedIndex = resultList.getMinSelectionIndex();
		
		Object[] completionsToRemove =  resultList.getSelectedValues();
		MWCompletionProvider provider = view.getProvider();
		
		for (Object c : completionsToRemove)
			provider.removeCompletion((Completion)c);
		
		search();
		
		// select an entry
		if (resultList.getModel().getSize() <= 0)
			return;
		else if (selectedIndex < 0)
			selectedIndex = 0;
		else if (selectedIndex >= resultList.getModel().getSize())
			selectedIndex = resultList.getModel().getSize()-1;
		resultList.setSelectedIndex(selectedIndex);
	}
	
	/**
	 * Opens the edit dialog, to edit the selected completion
	 */
	protected void edit ()
	{
		Completion oldCompletion = (Completion)resultList.getSelectedValue();
		ChangeCompletionDialog change = new ChangeCompletionDialog(this, view, oldCompletion);
		
		change.setVisible(true);
		
		if (change.isChanged())
			search();
	}
	
	
	@Override
	public void setVisible(boolean b)
	{
		if (b) this.pack();
		
		super.setVisible(b);
	}
	
	
	
	/* ======================================== */
	// 		CLASS: RESULT LIST MODEL
	/* ======================================== */
	
	/**
	 * @author ferring
	 * This class is used as model of the JList resultList.
	 */
	class ResultListModel implements ListModel
	{
//		private static final long serialVersionUID = 1L;
		
		private LinkedList<ListDataListener> listeners = new LinkedList<ListDataListener>();
		
		private Vector<Completion> data = new Vector<Completion>();
		
		
		/* ======================================== */
		// 		GET / ADD / CLEAR
		/* ======================================== */
		
		public int getSize()
		{
			return data.size();
		}

		public Object getElementAt(int index)
		{
			return data.get(index);
		}
		
		public void addAll(Collection<? extends Completion> c)
		{
			int start = data.size();
			int end = start + c.size();
			
			data.addAll(c);
			fireModelChanged(ListDataEvent.INTERVAL_ADDED, start, end);
		}
		
		public void add (Completion c)
		{
			data.add(c);
			fireModelChanged(ListDataEvent.INTERVAL_ADDED, data.size()-1, data.size());
		}
		
		public void clear()
		{
			data.clear();
			fireModelChanged(ListDataEvent.INTERVAL_REMOVED, 0, data.size()-1);
		}

		
		/* ======================================== */
		// 		HADNLING LISTENERS
		/* ======================================== */
		
		public void addListDataListener(ListDataListener l)
		{
			listeners.add(l);
		}

		public void removeListDataListener(ListDataListener l)
		{
			listeners.remove(l);
		}
		
		private void fireModelChanged (int reason, int startIndex, int endIndex)
		{
			ListDataEvent e = new ListDataEvent(this, reason, startIndex, startIndex);
			for (ListDataListener l : listeners)
			{
				switch (reason)
				{
				case ListDataEvent.INTERVAL_ADDED:					
					l.intervalAdded(e);
					break;
				case ListDataEvent.INTERVAL_REMOVED:
					l.intervalRemoved(e);
					break;
				}
			}
		}
	}
	
	
	
	
	
	/* ======================================== */
	// 		CLASS: TOOL TIP RENDERER
	/* ======================================== */
	
	/**
	 * @author ferring
	 *
	 */
	class ToolTipListRenderer extends BasicComboBoxRenderer 
	{
		private static final long serialVersionUID = 1L;
		
		public Component getListCellRendererComponent(JList list, Object value,
				int index, boolean isSelected, boolean cellHasFocus)
		{
			if (isSelected)
			{	
				this.setBackground(list.getSelectionBackground());
				this.setForeground(list.getSelectionForeground());
			}
			else
			{
				this.setBackground(list.getBackground());
				this.setForeground(list.getForeground());
			}
			
			String text = value == null ? "" : value.toString();
			this.setFont(list.getFont());
			this.setText(text);
			this.setToolTipText(text);
			
			return this;
		}
	}
}
