package lu.tudor.santec.org.fife.ui.autocomplete.dialog;

import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.KeyStroke;

import lu.tudor.santec.org.fife.ui.autocomplete.AutoCompletionTextComponent;
import lu.tudor.santec.org.fife.ui.autocomplete.Translator;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;
import com.jgoodies.forms.layout.FormSpec;
import com.jgoodies.forms.layout.RowSpec;
import com.jgoodies.forms.layout.Sizes;

/**
 * @author ferring
 * 
 * This opens a dialog to edit the hot keys.
 */
public class HotKeyDialog extends BaseDialogImpl
{
	private static final long serialVersionUID = 1L;
	
	private static final RowSpec SPACE_ROW_SPEC		= new RowSpec(Sizes.pixel(5));
	private static final RowSpec HOTKEY_ROW_SPEC 	= new RowSpec(RowSpec.FILL, Sizes.PREFERRED, FormSpec.NO_GROW);
	
	private static final String 			PLUS 	= " + ";
	private static final CellConstraints 	cc 		= new CellConstraints();
	
	private static final int QUICK_ADD_INDEX 				= 0;
	private static final int SAVE_COMPLETION_INDEX 			= 1;
	private static final int EDIT_COMPLETION_INDEX 			= 2;
	private static final int START_AUTO_COMPLETION_INDEX 	= 3;
//	private static final int AUTO_FILL_INDEX 				= 4;
	
	
	private AutoCompletionTextComponent textComponent;
	
	private JPanel 		hotKeyPanel;
	
	private FormLayout 	hotKeyLayout;
	
	private KeyStroke[] hotKeys = new KeyStroke[4];
	
	
	
	/**
	 * @param textComponent The AutoCompletionTextComponent
	 */
	public HotKeyDialog (JDialog owner, AutoCompletionTextComponent textComponent)
	{
		super(owner, 
				Translator.translate(Translator.HOTKEYS), 
				BaseDialog.OK_CANCEL_BUTTON_MODE, 
				new JPanel());
		init(textComponent);
	}
	
	
	/**
	 * @param textComponent The AutoCompletionTextComponent
	 */
	public HotKeyDialog (JFrame owner, AutoCompletionTextComponent textComponent)
	{
		super(owner, 
				Translator.translate(Translator.HOTKEYS), 
				BaseDialog.OK_CANCEL_BUTTON_MODE, 
				new JPanel());
		init(textComponent);
	}
	
	
	private void init (AutoCompletionTextComponent textComponent)
	{
		this.textComponent = textComponent;
		
		hotKeys[QUICK_ADD_INDEX] 			= textComponent.getQuickAddCompletionKey();
		hotKeys[SAVE_COMPLETION_INDEX] 		= textComponent.getSaveCompletionKey();
		hotKeys[EDIT_COMPLETION_INDEX] 		= textComponent.getEditCompletionKey();
		hotKeys[START_AUTO_COMPLETION_INDEX]= textComponent.getTriggerKey();
//		hotKeys[AUTO_FILL_INDEX] 			= ;
		
		initializeLayout();
		
		this.pack();
		this.setLocationRelativeTo(textComponent);
	}
	
	
	/**
	 * Initializes the layout of this dialog
	 */
	private void initializeLayout ()
	{
		JLabel 			headline = new JLabel(Translator.translate(Translator.HOTKEYS));
		headline.setOpaque(false);
		Translator		t		 = new Translator();
		
		hotKeyLayout = new FormLayout("f:p:g, 5px, f:max(p;100px)", "5px");
		hotKeyPanel = new JPanel (hotKeyLayout);
		hotKeyPanel.setOpaque(false);
		
		// add the fields to set the hotkeys
		appendHotKeyRow(t.t(Translator.QUICK_ADD_HOTKEY), 				QUICK_ADD_INDEX);
		appendHotKeyRow(t.t(Translator.SAVE_COMPLETION_HOTKEY), 		SAVE_COMPLETION_INDEX);
		appendHotKeyRow(t.t(Translator.EDIT_COMPLETION_HOTKEY), 		EDIT_COMPLETION_INDEX);
		appendHotKeyRow(t.t(Translator.START_AUTO_COMPLETION_HOTKEY), 	START_AUTO_COMPLETION_INDEX);
//		appendHotKeyRow(t.t(Translator.COMPLETE_UNAMBIGUOUSLY_HOTKEY), 	AUTO_FILL_INDEX);
		
		mainPanel.setLayout(new FormLayout("5px, f:p:g, 5px", 
				"5px, f:p, " +	// headline
				"5px, f:p:g, " +// hotkey selection
				"5px"));
		
		mainPanel.add(headline, 	cc.xy(2, 2));
		mainPanel.add(hotKeyPanel, 	cc.xy(2, 4));
	}
	
	
	/**
	 * This adds a new row containing a label with the description and a text field
	 * with the hot key, where the hot key can also be set.
	 * 
	 * @param description The description, what the hot key is for
	 * @param keyStrokeIndex The position of the hot key in the KeyStroke array hotKeys
	 */
	private void appendHotKeyRow (String description, final int keyStrokeIndex)
	{
		// the KeyStroke that is stored at the specified index
		KeyStroke startStroke 	= hotKeys[keyStrokeIndex];
		
		// the field to enter and read the KeyStroke
		JTextField keyField 	= new JTextField ();
		if (startStroke != null)
			keyField.setText(getKeyDescription(
					startStroke.getModifiers(), startStroke.getKeyCode()));
		
		// the label that contains the description of the KeyStroke
		JLabel descriptionLabel = new JLabel (description);
		
		// adds the row to the layout
		hotKeyLayout.appendRow(HOTKEY_ROW_SPEC);
		
		int rowCount = hotKeyLayout.getRowCount();
		
		// adds the description label and the hot key text field into the row
		hotKeyPanel.add(descriptionLabel,cc.xy(1, rowCount));
		hotKeyPanel.add(keyField, 		 cc.xy(3, rowCount));
		
		hotKeyLayout.appendRow(SPACE_ROW_SPEC);
		
		keyField.addKeyListener(new KeyListener()
		{
			public void keyTyped(KeyEvent e) {}
			public void keyReleased(KeyEvent e) {}
			
			public void keyPressed(KeyEvent e)
			{
				if (	   e.getKeyCode() == KeyEvent.VK_ALT
						|| e.getKeyCode() == KeyEvent.VK_ALT_GRAPH
						|| e.getKeyCode() == KeyEvent.VK_SHIFT
						|| e.getKeyCode() == KeyEvent.VK_CONTROL
						|| e.getKeyCode() == KeyEvent.VK_META)
					// if the key itself is a modifier key do nothing
					return;
				
				JTextField field = (JTextField)e.getSource();
				KeyStroke stroke;
				
				if (e.getKeyCode() == KeyEvent.VK_BACK_SPACE
						&& e.getModifiersEx() == 0)
				{
					// the KeyStroke will be deleted when pressing backspace
					field.setText("");
					stroke = null;
				}
				else
				{
					// otherwise the KeyStroke is set and its text is printed into the text field
					field.setText(getKeyDescription(e.getModifiersEx(), e.getKeyCode()));
					stroke = KeyStroke.getKeyStroke(e.getKeyCode(), e.getModifiersEx());
				}
				
				// set the KeyStroke at the specified index in the hot key array
				hotKeys[keyStrokeIndex] = stroke;
			}
		});
	}
	
	/**
	 * @param modifiers The modifier
	 * @param keyCode The key code
	 * @return The text of the key modifier and the key
	 */
	private String getKeyDescription (int modifiers, int keyCode)
	{
		StringBuffer keyText = new StringBuffer(
				KeyEvent.getModifiersExText(modifiers));
		if (keyText.length() > 0)
			keyText.append(PLUS);
		keyText.append(KeyEvent.getKeyText(keyCode));
		
		return keyText.toString();
	}
	
	/* (non-Javadoc)
	 * @see lu.tudor.santec.org.fife.ui.autocomplete.dialog.BaseDialogImpl#okActionCalled()
	 */
	@Override
	public void okActionCalled()
	{
		textComponent.setQuickAddCompletionKey(hotKeys[QUICK_ADD_INDEX]);
		textComponent.setSaveCompletionKey(hotKeys[SAVE_COMPLETION_INDEX]);
		textComponent.setEditCompletionKey(hotKeys[EDIT_COMPLETION_INDEX]);
		textComponent.setTriggerKey(hotKeys[START_AUTO_COMPLETION_INDEX]);
		
		super.okActionCalled();
	}
	
	/* (non-Javadoc)
	 * @see lu.tudor.santec.org.fife.ui.autocomplete.dialog.BaseDialogImpl#cancelActionCalled()
	 */
	@Override
	public void cancelActionCalled()
	{
		super.cancelActionCalled();
	}
}
