//===============================================================
// package : com.tetrasix.majix
// class : com.tetrasix.majix.Batch
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix;

import java.io.File;

import com.tetrasix.util.*;
import com.tetrasix.majix.templates.*;
import com.tetrasix.majix.rtf.*;
import com.tetrasix.majix.xml.*;

/**
 * This class defines the main entrypoint of the program for the
 * commandline-mode. This mode is applicable for use in batchfiles.
 * <p>
 * The usage of the commandline:
 * <p>
 * <i>java com.tetrasix.majix.MajixBatch [ -debug ] [ -dump ] [ -xp ]
 * [ -style <stylefilename> ] [ -tags <tagfilename> ] [ -template <templatename> ]
 * word_file xml_file</i>
 */
public class MajixBatch  {

	public static boolean styleSheetCall = false;
	/**
	 * The main entrypoint for the application in the stylesheet-mode.
	 * 
	 * @param stylemapfile the filename of the stylemapfile (.sty)
	 * @param tagsmapfile  the filename of the tagsmapfile  (.tdef)
	 * @param templatename the name of the template as defined in the majixt-file <majixt name="templatename">
	 * @param rtffilename  the filename of the rtf-file to convert
	 * @param xmlfilename  the filename of the destination xml-file
	 * @param checkparam   boolean-String for specifying the validation against the DTD.
	 * @param debuglog	   boolean-String for specifying the debug-level of logging.
	 */
	public static void main(String stylemapfile, String tagsmapfile, String templatename,
			                String rtffilename, String xmlfilename, String checkparam, String debuglog)
	{
		long startTime = System.currentTimeMillis();
		// Set Signal that program knows it has been called
		// from a xslt-stylesheet
		styleSheetCall = true;

		if (System.getProperty("java.version").startsWith("1.0")) {
			System.err.println("This application requires Java 1.1 or higher\n");
			System.exit(1);
		}
		Configuration.init("majix", true);

		boolean check = checkparam.equalsIgnoreCase("true");
		boolean dump  = false;


		if (rtffilename != null) {
			rtffilename = (new File(rtffilename)).getAbsolutePath();
		}
		if (xmlfilename != null) {
			xmlfilename = (new File(xmlfilename)).getAbsolutePath();
		}
		if (stylemapfile != null) {
			stylemapfile = (new File(stylemapfile)).getAbsolutePath();
		}
		if (tagsmapfile != null) {
			tagsmapfile = (new File(tagsmapfile)).getAbsolutePath();
		}

/*		System.out.println("Majix: templatename='"+templatename+"'");
		System.out.println("Majix: stylemapfile='"+stylemapfile+"'");
		System.out.println("Majix: tagsmapfile='"+tagsmapfile+"'");
		System.out.println("Majix: rtffilename='"+rtffilename+"'");
		System.out.println("Majix: xmlfilename='"+xmlfilename+"'");
*/
 		if (!rtffilename.endsWith(".rtf")) {
			System.err.println("Illegal name of input file : " + rtffilename);
			System.err.println("Only .DOC and .RTF files are supported");
			System.exit(1);
		}

		if (xmlfilename == null) {
			xmlfilename = rtffilename.substring(0, rtffilename.length() - 4) + ".xml";
		}

		ConversionTemplateFactory factory = new ConversionTemplateFactory();
		ConversionTemplate template
			= factory.getConversionTemplate(templatename);
		if (template == null) {
			System.err.println("Template unknown: " + templatename);
			System.exit(1);
		}

		if (stylemapfile != null) {
			RtfStyleMap stylemap = new RtfStyleMap(stylemapfile);
			template.initRtfAbstractStyleSheet(stylemap.getRtfAbstractStylesheet());
			template.setStyleMap(stylemap);
		}
		if (tagsmapfile != null) {
			template.setGeneratorParam(new XmlGeneratorParam(template, tagsmapfile));
		}

		if (debuglog.equals("true"))
			System.out.println("Majix: Converting file '"+rtffilename+"'");
		
		
		
		// #########################################################
		
		boolean ok = Converter.Convert(template, rtffilename, xmlfilename, dump, debuglog.equals("true"), null);

		// #########################################################
		
		
		
		
		if (ok && debuglog.equals("true")) {
			System.out.print("Whole Conversion performed in: ");
			System.out.print((System.currentTimeMillis() - startTime));
			System.out.println(" msec.");
		}
		else if (debuglog.equals("")) {
			System.err.println("Conversion failed");
		}
//		// check against DTD
//		if (check) {
//			startTime = System.currentTimeMillis();
//			ok = CheckerSAX.check(xmlfilename, null);
//			if (ok) {
//				System.out.print("Check performed : ");
//				System.out.print((System.currentTimeMillis() - startTime)/1000.0);
//				System.out.println(" sec.");
//			}
//			else {
//				System.out.println("Check failed");
//			}
//		}
	}


	
	/**
	 * The main entrypoint for the application in the commandline-mode.
	 * 
	 * @param args the commandline-arguments, given by the VM
	 */
	public static void main(String args[])
	{
		if (System.getProperty("java.version").startsWith("1.0")) {
			System.err.println("This application requires Java 1.1 or higher\n");
			System.exit(1);
		}

		if (args.length == 0) {
			usage();
			System.exit(1);
		}
		
		Configuration.init("majix", true);
		Configuration.setDefaultLocale(args);

		String rtffilename = null;
		String xmlfilename = null;
		String stylemapfile = null;
		String tagsmapfile = null;
		String templatename = "mydoc";

		boolean check = false;
		boolean debug = false;
		boolean dump  = false;

		for (int ii = 0; ii < args.length; ii++) {
			String s = args[ii];
			if (s.charAt(0) == '-') {
				if (s.equals("-xp")) {
					check = true;
				}
				else if (s.equals("-h")) {
					System.out.println("argument help:");
					usage();
					System.exit(1);
				}
				else if (s.equals("-dump")) {
					dump = true;
				}
				else if (s.equals("-debug")) {
					debug = true;
				}
				else if (s.equals("-template")) {
					ii++;
					if (ii < args.length) {
						templatename = args[ii];
					}
					else {
						System.err.println("missing templatename argument");
						usage();
						System.exit(1);
					}
				}
				else if (s.equals("-style")) {
					ii++;
					if (ii < args.length) {
						stylemapfile = args[ii];
					}
					else {
						System.err.println("missing stylefilename argument");
						usage();
						System.exit(1);
					}
				}
				else if (s.equals("-tags")) {
					ii++;
					if (ii < args.length) {
						tagsmapfile = args[ii];
					}
					else {
						System.err.println("missing tagfilename argument");
						usage();
						System.exit(1);
					}
				}
				else {
					System.err.println("unknown parameter: "+s);
					usage();
					System.exit(1);
				}
			}
			else {
				if (rtffilename == null) {
					rtffilename = s;
				}
				else if (xmlfilename == null) {
					xmlfilename = s;
				}
				else {
					System.err.println("unknown parameter: "+s);
					usage();
					System.exit(1);
				}
			}
		}

		if (rtffilename != null) {
			rtffilename = (new File(rtffilename)).getAbsolutePath();
		}
		if (xmlfilename != null) {
			xmlfilename = (new File(xmlfilename)).getAbsolutePath();
		}
		if (stylemapfile != null) {
			stylemapfile = (new File(stylemapfile)).getAbsolutePath();
		}
		if (tagsmapfile != null) {
			tagsmapfile = (new File(tagsmapfile)).getAbsolutePath();
		}

		if (rtffilename.toLowerCase().endsWith(".doc") == true) {
			if (! Configuration.getProperty("majix.enable.msword").equals("1")) {
				System.err.println("Processing of .DOC files is not enabled : " + rtffilename);
				System.exit(1);
			}
		}
 		else if (rtffilename.toLowerCase().endsWith(".rtf") == false) {
			System.err.println("Illegal name of input file : " + rtffilename);
			if (Configuration.getProperty("majix.enable.msword").equals("1")) {
				System.err.println("Only .DOC and .RTF files are supported");
			}
			else {
				System.err.println("Only .DOC and .RTF files are supported");
			}
			System.exit(1);
		}

		if (xmlfilename == null) {
			xmlfilename = rtffilename.substring(0, rtffilename.length() - 4) + ".xml";
		}

		System.out.println("Converting file " + rtffilename
						+ " into " + xmlfilename);

		long startTime = System.currentTimeMillis();

		ConversionTemplateFactory factory = new ConversionTemplateFactory();
		ConversionTemplate template
			= factory.getConversionTemplate(templatename);
		if (template == null) {
			System.err.println("Template unknown : " + templatename);
			System.exit(1);
		}

		if (stylemapfile != null) {
			RtfStyleMap stylemap = new RtfStyleMap(stylemapfile);
			template.initRtfAbstractStyleSheet(stylemap.getRtfAbstractStylesheet());
			template.setStyleMap(stylemap);
		}
		if (tagsmapfile != null) {
			template.setGeneratorParam(new XmlGeneratorParam(template, tagsmapfile));
		}

		boolean ok = Converter.Convert(template, rtffilename, xmlfilename, dump, debug, null);
		if (ok) {
			System.out.print("Conversion performed : ");
			System.out.print((System.currentTimeMillis() - startTime)/1000.0);
			System.out.println(" sec.");
		}
		else {
			System.err.println("Conversion failed");
			System.exit(1);
		}

		if (check) {
			startTime = System.currentTimeMillis();
			ok = CheckerSAX.check(xmlfilename, null);
			if (ok) {
				System.out.print("Check performed : ");
				System.out.print((System.currentTimeMillis() - startTime)/1000.0);
				System.out.println(" sec.");
			}
			else {
				System.err.println("Check failed");
				// XML is generated, but not valid, return --> 2 as statuscode
				System.exit(2);
			}
		}
	}

	/**
	 * This method prints the usage-format to the starting console.
	 */
	static void usage()
	{
		System.err.println("usage: java com.tetrasix.majix.MajixBatch");
		System.err.println("                [ -debug ]");
		System.err.println("                [ -dump ]");
		System.err.println("                [ -xp ]");
		System.err.println("                [ -style <stylefilename> ]");
		System.err.println("                [ -tags <tagfilename> ]");
		System.err.println("                [ -template <templatename> ]");
		System.err.println("                word_file xml_file");
	}
}


