//===============================================================
// package : com.tetrasix.majix.rtf
// class : com.tetrasix.majix.RtfReader
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix.rtf;

import java.io.*;

public class RtfReader {

	private PushbackInputStream _in;
	private StringBuffer _buf;
	private int _depth;
	private String _filename;
	private long _nbtoken;

	public RtfReader(String filename) throws FileNotFoundException
	{
		_filename = filename;
		open(new FileInputStream(filename));
	}
	
	public RtfReader(InputStream inStream) throws FileNotFoundException
	{
		open(inStream);
	}

	public void open(InputStream inStream) throws FileNotFoundException
	{
		_in = new PushbackInputStream(inStream);
		_buf = new StringBuffer();
		_depth = 0;
		_nbtoken = 0;
	}

	public RtfToken getNextToken() throws IOException
	{
		RtfToken tok;
		
		//_buf.setLength(0);
		_buf = new StringBuffer();
		_nbtoken++;

		for(;;) {
			int ch = _in.read();

			if (ch == -1) {
				return null;
			}
			else if (ch == '\\') {
				if (_buf.length() > 0) {
					_in.unread(ch);
					String data = _buf.toString();
					_buf.setLength(0);
					return new RtfToken(data);
				}

				ch = _in.read();

				if (ch == '*') {
					return new RtfToken(RtfToken.ASTERISK);
				}

				if (ch == '\'') {
					char ch1 = (char) _in.read();
					char ch2 = (char) _in.read();
					ch = Character.digit(ch1, 16) * 16 + Character.digit(ch2, 16);
					_buf.append((char)ch);
					String data = _buf.toString();
					return new RtfToken(data);
				}
				else if (ch == '\\') {
					return new RtfToken("\\");
				}
				else if (ch == '~') {
					return new RtfToken("\u00A0");
				}
				else if (ch == '-') {
					return new RtfToken("\u2010");
				}
				else if (ch == '_') {
					return new RtfToken("\u2011");
				}
				else if (ch == '{') {
					return new RtfToken("{");
				}
				else if (ch == '}') {
					return new RtfToken("}");
				}

				while (Character.isLetter((char)ch)) {
					_buf.append((char)ch);
					ch = _in.read();
				}
				String name = _buf.toString();
				_buf.setLength(0);

				if (ch == '-') {
					_buf.append((char)ch);
					ch = _in.read();
				}

				while (Character.isDigit((char)ch)) {
					_buf.append((char)ch);
					ch = _in.read();
				}

				if (! Character.isSpaceChar((char)ch)) {
					_in.unread(ch);
				}
				
				String data = _buf.toString();
				_buf.setLength(0);

				return new RtfToken(RtfToken.CONTROLWORD, name, data);
			}
			else if (ch == '{') {
				if (_buf.length() > 0) {
					_in.unread(ch);
					String data = _buf.toString();
					return new RtfToken(data);
				}

				_depth++;

				return new RtfToken(RtfToken.OPENGROUP);
			}
			else if (ch == '}') {
				if (_buf.length() > 0) {
					_in.unread(ch);
					String data = _buf.toString();
					return new RtfToken(data);
				}

				_depth--;

				return new RtfToken(RtfToken.CLOSEGROUP);
			}
			else if (ch == '\r') {
			}
			else if (ch == '\n') {
			}
			else {
				_buf.append((char)ch);
			}
		}
	}

	public int getDepth()
	{
		return _depth;
	}

	public long getTokenCount()
	{
		return _nbtoken;
	}

	public void close()
	{
		if (_in != null) {
			try {
				_in.close();
			}
			catch(IOException e) {
				System.out.println("Exception when closing " + _filename);
			}
			_in = null;
			_buf = null;
			_filename = null;
		}
	}

	public String getFileName()
	{
		return _filename;
	}
}


