//===============================================================
// package : com.tetrasix.majix.rtf
// class : com.tetrasix.majix.RtfStyleMap
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix.rtf;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import com.jclark.xml.parse.EntityManagerImpl;
import com.jclark.xml.parse.OpenEntity;
import com.jclark.xml.parse.StartElementEvent;
import com.jclark.xml.parse.io.ApplicationImpl;
import com.jclark.xml.parse.io.Parser;
import com.jclark.xml.parse.io.ParserImpl;
import com.tetrasix.util.ISO8859XMLWriter;

public class RtfStyleMap implements Cloneable {

	// map of external names
	private Hashtable _map  = new Hashtable(100);
	private Vector    _keys = new Vector();

	private final static int PARAGRAPH_STYLE = 1;
	private final static int CHARACTER_STYLE = 2;

	private RtfAbstractStylesheet _astylessheet = new RtfAbstractStylesheet();

	private AbstractStyleProvider _abstract_style_provider = null;

	public RtfStyleMap()
	{
	}

	public RtfStyleMap(String filename)
	{
		if (filename != null) {
			try {
				Load(filename);
			}
			catch (IOException e) {
				System.out.println("IO Exception when loading style sheet " + filename);
			}
		}
	}

	public RtfStyleMap(String filename, RtfAbstractStylesheet astylessheet)
	{
		_astylessheet = astylessheet;

		if (filename != null) {
			try {
				Load(filename);
			}
			catch (IOException e) {
				System.out.println("IO Exception when loading style sheet " + filename);
			}
		}
	}

	public Object clone()
	{
		RtfStyleMap newmap = new RtfStyleMap();
		newmap._map  = (Hashtable) _map.clone();
		newmap._keys = (Vector)    _keys.clone();
		newmap._abstract_style_provider = _abstract_style_provider;
		newmap._astylessheet = (RtfAbstractStylesheet)_astylessheet.clone();
		return newmap;
	}

	public void reset()
	{
		_map  = new Hashtable();
		_keys = new Vector();
	}

	public RtfAbstractStylesheet getRtfAbstractStylesheet()
	{
		return _astylessheet;
	}

	public void setAbstractStyleProvider(AbstractStyleProvider provider)
	{
		_abstract_style_provider = provider;
	}

	public void define(int type, String ename, String astyle, boolean predefined)
	{
		_map.put(ename, new StyleMapEntry(type, ename, astyle, predefined));
		_keys.addElement(ename);
	}

	public void defineParagraph(String ename, String astyle, boolean predefined)
	{
		define(PARAGRAPH_STYLE, ename, astyle, predefined);
	}

	public void defineCharacter(String ename, String astyle, boolean predefined)
	{
		define(CHARACTER_STYLE, ename, astyle, predefined);
	}

	public boolean isDefined(String ename)
	{
		return _map.get(ename) != null;
	}

	public String getAbstractStyleName(String ename, boolean paragraphstyle)
	{
		StyleMapEntry entry = null;
			
		if (ename != null) {
			entry = (StyleMapEntry)_map.get(ename);
		}

		if (entry == null) {
			if ((_abstract_style_provider == null) || (ename == null)) {
				if (paragraphstyle) {
					return "p";
				}
				else {
					return "cs";
				}
			}
			else {
				return _abstract_style_provider.lookup(this, ename, paragraphstyle);
			}
		}
		else {
			return entry._astyle;
		}
	}

	public String getAbstractStyleDescription(String name, boolean paragraphstyle)
	{
		String astyle = getAbstractStyleName(name, paragraphstyle);
		if (astyle != null) {
			return _astylessheet.getDescription(astyle);
		}
		else {
			return "undefined";
		}
	}

	public boolean isParagraphStyle(String ename)
	{
		StyleMapEntry entry = (StyleMapEntry)_map.get(ename);

		if (entry == null) {
			System.out.println("not found : " + ename);
			return false;
		}
		else {
			return entry._type == PARAGRAPH_STYLE;
		}
	}

	public boolean isCharacterStyle(String ename)
	{
		StyleMapEntry entry = (StyleMapEntry)_map.get(ename);

		if (entry == null) {
			return false;
		}
		else {
			return entry._type == CHARACTER_STYLE;
		}
	}

	public boolean isPredefined(String ename)
	{
		StyleMapEntry entry = (StyleMapEntry)_map.get(ename);

		if (entry == null) {
			return false;
		}
		else {
			return entry._predefined;
		}
	}

	public void clear()
	{
		_map.clear();
		_keys = new Vector();
	}

	public Enumeration getStyleNames()
	{
		return _keys.elements();
	}

	public void Dump()
	{
		System.out.println("Style map");
		for  (Enumeration  e = _keys.elements(); e.hasMoreElements() ;) {
			String key = e.nextElement().toString();
			StyleMapEntry entry = (StyleMapEntry)_map.get(key);
			System.out.println(entry._ename + " -> " + entry._astyle);
		}
	}

	public void Save(String filename) throws IOException
	{
		OutputStream ostream = new FileOutputStream(filename);
		ISO8859XMLWriter writer = new ISO8859XMLWriter(ostream);

		writer.markup("<?xml version=\"1.0\" encoding=\"ISO-8859-1\" standalone=\"yes\"?>");writer.write("\n");
		writer.markup("<!DOCTYPE stylemap [");writer.write("\n");
		writer.markup("<!ELEMENT stylemap (style*)>");writer.write("\n");
		writer.markup("<!ATTLIST stylemap majix-version CDATA #REQUIRED>");writer.write("\n");
		writer.markup("<!ELEMENT style EMPTY>");writer.write("\n");
		writer.markup("<!ATTLIST style");writer.write("\n");
		writer.markup("          type   (para|char) #REQUIRED");writer.write("\n");
		writer.markup("          ename  CDATA #REQUIRED");writer.write("\n");
		writer.markup("          astyle CDATA #REQUIRED");writer.write("\n");
		writer.markup("          atag   CDATA #IMPLIED>");writer.write("\n");
		writer.markup("]>");writer.write("\n");

		writer.startElement("stylemap");writer.write("\n");

		for  (Enumeration e = _keys.elements(); e.hasMoreElements() ;) {
			String key = e.nextElement().toString();
			StyleMapEntry entry = (StyleMapEntry)_map.get(key);
			writer.startElement("style");
			writer.attribute("type",   
				(entry._type == PARAGRAPH_STYLE) ? "para" : "char");
			writer.attribute("ename",  entry._ename);
			writer.attribute("astyle", entry._astyle);
			if (entry._predefined) {
				writer.attribute("predefined", "1");
			}
			writer.endElement("style");
			writer.write("\n");
		}

		writer.endElement("stylemap");writer.write("\n");

		writer.flush();
		ostream.close();
	}

	public void Load(String filename) throws IOException
	{
		clear();

		Parser parser = new ParserImpl();
		OpenEntity entity = EntityManagerImpl.openFile(filename);

		parser.setApplication(new StyleMapLoaderApplication());
		parser.parseDocument(entity);
	}

	public RtfAbstractStylesheet getAbstractStyleSheet()
	{
		return _astylessheet;
	}

	public void setAbstractStyleSheet(RtfAbstractStylesheet ssheet)
	{
		_astylessheet = (RtfAbstractStylesheet)ssheet.clone();
	}

	private class StyleMapEntry {

		public int     _type;
		public String  _ename; 
		public String  _astyle;
		public boolean _predefined;

		StyleMapEntry(int type, String ename, String astyle, boolean predefined)
		{
			_type       = type;
			_ename      = ename;
			_astyle     = astyle;
			_predefined = predefined;
		}
	}

	private class StyleMapLoaderApplication extends ApplicationImpl {

		public void startElement(StartElementEvent event) throws IOException
		{
			int    type = 0;
			String ename = null;
			String astyle = null;
			String atag = null;
			boolean predefined = false;

			if (event.getName().equals("style")) {
				int nAtts = event.getAttributeCount();
				for (int ii = 0; ii < nAtts; ii++) {
					String name = event.getAttributeName(ii);
					String value = event.getAttributeValue(ii);
					if (name.equals("type")) {
						if (value.equals("para")) {
							type = PARAGRAPH_STYLE;
						}
						else {
							type = CHARACTER_STYLE;
						}
					}
					else if (name.equals("ename")) {
						ename = value;
					}
					else if (name.equals("astyle")) {
						astyle = value;
					}
					else if (name.equals("atag")) {
						atag = value;
					}
					else if (name.equals("predefined")) {
						predefined = true;
					}
				}
	
				if ((type != 0) && (ename != null) && (astyle != null)) {
					define(type, ename, astyle, predefined);
				}
				else {
					System.out.println("missing ename or astyle in <style>");
				}
			}
		}
	}
}
