//===============================================================
// package : com.tetrasix.majix.uis
// class : com.tetrasix.majix.uis.ConfigEditor
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix.uis;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.SystemColor;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.util.ResourceBundle;
import java.util.Vector;

import javax.swing.Box;
import javax.swing.ButtonGroup;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import com.tetrasix.util.Configuration;
import com.tetrasix.util.RunSAXON;
import com.tetrasix.util.RunXSLP;
import com.tetrasix.util.RunXalan;

public class ConfigEditor extends JFrame
		implements ActionListener {

	//=====================================
	// FIELDS

	private JButton _cancelButton;
	private JButton _closeButton;
	private JButton _helpButton;

	private JFlexibleList _list;
	private JPanel   _elementEditionPanel;

	static private int _DEFAULT_WIDTH = 600;
	static private int _DEFAULT_HEIGHT = 500;

	static private final String _HELP_TOPIC = "config_editor";

    private JFrame _owner;
    private JFrame _thisFrame;

	private ResourceBundle _res = null;

	//=====================================
	// CONSTRUCTORS AND INITIALIZERS

	/**
	 * Constructs the window but does not displays it.
	 */

	public ConfigEditor(JFrame owner, boolean enable_cancel)
	{
		_res = ResourceBundle.getBundle("com.tetrasix.majix.uis.ConfigEditorResources");

		setTitle(_res.getString("frameTitle"));

		_owner = owner;
        _thisFrame = this;
        LFlist.add(this);
        
		build();

		if (enable_cancel) {
			enableEvents(AWTEvent.WINDOW_EVENT_MASK);
		}
		else {
			_cancelButton.setEnabled(false);
		}

	
		LoadList();
		pack();
		center();
		show();
	}

	/**
	 * Builds the widgets of the window.
	 */

    void build()
    {
        // first panel : a list with the associations between the tag descriptions
        // and the tag names

        Box mainbox = Box.createVerticalBox();

        Box plist = Box.createVerticalBox();
        plist.add(new JLabel(_res.getString("config_items_label")));
        _list = new JFlexibleList(10, this);
        plist.add(_list.getComponent());

        // 

        _elementEditionPanel = new JPanel();
            
        // eight panel : a row of three buttons "Cancel", "Close" and "Reset"

        JPanel pbutton2 = new JPanel();

        if (Configuration.getProperty("majix.enable.doc").equals("1")) {
            pbutton2.add( _helpButton = new JButton(_res.getString("help_button")));
            _helpButton.setActionCommand("help");
            _helpButton.addActionListener(this);
        }

        pbutton2.add( _cancelButton = new JButton(_res.getString("cancel_button")));
        _cancelButton.setActionCommand("cancel");
        _cancelButton.addActionListener(this);

        pbutton2.add( _closeButton = new JButton(_res.getString("close_button")));
        _closeButton.setActionCommand("close");
        _closeButton.addActionListener(this);

        Box hbox = Box.createHorizontalBox();
        hbox.add(plist);
        hbox.add(Box.createHorizontalStrut(4));
        hbox.add(_elementEditionPanel);
        hbox.add(Box.createHorizontalGlue());
        
        mainbox.add(Box.createHorizontalStrut(600));
        mainbox.add(hbox);
        mainbox.add(pbutton2);

        getContentPane().add(mainbox);

        setBackground(SystemColor.control);
    }

	//=====================================
	// EVENT PROCESSING

	protected void processWindowEvent(WindowEvent e)
	{
		if (e.getID() == WindowEvent.WINDOW_CLOSING) {
			setVisible(false);
            LFlist.remove(this);
			dispose();
		}
		else {
			super.processWindowEvent(e);
		}
	}

	public void actionPerformed(ActionEvent e)
	{
		String s = e.getActionCommand();

		if ("close".equals(s)) {
			Configuration.save();
			setVisible(false); 
            LFlist.remove(this);
			dispose();
		}
		else if ("cancel".equals(s)) {
			Configuration.reload();
			setVisible(false);
		}
		else if ("help".equals(s)) {
			showHelp();
		}
		else {
			System.out.println("unexpected action command : " + s);
		}
	}

	//=====================================
	// AUXILIARY FUNCTIONS

	void LoadList()
	{
        _list.addItem(new TemplateItem());
        _list.addItem(new HelpItem());
		_list.addItem(new MSWordItem());
		_list.addItem(new NsgmlsItem());
		_list.addItem(new XslItem());
        _list.addItem(new SaxItem());
        _list.addItem(new LookFeelItem());
	}

	void center()
	{
		Dimension wd = getSize();
		Dimension sd = Toolkit.getDefaultToolkit().getScreenSize();

		setLocation((sd.width - wd.width) / 2, (sd.height - wd.height) / 2);
	}

	private void showHelp()
	{
		ShowHelp.load(_HELP_TOPIC);
	}

	abstract class AbstractListElement1 
			implements JEditableListElement, ActionListener  {
		
		private JFlexibleList _owner;
		private boolean      _changed;

		AbstractListElement1()
		{
			_changed = false;
		}

		public void setOwner(Object list)
		{
			_owner = (JFlexibleList)list;
		}

		public void setChanged(boolean changed)
		{
			_changed = changed;
		}

		void UpdateParam()
		{
		}

		public boolean isChanged()
		{
			return _changed;
		}

		public void startEdit(Object context)
		{
		}

		public void validateEdit(Object context)
		{
		}

		public void cancelEdit(Object context)
		{
		}

		public void actionPerformed(ActionEvent e)
		{
			if (e.getSource() instanceof JButton) {
				String s = e.getActionCommand();
				if (s.equals("validate")) {
					_list.validateEditList();
				}
			}
		}
        
        Component toLeft(Component c)
        {
            Box box = Box.createHorizontalBox();
            box.add(c);
            box.add(Box.createHorizontalGlue());
            return box;
        }
	}

	class HelpItem 
			extends AbstractListElement1{

		private JCheckBox  _checkbox = null;
		private JButton    _button   = null;
		private Box        _box      = null;
		private JTextField _text1    = null;
        private JLabel     _label1   = null;
        private JLabel     _label2   = null;
		private JTextField _text2    = null;

		HelpItem()
		{
		}

		boolean enabled()
		{
			return Configuration.getProperty("majix.enable.doc").equals("1");
		}

		void enable(boolean state)
		{
			Configuration.setProperty("majix.enable.doc", state ? "1" : "");
		}

		public String description()
		{
			return _res.getString("help.description") + " (" 
					+ _res.getString(enabled() ? "yes" : "no") + ")";
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_checkbox  = new JCheckBox(_res.getString("help.checkbox"));
			if (Configuration.getProperty("majix.enable.doc").equals("1")) {
				_checkbox.setSelected(true);
			}
            _box.add(toLeft(_checkbox));
            _box.add(Box.createVerticalStrut(4));

            _box.add(toLeft(_label1 = new JLabel(_res.getString("help.label1"))));
			_box.add(_text1 = new JTextField(40));
			_text1.setText(Configuration.getProperty("majix.dir.doc"));
            _box.add(Box.createVerticalStrut(4));

			_box.add(toLeft(_label2 = new JLabel(_res.getString("help.label2"))));
			_box.add(_text2 = new JTextField(40));
			_text2.setText(Configuration.getProperty("majix.tool.browser"));
            _box.add(Box.createVerticalStrut(4));

            _button = new JButton(_res.getString("validate_button"));
            _button.addActionListener(this);
            _button.setActionCommand("validate");
			_box.add(toLeft(_button));

            _box.add(Box.createVerticalGlue());

             _elementEditionPanel.add(_box);

			validate();
            pack();
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

            _box = null;
			_checkbox = null;
			_button = null;
			_label1 = null;
			_label2 = null;
			_text1 = null;
			_text2 = null;

			validate();
		}

		public void validateEdit(Object context)
		{
			enable(_checkbox.isSelected());
			Configuration.setProperty("majix.dir.doc", _text1.getText());
			Configuration.setProperty("majix.tool.browser", _text2.getText());
		
			setChanged(true);
		}
	}


	class NsgmlsItem 
			extends AbstractListElement1{

        private Box        _box;
		private JCheckBox  _checkbox1 = null;
		private JCheckBox  _checkbox2 = null;
		private JButton    _button = null;
		private JLabel     _label1 = null;
		private JTextField _text1 = null;
		private JLabel     _label2 = null;
		private JTextField _text2 = null;

		private final String _PATH_PREFIX = "SGML_CATALOG_FILES=";

		NsgmlsItem()
		{
		}

		boolean enabled()
		{
			return Configuration.getProperty("majix.enable.nsgmls").equals("1");
		}

		boolean isdefault()
		{
			return Configuration.getProperty("majix.check.default").equals("nsgmls");
		}

		void enable(boolean state)
		{
			Configuration.setProperty("majix.enable.nsgmls", state ? "1" : "");
		}

		void makedefault(boolean state)
		{
			Configuration.setProperty("majix.check.default", state ? "nsgmls" : "sax");
		}

		String getCatalogPaths()
		{

			String paths = Configuration.getProperty("majix.tool.nsgmls.catalog");
			if (paths.startsWith(_PATH_PREFIX)) {
				paths = paths.substring(_PATH_PREFIX.length());
			}
			return paths;
		}

		void setCatalogPaths(String paths)
		{
			Configuration.setProperty("majix.tool.nsgmls.catalog",
					_PATH_PREFIX + paths);
		}

		public String description()
		{
			return _res.getString("nsgmls.description") + " (" 
					+ _res.getString(enabled() ? "yes" : "no") + ")";
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();

			_checkbox1  = new JCheckBox(_res.getString("nsgmls.checkbox1"));
			if (Configuration.getProperty("majix.enable.nsgmls").equals("1")) {
				_checkbox1.setSelected(true);
			}

			_checkbox2  = new JCheckBox(_res.getString("nsgmls.checkbox2"));
			if (Configuration.getProperty("majix.check.default").equals("nsgmls")) {
				_checkbox2.setSelected(true);
			}

			_box.add(toLeft(_checkbox1));
            _box.add(Box.createVerticalStrut(4));
			_box.add(toLeft(_checkbox2));
            _box.add(Box.createVerticalStrut(4));

			_box.add(toLeft(_label1 = new JLabel(_res.getString("nsgmls.label1"))));
			_box.add(_text1 = new JTextField(40));
			_text1.setText(Configuration.getProperty("majix.tool.nsgmls"));
            _box.add(Box.createVerticalStrut(4));

			_box.add(toLeft(_label2 = new JLabel(_res.getString("nsgmls.label2"))));
			_box.add(_text2 = new JTextField(40));
			_text2.setText(getCatalogPaths());
            _box.add(Box.createVerticalStrut(4));

            _box.add(toLeft(_button = new JButton(_res.getString("validate_button"))));
            _button.addActionListener(this);
            _button.setActionCommand("validate");

			_elementEditionPanel.add(_box);

			validate();
            pack();
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

			_checkbox1 = null;
			_checkbox2 = null;
			_button = null;
			_label1 = null;
			_label2 = null;
			_text1 = null;
			_text2 = null;

			validate();
		}

		public void validateEdit(Object context)
		{
			enable(_checkbox1.isSelected());
			makedefault(_checkbox2.isSelected());
			Configuration.setProperty("majix.tool.nsgmls", _text1.getText());
			setCatalogPaths(_text2.getText());
		
			setChanged(true);
		}
	}


	class XslItem 
			extends AbstractListElement1{

        private Box           _box = null;
		private JButton       _button = null;
		private JRadioButton  _radiobutton1 = null;
		private JRadioButton  _radiobutton2 = null;
		private JRadioButton  _radiobutton3 = null;
		private JRadioButton  _radiobutton4 = null;
		private JTextField    _text2 = null;
        private JLabel        _label2 = null;
		private ButtonGroup   _buttongroup = null;

		XslItem()
		{
		}

		public String description()
		{
			return _res.getString("xsl.description");
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_box.add(toLeft(_label2 = new JLabel(_res.getString("xsl.label2"))));
			_box.add(_text2 = new JTextField(40));
			_text2.setText(Configuration.getProperty("xsl.stylesheet.default"));
			

			String processor = Configuration.getProperty("majix.xsl.processor");
			int state;
			if (processor.equals("xalan")) {
				state = 1;
			}
			else if (processor.equals("xslp")) {
				state = 2;
			}
			else if (processor.equals("saxon")) {
				state = 3;
			}
			else {
				state = 0;
			}

			_buttongroup = new ButtonGroup();

			// XT
			_box.add(toLeft(_radiobutton1 = new JRadioButton(_res.getString("xsl.label1a"), state == 0)));
            _buttongroup.add(_radiobutton1);

			// Xalan
			if (RunXalan.isAvailable()) {
				_box.add(toLeft(_radiobutton2  = new JRadioButton(_res.getString("xsl.label1b"), state == 1)));
                _buttongroup.add(_radiobutton2);
			}

			// XSL:P
			if (RunXSLP.isAvailable()) {
				_box.add(toLeft(_radiobutton3  = new JRadioButton(_res.getString("xsl.label1c"), state == 2)));
                _buttongroup.add(_radiobutton3);
			}

            // SAXON
            if (RunSAXON.isAvailable()) {
                _box.add(toLeft(_radiobutton4  = new JRadioButton(_res.getString("xsl.label1d"), state == 3)));
                _buttongroup.add(_radiobutton4);
            }

            _button = new JButton(_res.getString("validate_button"));
            _button.addActionListener(this);
            _button.setActionCommand("validate");
			_box.add(toLeft(_button));

			_elementEditionPanel.add(_box);

			validate();
            pack();
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

			_button = null;
			_label2 = null;
			_text2 = null;

			validate();
		}

		public void validateEdit(Object context)
		{
			if ((_radiobutton3 != null) && (_radiobutton3.isSelected())) {
				Configuration.setProperty("majix.xsl.processor", "xslp");
			}
			else if ((_radiobutton2 != null) && (_radiobutton2.isSelected())) {
				Configuration.setProperty("majix.xsl.processor", "xalan");
			}
			else if ((_radiobutton4 != null) && (_radiobutton4.isSelected())) {
				Configuration.setProperty("majix.xsl.processor", "saxon");
			}
			else {
				Configuration.setProperty("majix.xsl.processor", "xt");
			}

			Configuration.setProperty("xsl.stylesheet.default", _text2.getText());
		
			setChanged(true);
		}
	}


	class MSWordItem 
			extends AbstractListElement1{

        private Box        _box = null;
		private JCheckBox  _checkbox = null;
		private JButton    _button = null;

		MSWordItem()
		{
		}

		boolean enabled()
		{
			return Configuration.getProperty("majix.enable.msword").equals("1");
		}

		void enable(boolean state)
		{
			Configuration.setProperty("majix.enable.msword", state ? "1" : "");
		}

		public String description()
		{
			return _res.getString("msword.description") + " (" 
					+ _res.getString(enabled() ? "yes" : "no") + ")";
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_checkbox  = new JCheckBox(_res.getString("msword.checkbox"));
			if (Configuration.getProperty("majix.enable.msword").equals("1")) {
				_checkbox.setSelected(true);
			}
            _box.add(toLeft(_checkbox));
            _box.add(Box.createVerticalStrut(4));

            _button = new JButton(_res.getString("validate_button"));
            _button.addActionListener(this);
            _button.setActionCommand("validate");
			_box.add(toLeft(_button));

			_elementEditionPanel.add(_box);

			validate();
            pack();
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

            _box = null;
			_checkbox = null;
			_button = null;

			validate();
		}

		public void validateEdit(Object context)
		{
			enable(_checkbox.isSelected());
		
			setChanged(true);
		}
	}

	class SaxItem 
			extends AbstractListElement1{ 

        private Box        _box = null;
		private JButton    _button = null;
		private JTextField _text1 = null;
		private JTextField _text2 = null;
        private JLabel     _label1 = null;
        private JLabel     _label2 = null;
        
		SaxItem()
		{
		}

		public String description()
		{
			return _res.getString("sax.description");
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_box.add(toLeft(_label2 = new JLabel(_res.getString("sax.label2"))));
			_box.add(_text2 = new JTextField(40));
            _box.add(Box.createVerticalStrut(4));

			String driver = Configuration.getProperty("sax.driver.class");
			if (driver.equals("")) {
				driver = "com.jclark.xml.sax.CommentDriver";
			}
			_text2.setText(driver);
			
            _button = new JButton(_res.getString("validate_button"));
            _button.addActionListener(this);
            _button.setActionCommand("validate");
			_box.add(toLeft(_button));

			_elementEditionPanel.add(_box);

			validate();
            pack();
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

            _box = null;
			_button = null;
			_label1 = null;
			_label2 = null;
			_text1 = null;
			_text2 = null;

			validate();
		}

		public void validateEdit(Object context)
		{
			Configuration.setProperty("sax.driver.class", _text2.getText());
		
			setChanged(true);
		}
	}
    
    class LookFeelItem 
            extends AbstractListElement1 {
                
        private Box          _box          = null;
        private JButton      _button       = null;
        private JRadioButton _radiobuttons[];
        private ButtonGroup  _buttongroup  = null;
        UIManager.LookAndFeelInfo _infos[]   = null;
        
        LookFeelItem()
        {
        }

        public String description()
        {
            return _res.getString("lookandfeel.description");
        }


        public void startEdit(Object context)
        {
            _box = Box.createVerticalBox();
            
            
            _infos = UIManager.getInstalledLookAndFeels();
            _radiobuttons = new JRadioButton[_infos.length];

            String lookandfeel = Configuration.getProperty("majix.lookandfeel");
            _buttongroup = new ButtonGroup();
            
            int ii;
            int state = 0;
            for (ii = 0; ii < _infos.length; ii++) {
                if (lookandfeel.equals(_infos[ii].getClassName())) {
                    state = ii;
                }
                _box.add(toLeft(_radiobuttons[ii] 
                        = new JRadioButton(_infos[ii].getName(), state == ii)));
                _buttongroup.add(_radiobuttons[ii]);
            }

            _button = new JButton(_res.getString("validate_button"));
            _button.addActionListener(this);
            _button.setActionCommand("validate");
            _box.add(toLeft(_button));

            _elementEditionPanel.add(_box);

            validate();
            pack();
        }

        public void cancelEdit(Object context)
        {
            _elementEditionPanel.remove(_box);

            _box          = null;
            _button       = null;
            _buttongroup  = null;
            _radiobuttons = null;
            _infos        = null;
            
            validate();
        }

        public void validateEdit(Object context)
        {
            try {

                int ii;
                for (ii = 0; ii < _infos.length; ii++) {
                    if (_radiobuttons[ii].isSelected()) {
                        Configuration.setProperty("majix.lookandfeel", _infos[ii].getClassName());
                        UIManager.setLookAndFeel(_infos[ii].getClassName());
                    }
                }
                
                LFlist.updateUI();
            }
            catch (Exception e) {
                System.err.println("Unable to change look and feel");
            }
            
            setChanged(true);
        }
    }

    class TemplateItem 
            extends AbstractListElement1
            implements ListSelectionListener {
         
               
        private JPanel       _panel            = null;
        private JButton      _newButton        = null;
        private JButton      _deleteButton     = null;
        private JButton      _upButton         = null;
        private JButton      _downButton       = null;
        private JButton      _validateButton   = null;
        
        private JLabel            _label               = null;
        private JList             _templatesList       = null;
        private DefaultListModel  _templates           = null;
        private Vector            _templatePaths       = null;
        private JScrollPane       _templatesScrollPane = null;
            
        TemplateItem()
        {
        }

        public String description()
        {
            return _res.getString("template.description");
        }


        public void startEdit(Object context)
        {
            _panel = new JPanel();
            _panel.setLayout(new GridBagLayout());
            
            _templates = new DefaultListModel();
            _templatePaths = new Vector();
            
            int count = Configuration.getTemplateCount();
            for (int ii = 0; ii < count; ii++) {
                _templates.addElement(Configuration.getTemplateName(ii));
                _templatePaths.addElement(Configuration.getTemplatePath(ii));
            }
                
            addComponent(_panel, _label = new JLabel("Template name"),
                         0, 0, 1, 1,
                         GridBagConstraints.BOTH, GridBagConstraints.CENTER);
            addComponent(_panel, _templatesScrollPane = new JScrollPane(_templatesList = new JList(_templates)),
                         0, 1, 1, 5,
                         GridBagConstraints.BOTH, GridBagConstraints.CENTER);
            _templatesList.setFixedCellWidth(100);
                                     
            addComponent(_panel, _newButton = new JButton("New"),
                         1, 1, 2, 1,
                         GridBagConstraints.BOTH, GridBagConstraints.CENTER);
            addComponent(_panel, _deleteButton = new JButton("Delete"),
                         1, 2, 2, 1,
                         GridBagConstraints.BOTH, GridBagConstraints.CENTER);
            addComponent(_panel, _upButton = new JButton("Up"),
                         1, 3, 1, 1,
                         GridBagConstraints.BOTH, GridBagConstraints.CENTER);
            addComponent(_panel, _downButton = new JButton("Down"),
                         2, 3, 1, 1,
                         GridBagConstraints.BOTH, GridBagConstraints.CENTER);
            addComponent(_panel, _validateButton = new JButton(_res.getString("validate_button")),
                         1, 4, 2, 1,
                         GridBagConstraints.BOTH, GridBagConstraints.CENTER);
            
            
            _templatesList.addListSelectionListener(this);
            
            _newButton.addActionListener(this);
            _newButton.setActionCommand("new");

            _deleteButton.addActionListener(this);
            _deleteButton.setActionCommand("delete");
            
            _upButton.addActionListener(this);
            _upButton.setActionCommand("up");
            
            _downButton.addActionListener(this);
            _downButton.setActionCommand("down");
            
            _validateButton.addActionListener(this);
            _validateButton.setActionCommand("validate");

            _elementEditionPanel.add(_panel);

            validate();
            pack();
            
            checkButtons();
        }

        public void cancelEdit(Object context)
        {
            _elementEditionPanel.remove(_panel);

            _panel               = null;
            _templatesList       = null;
            _templatesScrollPane = null;
            _templates           = null;
            _newButton           = null;
            _deleteButton        = null;
            _upButton            = null;
            _downButton          = null;
            _validateButton      = null;
            
            validate();
        }

        public void validateEdit(Object context)
        {
            Configuration.clearTemplates();
            int size = _templates.size();
            for (int ii = 0; ii < size; ii++) {
                Configuration.addTemplate(
                        (String) _templates.elementAt(ii), 
                        (String) _templatePaths.elementAt(ii));
            }
            
            setChanged(true);
        }
 
        public void actionPerformed(ActionEvent e)
        {
            if (e.getSource() instanceof JButton) {
                String s = e.getActionCommand();
                if (s.equals("new")) {
                    doNew();
                }
                else if (s.equals("delete")) {
                    doDelete();
                }
                else if (s.equals("up")) {
                    doUp();
                }
                else if (s.equals("down")) {
                    doDown();
                }
                else {
                    super.actionPerformed(e);
                }
            }
        }

        private void doNew()
        {
            JFileChooser chooser = new JFileChooser(System.getProperty("user.dir"));
            chooser.setDialogTitle("Select TDEF file");
            javax.swing.filechooser.FileFilter filter = new ExtensionFileFilter("tdef", "Template definition files");
            chooser.setFileFilter(filter);
            
            int option = chooser.showOpenDialog(_thisFrame);
            if (option == JFileChooser.APPROVE_OPTION) {
                String templatename = chooser.getSelectedFile().getName();
                String templatepath = chooser.getSelectedFile().getPath();
                templatename = templatename.substring(0, templatename.length() - 5);
                _templates.addElement(templatename);
                _templatePaths.addElement(templatepath);
            }
            checkButtons();
        }

        private void doDelete()
        {
            int selected = _templatesList.getSelectedIndex();
            if (selected > 0) {
                _templates.removeElementAt(selected);
                _templatePaths.removeElementAt(selected);
            }
            checkButtons();
        }

        private void doUp()
        {
            int selected = _templatesList.getSelectedIndex();
            if (selected > 0) {
                Object o = _templates.elementAt(selected);
                _templates.setElementAt(_templates.elementAt(selected - 1), selected);
                _templates.setElementAt(o, selected - 1);
                _templatesList.setSelectedIndex(selected - 1);

                o = _templatePaths.elementAt(selected);
                _templatePaths.setElementAt(_templates.elementAt(selected - 1), selected);
                _templatePaths.setElementAt(o, selected - 1);
            }
        }

        private void doDown()
        {
            int selected = _templatesList.getSelectedIndex();
            if ((selected >= 0) && (selected < _templates.size())) {
                Object o = _templates.elementAt(selected);
                _templates.setElementAt(_templates.elementAt(selected + 1), selected);
                _templates.setElementAt(o, selected + 1);
                _templatesList.setSelectedIndex(selected + 1);

                o = _templatePaths.elementAt(selected);
                _templatePaths.setElementAt(_templates.elementAt(selected + 1), selected);
                _templatePaths.setElementAt(o, selected + 1);
            }
        }
        
        public void valueChanged(ListSelectionEvent e)
        {
            checkButtons();
        }
        
        private void checkButtons()
        {
            int selected = _templatesList.getSelectedIndex();
            boolean enabled = selected >= 0;
            _deleteButton.setEnabled(enabled);
            _upButton.setEnabled(enabled);
            _downButton.setEnabled(enabled);
        }
    }

    private static void addComponent (
        Container container, Component component, 
        int gridx, int gridy, int gridwidth, int gridheight,
        int fill, int anchor)
    {
        LayoutManager lm = container.getLayout();
        if (! (lm instanceof GridBagLayout)) {
              System.out.println("Invalid layout : " + lm);
              container.add(component);
              return;
        }
        
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = gridx;
        gbc.gridy = gridy;
        gbc.gridwidth = gridwidth;
        gbc.gridheight = gridheight;
        gbc.fill = fill;
        gbc.anchor = anchor;
        gbc.insets = new Insets(2, 2, 2, 2);
        ((GridBagLayout)lm).setConstraints(component, gbc);
        container.add(component);
    }

}