//===============================================================
// package : com.tetrasix.majix.uis
// class : com.tetrasix.majix.uis.RtfStylesEditor
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================
/**
 * This class provides an window to interactively edit a style map
 * 
 * @version 1.1
 */
package com.tetrasix.majix.uis;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import javax.swing.Box;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import com.jacob.activeX.ActiveXComponent;
import com.jacob.com.Dispatch;
import com.jacob.com.Variant;
import com.tetrasix.majix.rtf.RtfAbstractStylesheet;
import com.tetrasix.majix.rtf.RtfAnalyser;
import com.tetrasix.majix.rtf.RtfDocument;
import com.tetrasix.majix.rtf.RtfReader;
import com.tetrasix.majix.rtf.RtfStyleMap;
import com.tetrasix.majix.rtf.RtfStyleSheet;
import com.tetrasix.majix.xml.ConversionTemplate;
import com.tetrasix.util.Configuration;

public class RtfStylesEditor extends JFrame implements ActionListener {

    private ConversionTemplate _template;

    private RtfStyleMap _map;

    private MajixEventListener _listener;

    private JFlexibleList _list;

    private JPanel _elementEditionPanel;

    private JButton _addFromRtfButton;

    private JButton _cancelButton;

    private JButton _closeButton;

    private JButton _saveButton;

    private JButton _loadButton;

    private JButton _resetButton;

    private JButton _helpButton;

    private ResourceBundle _res;

    static private final String _HELP_TOPIC = "rtf_styles_editor";

    /**
     * create the editor but does not show it
     */
    public RtfStylesEditor() {
        _res = ResourceBundle
                .getBundle("com.tetrasix.majix.uis.RtfStylesEditorResources");
        setTitle(_res.getString("frameTitle"));
        enableEvents(AWTEvent.WINDOW_EVENT_MASK);
        build();
        pack();
        LFlist.add(this);
    }

    /**
     * load the editor with a style map to edit
     */
    public void Edit(ConversionTemplate template, MajixEventListener listener) {
        _template = template;
        _listener = listener;
        _map = (RtfStyleMap) _template.getStyleMap().clone();
        LoadList();
        center();
        show();
    }

    /**
     * build the widgets
     */
    private void build() {
        Box mainbox = Box.createVerticalBox();
        Box plist = Box.createVerticalBox();
        plist.add(new JLabel(_res.getString("list_label")));
        _list = new JFlexibleList(15, this, 200);
        plist.add(_list.getComponent());
        _elementEditionPanel = new JPanel();
        JPanel pbutton1 = new JPanel();
        pbutton1.add(_saveButton = new JButton(_res.getString("save_button")));
        _saveButton.addActionListener(this);
        _saveButton.setActionCommand("save");
        pbutton1.add(_loadButton = new JButton(_res.getString("load_button")));
        _loadButton.addActionListener(this);
        _loadButton.setActionCommand("load");
        pbutton1
                .add(_resetButton = new JButton(_res.getString("reset_button")));
        _resetButton.setActionCommand("reset");
        _resetButton.addActionListener(this);
        //		pbutton1.add( _defaultButton = new
        // JButton(_res.getString("default_button")));
        //		_defaultButton.setActionCommand("default");
        //		_defaultButton.addActionListener(this);
        JPanel pbutton2 = new JPanel();
        if (Configuration.getProperty("majix.enable.doc").equals("1")) {
            pbutton2.add(_helpButton = new JButton(_res
                    .getString("help_button")));
            _helpButton.setActionCommand("help");
            _helpButton.addActionListener(this);
        }
        pbutton2.add(_addFromRtfButton = new JButton(_res
                .getString("add_from_rtf_button")));
        _addFromRtfButton.setActionCommand("addfromrtf");
        _addFromRtfButton.addActionListener(this);
        pbutton2.add(_cancelButton = new JButton(_res
                .getString("cancel_button")));
        _cancelButton.setActionCommand("cancel");
        _cancelButton.addActionListener(this);
        pbutton2
                .add(_closeButton = new JButton(_res.getString("close_button")));
        _closeButton.setActionCommand("close");
        _closeButton.addActionListener(this);
        Box hbox = Box.createHorizontalBox();
        hbox.add(plist);
        hbox.add(Box.createHorizontalStrut(4));
        hbox.add(_elementEditionPanel);
        hbox.add(Box.createHorizontalGlue());
        mainbox.add(hbox);
        mainbox.add(pbutton1);
        mainbox.add(pbutton2);
        mainbox.add(Box.createHorizontalStrut(600));
        setBackground(SystemColor.control);
        getContentPane().add(mainbox);
    }

    public void actionPerformed(ActionEvent e) {
        EnableButtons(false);
        String s = e.getActionCommand();
        boolean ret = true;
        if ("close".equals(s)) {
            SaveStyle();
            SaveAsDefault();
            setVisible(false);
            LFlist.remove(this);
            dispose();
        } else if ("cancel".equals(s)) {
            setVisible(false);
            LFlist.remove(this);
            dispose();
        } else if ("save".equals(s)) {
            UpdateStyleMap();
            SaveToFile();
        } else if ("load".equals(s)) {
            LoadFromFile();
        } else if ("reset".equals(s)) {
            reset();
        } else if ("default".equals(s)) {
            reset_default();
        } else if ("addfromrtf".equals(s)) {
            addFromWord();
        } else if ("help".equals(s)) {
            showHelp();
        } else {
            System.out.println("unexpected action command : " + s);
        }
        EnableButtons(true);
    }

    protected void processWindowEvent(WindowEvent e) {
        if (e.getID() == WindowEvent.WINDOW_CLOSING) {
            setVisible(false);
            LFlist.remove(this);
            dispose();
        } else {
            super.processWindowEvent(e);
        }
    }

    void LoadList() {
        _list.removeAll();
        _list.addItem(new RtfStylesEditor.SeparatorListElement1(_res
                .getString("sep_pstyles")));
        for (Enumeration e1 = _map.getStyleNames(); e1.hasMoreElements();) {
            String ename = e1.nextElement().toString();
            String astyle = _map.getAbstractStyleName(ename, true);
            boolean predefined = _map.isPredefined(ename);
            if (_map.isParagraphStyle(ename) && predefined
                    && ename.charAt(0) != '*') {
                _list.addItem(new ParagraphListElement(ename, astyle,
                        predefined));
            }
        }
        _list.addItem(new RtfStylesEditor.SeparatorListElement1(_res
                .getString("sep_pstyles_pseudo")));
        for (Enumeration e1 = _map.getStyleNames(); e1.hasMoreElements();) {
            String ename = e1.nextElement().toString();
            String astyle = _map.getAbstractStyleName(ename, true);
            boolean predefined = _map.isPredefined(ename);
            if (_map.isParagraphStyle(ename) && predefined
                    && ename.charAt(0) == '*') {
                _list.addItem(new ParagraphListElement(ename, astyle,
                        predefined));
            }
        }
        _list.addItem(new RtfStylesEditor.SeparatorListElement1(_res
                .getString("sep_pstyles_user")));
        for (Enumeration e1 = _map.getStyleNames(); e1.hasMoreElements();) {
            String ename = e1.nextElement().toString();
            String astyle = _map.getAbstractStyleName(ename, true);
            boolean predefined = _map.isPredefined(ename);
            if (_map.isParagraphStyle(ename) && !predefined) {
                _list.addItem(new ParagraphListElement(ename, astyle,
                        predefined));
            }
        }
        _list.addItem(new RtfStylesEditor.SeparatorListElement1(_res
                .getString("sep_cstyles")));
        for (Enumeration e2 = _map.getStyleNames(); e2.hasMoreElements();) {
            String ename = e2.nextElement().toString();
            String astyle = _map.getAbstractStyleName(ename, false);
            boolean predefined = _map.isPredefined(ename);
            if (_map.isCharacterStyle(ename) & predefined) {
                _list.addItem(new CharacterListElement(ename, astyle,
                        predefined));
            }
        }
        _list.addItem(new RtfStylesEditor.SeparatorListElement1(_res
                .getString("sep_cstyles_user")));
        for (Enumeration e2 = _map.getStyleNames(); e2.hasMoreElements();) {
            String ename = e2.nextElement().toString();
            String astyle = _map.getAbstractStyleName(ename, false);
            boolean predefined = _map.isPredefined(ename);
            if (_map.isCharacterStyle(ename) & !predefined) {
                _list.addItem(new CharacterListElement(ename, astyle,
                        predefined));
            }
        }
    }

    static String getStyleDesc(String style) {
        int index_semicolon = style.indexOf(':');
        return style.substring(index_semicolon + 2);
    }

    static String getStyleName(String style) {
        int index_semicolon = style.indexOf(':');
        return style.substring(0, index_semicolon - 1);
    }

    void UpdateStyleMap() {
        _map.clear();
        for (Enumeration e = _list.getElements(); e.hasMoreElements();) {
            RtfStylesEditor.ListElement elem = (RtfStylesEditor.ListElement) e
                    .nextElement();
            elem.UpdateStyleMap();
        }
    }

    void SaveStyle() {
        UpdateStyleMap();
        _listener.onEvent(new MajixEvent(MajixEvent.EVENT_UPDATE_STYLE_MAP,
                _map.clone()));
    }

    String getWordFileName() {
        FileDialog fd = new FileDialog(this,
                _res.getString("select_rtf_label"), FileDialog.LOAD);
        FilenameFilter ff;
        if (Configuration.getProperty("majix.enable.msword").equals("1")) {
            ff = new ExtensionFilenameFilter(".doc", ".rtf");
            fd.setFile("*.doc;*.rtf");
        } else {
            ff = new ExtensionFilenameFilter(".rtf");
            fd.setFile("*.rtf");
        }
        fd.setFilenameFilter(ff);
        Point loc = getLocationOnScreen();
        loc.translate(10, 10);
        fd.setLocation(loc);
        fd.show();
        if (fd.getFile() != null) {
            return fd.getDirectory() + fd.getFile();
        } else {
            return null;
        }
    }

    void addFromWord() {
        String filename = getWordFileName();
        if (filename == null) { return; }
        String rtfinputfile = filename;
        boolean must_delete = false;
        if (filename.toLowerCase().endsWith(".doc")) {
            String docfilename = filename;
            String rtffilename = filename.substring(0, filename.length() - 4)
                    + ".rtf";
            try {
                ActiveXComponent app = new ActiveXComponent("Word.Application");
                try {
                    Object documents = app.getProperty("Documents")
                            .toDispatch();
                    Object document = Dispatch.invoke(
                            documents,
                            "Open",
                            Dispatch.Method,
                            new Object[] { docfilename, new Variant(false),
                                    new Variant(true)}, new int[1])
                            .toDispatch();
                    Dispatch.invoke(document, "SaveAs", Dispatch.Method,
                            new Object[] { rtffilename, new Variant(6)},
                            new int[1]).toDispatch();
                    Dispatch.invoke(document, "Close", Dispatch.Method,
                            new Object[] { new Variant(0)}, new int[1])
                            .toDispatch();
                    Dispatch.invoke(app.getObject(), "Quit", Dispatch.Method,
                            new Object[] { new Variant(-2)}, new int[1])
                            .toDispatch();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            } catch (Exception e) {
                System.out.println("Unexpected exception : " + e.getMessage());
                return;
            }
            if (rtfinputfile == null) {
                System.out.println("Unable to convert into RTF : " + filename);
                return;
            } else {
                must_delete = true;
            }
        }
        try {
            RtfReader reader;
            reader = new RtfReader(filename);
            RtfAnalyser ana = new RtfAnalyser(reader, new PrintWriter(
                    System.out));
            RtfDocument doc = ana.parse();
            RtfStyleSheet ssheet = doc.getStyleSheet();
            for (int ii = 0; ii < 255; ii++) {
                if (ssheet.isParagraphStyle(ii)) {
                    String name = ssheet.getStyleName(ii);
                    if (!_map.isDefined(name)) {
                        _map.defineParagraph(name, "p", false);
                    }
                }
            }
            for (int jj = 0; jj < 255; jj++) {
                if (ssheet.isCharacterStyle(jj)) {
                    String name = ssheet.getStyleName(jj);
                    if (!_map.isDefined(name)) {
                        _map.defineCharacter(name, "cs", false);
                    }
                }
            }
            LoadList();
        } catch (FileNotFoundException e) {
            System.out.println("File not found : " + filename);
            return;
        } catch (IOException e) {
            System.out.println("IO Exception on : " + filename);
            return;
        }
        if (must_delete) {
            if (!(new File(rtfinputfile).delete())) {
                System.out.println("Unable to delete file " + rtfinputfile);
            }
        }
    }

    void EnableButtons(boolean enabled) {
        _saveButton.setEnabled(enabled);
        _loadButton.setEnabled(enabled);
        _resetButton.setEnabled(enabled);
        //		_defaultButton.setEnabled(enabled);
        _addFromRtfButton.setEnabled(enabled);
        _cancelButton.setEnabled(enabled);
        _closeButton.setEnabled(enabled);
    }

    String getStyleFileName(boolean save) {
        FileDialog fd = new FileDialog(this,
                _res.getString("select_sty_label"), save ? FileDialog.SAVE
                        : FileDialog.LOAD);
        FilenameFilter ff = new ExtensionFilenameFilter(".sty");
        fd.setFilenameFilter(ff);
        fd.setFile("*.sty");
        Point loc = getLocationOnScreen();
        loc.translate(10, 10);
        fd.setLocation(loc);
        fd.show();
        if (fd.getFile() != null) {
            String file = fd.getFile();
            if (file.endsWith(".*.*")) {
                file = file.substring(0, file.length() - 4);
            }
            return fd.getDirectory() + file;
        } else {
            return null;
        }
    }

    void LoadFromFile() {
        String filename = getStyleFileName(false);
        if (filename != null) {
            try {
                _map.Load(filename);
                LoadList();
            } catch (IOException e) {
                System.out.println("IO Exception when loading stytle map from "
                        + filename + " : " + e.getMessage());
            }
        }
    }

    private String getStyleFileName() {
        String filesep = System.getProperty("file.separator");
        String userdir = System.getProperty("user.dir");
        String appdir = System.getProperty("majix.dir", userdir);
        String filename = System.getProperty("default.sty", "default.sty");
        return appdir + filesep + filename;
    }

    void SaveAsDefault() {
        String filename = getStyleFileName();
        SaveToFile(filename);
    }

    void SaveToFile() {
        String filename = getStyleFileName(true);
        if (filename != null) {
            SaveToFile(filename);
        }
    }

    void SaveToFile(String filename) {
        try {
            _map.Save(filename);
        } catch (IOException e) {
            System.out.println("IO Exception when saving style sheet to "
                    + filename);
        }
    }

    void reset() {
        _map = (RtfStyleMap) _template.getStyleMap().clone();
        LoadList();
    }

    void reset_default() {
        _map = (RtfStyleMap) _template.getStyleMap().clone();
        LoadList();
    }

    void center() {
        Dimension wd = getSize();
        Dimension sd = Toolkit.getDefaultToolkit().getScreenSize();
        setLocation((sd.width - wd.width) / 2, (sd.height - wd.height) / 2);
    }

    private void showHelp() {
        ShowHelp.load(_HELP_TOPIC);
    }

    private String translate(String name) {
        String ret;
        try {
            ret = _res.getString(name);
        } catch (MissingResourceException e) {
            ret = name;
        }
        return ret;
    }

    abstract class ListElement implements JEditableListElement {

        abstract String getStyleName();

        abstract String getAbstractStyle();

        abstract String getAbstractTag();

        public void UpdateStyleMap() {
        }

        Component toLeft(Component c) {
            Box box = Box.createHorizontalBox();
            box.add(c);
            box.add(Box.createHorizontalGlue());
            return box;
        }
    }

    abstract class StyleListElement extends ListElement {

        String _ename;

        String _astyle;

        String _desc;

        JFlexibleList _owner;

        boolean _changed;

        boolean _predefined;

        Box _box = null;

        JLabel _label1 = null;

        JLabel _label2 = null;

        JComboBox _combobox = null;

        DefaultComboBoxModel _comboboxmodel = null;

        JButton _button = null;

        StyleListElement(String ename, String astyle, boolean predefined) {
            RtfAbstractStylesheet assheet = _map.getAbstractStyleSheet();
            _ename = ename;
            _astyle = astyle;
            _desc = assheet.getDescription(astyle);
            _changed = false;
            _predefined = predefined;
        }

        public void setOwner(Object list) {
            _owner = (JFlexibleList) list;
        }

        String getStyleName() {
            return _ename;
        }

        String getAbstractStyle() {
            return _astyle;
        }

        public void setChanged(boolean changed) {
            _changed = changed;
        }

        public boolean isChanged() {
            return _changed;
        }

        public void cancelEdit(Object context) {
            _elementEditionPanel.remove(_box);
            _box = null;
            _label1 = null;
            _label2 = null;
            _combobox = null;
            _comboboxmodel = null;
            _button = null;
            validate();
        }

        public void actionPerformed(ActionEvent e) {
            if (e.getSource() instanceof JButton) {
                String s = e.getActionCommand();
                if (s.equals("validate")) {
                    _list.validateEditList();
                }
            }
        }

        String getAbstractTag() {
            return "";
        }
    }

    class ParagraphListElement extends StyleListElement implements
            ActionListener {

        ParagraphListElement(String ename, String astyle, boolean predefined) {
            super(ename, astyle, predefined);
        }

        public String description() {
            String ename = _ename;
            String ret = (_predefined ? translate(_ename) : _ename) + " : "
                    + translate(_desc);
            return ret;
        }

        public void startEdit(Object context) {
            _box = Box.createVerticalBox();
            _label1 = new JLabel(_res.getString("ple_label1")
                    + translate(_ename) + "                   ");
            _label2 = new JLabel(_res.getString("ple_label2"));
            _comboboxmodel = new DefaultComboBoxModel();
            _combobox = new JComboBox(_comboboxmodel);
            LoadComboBox();
            _button = new JButton(_res.getString("validate_button"));
            _button.addActionListener(this);
            _button.setActionCommand("validate");
            RtfAbstractStylesheet assheet = _map.getAbstractStyleSheet();
            _comboboxmodel.setSelectedItem(assheet.getDescription(_astyle));
            _box.add(toLeft(_label1));
            _box.add(Box.createVerticalStrut(4));
            _box.add(toLeft(_label2));
            _box.add(Box.createVerticalStrut(4));
            _box.add(_combobox);
            _box.add(Box.createVerticalStrut(4));
            _box.add(toLeft(_button));
            _elementEditionPanel.add(_box);
            validate();
        }

        private void LoadComboBox() {
            RtfAbstractStylesheet assheet = _map.getAbstractStyleSheet();
            for (Enumeration e = assheet.getParagraphStyleNames(); e
                    .hasMoreElements();) {
                String item = e.nextElement().toString();
                _combobox.addItem(_predefined ? translate(assheet
                        .getDescription(item)) : assheet.getDescription(item));
            }
        }

        public void validateEdit(Object context) {
            String desc = (String) _comboboxmodel.getSelectedItem();
            if (desc != null) {
                RtfAbstractStylesheet assheet = _map.getAbstractStyleSheet();
                _astyle = assheet.getNameFromDescription(desc);
                _desc = desc;
                setChanged(true);
            }
        }

        public void UpdateStyleMap() {
            _map.defineParagraph(_ename, _astyle, _predefined);
        }
    }

    class CharacterListElement extends StyleListElement implements
            ActionListener {

        CharacterListElement(String ename, String astyle, boolean predefined) {
            super(ename, astyle, predefined);
        }

        public String description() {
            return (_predefined ? translate(_ename) : _ename) + " : "
                    + translate(_desc);
        }

        public void startEdit(Object context) {
            _box = Box.createVerticalBox();
            _label1 = new JLabel(_res.getString("cle_label1")
                    + translate(_ename) + "                   ");
            _label2 = new JLabel(_res.getString("cle_label2"));
            _comboboxmodel = new DefaultComboBoxModel();
            _combobox = new JComboBox(_comboboxmodel);
            LoadComboBox();
            _button = new JButton(_res.getString("validate_button"));
            _button.addActionListener(this);
            _button.setActionCommand("validate");
            RtfAbstractStylesheet assheet = _map.getAbstractStyleSheet();
            _comboboxmodel.setSelectedItem(assheet.getDescription(_astyle));
            _box.add(toLeft(_label1));
            _box.add(Box.createVerticalStrut(4));
            _box.add(toLeft(_label2));
            _box.add(Box.createVerticalStrut(4));
            _box.add(_combobox);
            _box.add(Box.createVerticalStrut(4));
            _box.add(toLeft(_button));
            _elementEditionPanel.add(_box);
            validate();
        }

        private void LoadComboBox() {
            RtfAbstractStylesheet assheet = _map.getAbstractStyleSheet();
            for (Enumeration e = assheet.getCharacterStyleNames(); e
                    .hasMoreElements();) {
                String item = e.nextElement().toString();
                _combobox.addItem(_predefined ? translate(assheet
                        .getDescription(item)) : assheet.getDescription(item));
            }
        }

        public void validateEdit(Object context) {
            String desc = (String) _combobox.getSelectedItem();
            if (desc != null) {
                RtfAbstractStylesheet assheet = _map.getAbstractStyleSheet();
                _astyle = assheet.getNameFromDescription(desc);
                _desc = desc;
                setChanged(true);
            }
        }

        public void UpdateStyleMap() {
            _map.defineCharacter(_ename, _astyle, _predefined);
        }
    }

    class SeparatorListElement1 extends ListElement {

        private String _text;

        private JFlexibleList _owner;

        SeparatorListElement1(String text) {
            _text = text;
        }

        public void setOwner(Object list) {
            _owner = (JFlexibleList) list;
        }

        public void startEdit(Object context) {
        }

        public void validateEdit(Object context) {
        }

        public void cancelEdit(Object context) {
        }

        public String description() {
            return "----- " + _text + " -----";
        }

        String getStyleName() {
            return "";
        }

        String getAbstractStyle() {
            return "";
        }

        String getAbstractTag() {
            return "";
        }

        public boolean isChanged() {
            return false;
        }

        public void setChanged(boolean changed) {
        }
    }
}