//===============================================================
// package : com.tetrasix.majix.uis
// class : com.tetrasix.majix.uis.XmlTagsEditor
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix.uis;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.Label;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Enumeration;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import com.tetrasix.majix.xml.ConversionTemplate;
import com.tetrasix.majix.xml.XmlGeneratorParam;
import com.tetrasix.majix.xml.XmlTagMap;
import com.tetrasix.util.Configuration;
import com.tetrasix.util.FlexibleList;

public class XmlTagsEditor extends JFrame
		implements ActionListener {

	//=====================================
	// FIELDS

	private ConversionTemplate _template;

	private XmlGeneratorParam _param;

	private JFlexibleList _list;
	private MajixEventListener _listener;

	private JPanel   _elementEditionPanel;

	private JButton _saveButton;
	private JButton _loadButton;
	private JButton _resetButton;
	private JButton _cancelButton;
	private JButton _closeButton;
	private JButton _helpButton;

	static private int _DEFAULT_WIDTH = 500;
	static private int _DEFAULT_HEIGHT = 500;

	static private final String _HELP_TOPIC = "tags_editor";

	private ResourceBundle _res = null;

	//=====================================
	// CONSTRUCTORS AND INITIALIZERS

	/**
	 * Constructs the window but does not displays it.
	 */

	XmlTagsEditor()
	{
		_res = ResourceBundle.getBundle("com.tetrasix.majix.uis.XmlTagsEditorResources");

		setTitle(_res.getString("frameTitle"));

		enableEvents(AWTEvent.WINDOW_EVENT_MASK);
		build();
        LFlist.add(this);
	}

	/**
	 * initializes the window with the current parameters.
	 */

	void Edit(ConversionTemplate template, MajixEventListener listener)
	{
		_template = template;
		_param = (XmlGeneratorParam) _template.getGeneratorParam().clone();
		_listener = listener;

		LoadList();
		pack();
		center();
		show();
	}

	/**
	 * Builds the widgets of the window.
	 */

	void build()
	{
        Box mainbox = Box.createVerticalBox();

        Box plist = Box.createVerticalBox();
		plist.add(new Label(_res.getString("tag_list_label")));

		_list = new JFlexibleList(15, this, 200);
		plist.add(_list.getComponent());


		// seventh panel : a row of three buttons "Save", "Load" and "Reset"
		
		JPanel pbutton1 = new JPanel();

		pbutton1.add( _saveButton = new JButton(_res.getString("save_button")));
		_saveButton.setActionCommand("save");
		_saveButton.addActionListener(this);

		pbutton1.add( _loadButton = new JButton(_res.getString("load_button")));
		_loadButton.setActionCommand("load");
 		_loadButton.addActionListener(this);

		pbutton1.add( _resetButton = new JButton(_res.getString("reset_button")));
		_resetButton.setActionCommand("reset");
		_resetButton.addActionListener(this);

		// eight panel : a row of three buttons "Cancel", "Close" and "Reset"

		JPanel pbutton2 = new JPanel();

		if (Configuration.getProperty("majix.enable.doc").equals("1")) {
			pbutton2.add( _helpButton = new JButton(_res.getString("help_button")));
			_helpButton.setActionCommand("help");
			_helpButton.addActionListener(this);
		}

		pbutton2.add( _cancelButton = new JButton(_res.getString("cancel_button")));
		_cancelButton.setActionCommand("cancel");
		_cancelButton.addActionListener(this);

		pbutton2.add( _closeButton = new JButton(_res.getString("close_button")));
		_closeButton.setActionCommand("close");
		_closeButton.addActionListener(this);

        _elementEditionPanel = new JPanel();

        Box hbox = Box.createHorizontalBox();
        hbox.add(plist);
        hbox.add(Box.createHorizontalStrut(4));
        hbox.add(_elementEditionPanel);
        hbox.add(Box.createHorizontalGlue());
        
        mainbox.add(Box.createHorizontalStrut(600));
        mainbox.add(hbox);
        mainbox.add(pbutton2);

        getContentPane().add(mainbox);

		setBackground(SystemColor.control);
	}

	//=====================================
	// EVENT PROCESSING

	protected void processWindowEvent(WindowEvent e)
	{
		if (e.getID() == WindowEvent.WINDOW_CLOSING) {
			setVisible(false);
            LFlist.remove(this);
			dispose();
		}
		else {
			super.processWindowEvent(e);
		}
	}

	public void actionPerformed(ActionEvent e)
	{
		EnableButtons(false);

		String s = e.getActionCommand();

		if ("close".equals(s)) {
			Apply();
			SaveAsDefault();
			setVisible(false); 
            LFlist.remove(this);
			dispose();
		}
		else if ("cancel".equals(s)) {
			setVisible(false);
		}
		else if ("save".equals(s)) {
			SaveToFile();
		}
		else if ("load".equals(s)) {
			LoadFromFile(_template);
		}
		else if ("reset".equals(s)) {
			reset();
		}
		else if ("help".equals(s)) {
			showHelp();
		}
		else {
			System.out.println("unexpected action command : " + s);
		}

		EnableButtons(true);
	}

	void LoadFromFile(ConversionTemplate template)
	{
		String filename = getTagParamFileName(false);
		if (filename != null) {
			try {
				_param = new XmlGeneratorParam();
				template.initGenParam(_param);
				_param.Load(filename);
				LoadList();
			}
			catch (IOException e) {
				System.out.println("IO Exception when loading generator parameters" + e.getMessage());
			}
		}
	}

	private String getTagFileName()
	{
		String filesep  = System.getProperty("file.separator");
		String userdir  = System.getProperty("user.dir");
		String appdir   = System.getProperty("majix.dir", userdir);
		String filename = System.getProperty("default.tdef", "default.tdef");

		return appdir + filesep + filename;
	}

	void SaveAsDefault()
	{
		String filename = getTagFileName();
		SaveToFile(filename);
	}

	void SaveToFile()
	{
		String filename = getTagParamFileName(true);
		if (filename != null) {
			SaveToFile(filename);
		}
	}

	void SaveToFile(String filename)
	{
		try {
			UpdateParam();
			_param.Save(filename);
		}
		catch (IOException e) {
			System.out.println("IO Exception when saving style sheet to " + filename);
		}
	}

	void reset()
	{
		_param = (XmlGeneratorParam) _template.getGeneratorParam().clone();
		LoadList();
	}

	void Apply()
	{
		UpdateParam();
		_template.setGeneratorParam(_param);

	}

	//=====================================
	// AUXILIARY FUNCTIONS

	void EnableButtons(boolean enabled)
	{
		_saveButton.setEnabled(enabled);
		_loadButton.setEnabled(enabled);
		_resetButton.setEnabled(enabled);
		_cancelButton.setEnabled(enabled);
		_closeButton.setEnabled(enabled);
	}

	String getTagParamFileName(boolean save)
	{
		FileDialog fd = new FileDialog(this, _res.getString("select_tdef_label"), 
										save ? FileDialog.SAVE : FileDialog.LOAD);

		FilenameFilter ff = new ExtensionFilenameFilter(".tdef");
		fd.setFilenameFilter(ff);
		fd.setFile("*.tdef");
		Point loc = getLocationOnScreen();
		loc.translate(10, 10);
		fd.setLocation(loc);
		fd.show();
		if (fd.getFile() != null) {

			String file = fd.getFile();

			if (file.endsWith(".*.*")) {
				file = file.substring(0, file.length() - 4);
			}

			return fd.getDirectory() + file;
		}
		else {
			return null;
		}
	}

	private String translate(String name)
	{
		String ret;

		try {
			ret = _res.getString(name);
		}
		catch (MissingResourceException e) {
			ret = name;
		}

		return ret;
	}

	void LoadList()
	{
		_list.removeAll();

		_list.addItem(new XmlTagsEditor.SeparatorListElement2(_res.getString("dtd_spec_separator")));
		
		_list.addItem(new DtdnameListElement(_param.getDtdName()));
		_list.addItem(new DtdsystemidListElement(_param.getDtdSystemId()));
		_list.addItem(new DtdpublicidListElement(_param.getDtdPublicId()));


		LoadSublist(XmlTagMap.TC_INFO,     "info_element_separator");
		LoadSublist(XmlTagMap.TC_SECTION,  "section_element_separator");
		LoadSublist(XmlTagMap.TC_BODY,     "body_element_separator");
		LoadSublist(XmlTagMap.TC_TABLE,    "table_element_separator");
		LoadSublist(XmlTagMap.TC_INLINE,   "inline_element_separator");
		LoadSublist(XmlTagMap.TC_COLOR,    "color_element_separator");
		LoadSublist(XmlTagMap.TC_CHARPROP, "charprop_element_separator");
		LoadSublist(XmlTagMap.TC_ENTITIES, "entities_separator");
		LoadSublist(XmlTagMap.TC_SPECIAL,  "special_element_separator");
	}

	private void LoadSublist(int category, String title)
	{
		XmlTagMap map = _param.getTags();

		_list.addItem(new XmlTagsEditor.SeparatorListElement2(_res.getString(title)));

		for (Enumeration e = map.getTags(); e.hasMoreElements(); ) {
			String id = e.nextElement().toString();
			if (map.getCategory(id) == category) {
				String desc = map.getDescription(id);
				String tag = map.getActualTag(id, false);
				String attrs = map.getAttributes(id);
				String desc2 = map.getDescription2(id);
				String attr2 = map.getAttribute2(id);
				String desc3 = map.getDescription3(id);
				String attr3 = map.getAttribute3(id);
				switch(map.getType(id)) {
				case XmlTagMap.TC_TYPE_ATTR:
					_list.addItem(new AttributeListElement(id, desc, attrs));
					break;
				case XmlTagMap.TC_TYPE_BOOL:
					_list.addItem(new BooleanListElement(id, desc, attrs));
					break;
				case XmlTagMap.TC_TYPE_TAG:
					_list.addItem(new StandardTagListElement(id, desc, tag, attrs));
					break;
				case XmlTagMap.TC_TYPE_TAG2:
					_list.addItem(new ExtendedTagListElement(id, desc, tag, attrs, desc2, attr2));
					break;
				case XmlTagMap.TC_TYPE_TAG3:
					_list.addItem(new ExtendedTagListElement(id, desc, tag, attrs, desc2, attr2, desc3, attr3));
					break;
				case XmlTagMap.TC_TYPE_ENT:
					_list.addItem(new EntityListElement(id, desc, tag));
					break;
				case XmlTagMap.TC_TYPE_SPECIAL:
					_list.addItem(new SpecialListElement(id, desc, tag));
					break;
				default:
					System.out.println("unknown tag list element type");
				}
			}
		}
	}

	void UpdateParam()
	{
		for (Enumeration e = _list.getElements(); e.hasMoreElements() ; ) {
			AbstractListElement2 elem = (AbstractListElement2) e.nextElement();
			elem.UpdateParam();
		}
	}

	void center()
	{
		Dimension wd = getSize();
		Dimension sd = Toolkit.getDefaultToolkit().getScreenSize();

		setLocation((sd.width - wd.width) / 2, (sd.height - wd.height) / 2);
	}

	private void showHelp()
	{
		ShowHelp.load(_HELP_TOPIC);
	}

	abstract class AbstractListElement2 
			implements JEditableListElement, ActionListener  {
		
        protected Box           _box;
		private JFlexibleList _owner;
		private boolean       _changed;

		AbstractListElement2()
		{
			_changed = false;
		}

		public void setOwner(Object list)
		{
			_owner = (JFlexibleList)list;
		}

		public void setChanged(boolean changed)
		{
			_changed = changed;
		}

		void UpdateParam()
		{
		}

		public boolean isChanged()
		{
			return _changed;
		}

		public void startEdit(Object context)
		{
		}

		public void validateEdit(Object context)
		{
		}

		public void cancelEdit(Object context)
		{
		}

		public void actionPerformed(ActionEvent e)
		{
			if (e.getSource() instanceof JButton) {
				String s = e.getActionCommand();
				if (s.equals("validate")) {
					_list.validateEditList();
				}
			}
		}

        Component toLeft(Component c)
        {
            Box box = Box.createHorizontalBox();
            box.add(c);
            box.add(Box.createHorizontalGlue());
            return box;
        }
	}

	private class StandardTagListElement 
			extends AbstractListElement2 {

		private JLabel _label1 = null;
		private JLabel _label2 = null;
		private JLabel _label3 = null;
		private JTextField _text1 = null;
		private JTextField _text2 = null;
		private JButton _button = null;

		private String _id;
		private String _desc;
		private String _tag;
		private String _attrs;

		StandardTagListElement(String id, String desc, String tag, String attrs)
		{
			_id = id;
			_desc = desc;
			_tag = tag;
			_attrs = attrs;
		}

		public String description()
		{
			StringBuffer ret = new StringBuffer();
			ret.append(translate(_desc));
			ret.append("  :  ");
			if (! _tag.equals("")) {
				ret.append("<");
				ret.append(_tag);
				if ((_attrs!= null) && (_attrs.length() > 0)) {
					ret.append(" ");
					ret.append(_attrs);
				}
				ret.append(">");
			}

			return ret.toString();
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_label1 = new JLabel(_res.getString("modif_tag_map_label")
							+ translate(_desc) + "                   ");

			_label2 = new JLabel(_res.getString("tag_label"));
			_text1 = new JTextField(16);
			_text1.setText(_tag);
			
			_label3 = new JLabel(_res.getString("attributes_label"));
			_text2 = new JTextField(16);
			_text2.setText(_attrs);

			_button = new JButton(_res.getString("validate_button"));
			_button.addActionListener(this);
			_button.setActionCommand("validate");

			_box.add(toLeft(_label2));
			_box.add(toLeft(_text1));
			_box.add(toLeft(_label3));
			_box.add(toLeft(_text2));
			_box.add(toLeft(_button));

			_elementEditionPanel.add(_box);

            pack();
			validate();
		}

		public void validateEdit(Object context)
		{
			_tag = _text1.getText();
			_attrs = _text2.getText();
			setChanged(true);
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

			_label1 = null;
			_label2 = null;
			_label3 = null;
			_text1 = null;
			_text2 = null;
			_button = null;

			validate();
		}

		void UpdateParam()
		{
			XmlTagMap map = _param.getTags();
			map.setActualTag(_id, _tag);
			map.setAttributes(_id, _attrs);
		}
	}


	private class ExtendedTagListElement 
			extends AbstractListElement2 {

		private JLabel _label1 = null;
		private JLabel _label2 = null;
		private JLabel _label3 = null;
		private JLabel _label3a = null;
		private JLabel _label4 = null;

		private JTextField _text1 = null;
		private JTextField _text2 = null;
		private JTextField _text2a = null;
		private JTextField _text3 = null;

		private JButton _button = null;

		private String _id;
		private String _desc;
		private String _tag;
		private String _attrs;
		private String _desc2;
		private String _attr2;
		private String _desc3;
		private String _attr3;

		ExtendedTagListElement(String id, String desc,
							   String tag, String attrs,
							   String desc2, String attr2)
		{
			_id = id;
			_desc = desc;
			_tag = tag;
			_attrs = attrs;
			_desc2 = desc2;
			_attr2 = attr2;
			_desc3 = null;
			_attr3 = null;
		}

		ExtendedTagListElement(String id, String desc,
							   String tag, String attrs,
							   String desc2, String attr2,
							   String desc3, String attr3)
		{
			_id = id;
			_desc = desc;
			_tag = tag;
			_attrs = attrs;
			_desc2 = desc2;
			_attr2 = attr2;
			_desc3 = desc3;
			_attr3 = attr3;
		}

		public String description()
		{
			StringBuffer ret = new StringBuffer();
			ret.append(translate(_desc));
			ret.append("  :  ");
			if (! _tag.equals("")) {
				ret.append("<");
				ret.append(_tag);
				if ((_attr2!= null) && (_attr2.length() > 0)) {
					ret.append(" ");
					ret.append(_attr2);
					ret.append("='...'");
				}
				if ((_attr3!= null) && (_attr3.length() > 0)) {
					ret.append(" ");
					ret.append(_attr3);
					ret.append("='...'");
				}
				if ((_attrs!= null) && (_attrs.length() > 0)) {
					ret.append(" ");
					ret.append(_attrs);
				}
				ret.append(">");
			}

			return ret.toString();
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_label1 = new JLabel(_res.getString("modif_tag_map_label")
					+ translate(_desc) + "                   ");

			_label2 = new JLabel(_res.getString("tag_label"));
			_text1 = new JTextField(16);
			_text1.setText(_tag);
			
			_label3 = new JLabel(translate(_desc2));
			_text2 = new JTextField(16);
			_text2.setText(_attr2);

			if (_desc3 != null) {
				_label3a = new JLabel(translate(_desc3));
				_text2a = new JTextField(16);
				_text2a.setText(_attr3);
			}

			_label4 = new JLabel(_res.getString("attributes_label"));
			_text3 = new JTextField(16);
			_text3.setText(_attrs);

			_button = new JButton(_res.getString("validate_button"));
			_button.addActionListener(this);
			_button.setActionCommand("validate");

			_box.add(toLeft(_label2));
			_box.add(toLeft(_text1));
			_box.add(toLeft(_label3));
			_box.add(toLeft(_text2));

			if (_label3a != null) {
				_box.add(toLeft(_label3a));
				_box.add(toLeft(_text2a));
			}

			_box.add(toLeft(_label4));
			_box.add(toLeft(_text3));
			_box.add(toLeft(_button));
			
			_elementEditionPanel.add(_box);

            pack();
			validate();
		}

		public void validateEdit(Object context)
		{
			_tag = _text1.getText();
			_attr2 = _text2.getText();
			_attr3 = _text2a.getText();
			_attrs = _text3.getText();
			setChanged(true);
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

            _box = null;
            
			_label1 = null;
			_label2 = null;
			_label3 = null;
			_label3a = null;
			_label4 = null;

			_text1 = null;
			_text2 = null;
			_text2a = null;
			_text3 = null;
			_button = null;

			validate();
		}

		void UpdateParam()
		{
			XmlTagMap map = _param.getTags();
			map.setActualTag(_id, _tag);
			map.setAttributes(_id, _attrs, _desc2, _attr2, _desc3, _attr3);
		}
	}


	class SeparatorListElement2 extends AbstractListElement2 {
		
		private String       _text;
		private FlexibleList _owner;
 
		SeparatorListElement2(String text)
		{
			_text  = text;
		}

		public String description()
		{
			return "---------------" + _text + "---------------";
		}

		String getStyleName()
		{
			return "";
		}

		String getAbstractStyle()
		{
			return "";
		}

		String getAbstractTag()
		{
			return "";
		}
	}

	abstract class SingleValueListElement 
			extends AbstractListElement2{

		private JLabel     _label = null;
		private JPanel     _panel = null;
		private JTextField _text = null;
		private JButton    _button = null;

		String _listLabel;
		String _textLabel;
		String _value;

		SingleValueListElement(String value)
		{
			_listLabel = "";
			_textLabel = "";
			_value = value;
		}

		SingleValueListElement(String listLabel, String textLabel, String value)
		{
			_listLabel = listLabel;
			_textLabel = textLabel;
			_value = value;
		}

		public String description()
		{
			return translate(_listLabel) + _value;
		}

		public void setListLabel(String label)
		{
			_listLabel = label;
		}

		public void setTextLabel(String label)
		{
			_textLabel = label;
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_label  = new JLabel(translate(_textLabel));
			_text   = new JTextField(40);
			_text.setText(_value);
			
			_button = new JButton(_res.getString("validate_button"));
			_button.addActionListener(this);
			_button.setActionCommand("validate");

			_box.add(toLeft(_label));
			_box.add(toLeft(_text));
			_box.add(toLeft(_button));

			_elementEditionPanel.add(_box);

            pack();
			validate();
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

			_panel = null;
			_label = null;
			_text = null;
			_button = null;

			validate();
		}

		public void validateEdit(Object context)
		{
			_value = _text.getText();
			setChanged(true);
		}
	}

	class AttributeListElement extends SingleValueListElement {

		String _id;

		AttributeListElement(String id, String desc, String attrname)
		{
			super(attrname);

			_id = id;
			setListLabel(desc);
			setTextLabel(desc);
		}

		public String description()
		{
			return translate(_listLabel) + " : " + 
				(((_value == null) || _value.equals("")) ? "" : _value + "=\"...\"");
		}

		public void UpdateParam()
		{
			XmlTagMap map = _param.getTags();
			map.setAttributes(_id, _value);
		}
	}

	class SpecialListElement extends SingleValueListElement {

		String _id;

		SpecialListElement(String id, String desc, String value)
		{
			super(value);

			_id = id;
			setListLabel(desc);
			setTextLabel(desc);
		}

		public String description()
		{
			return translate(_listLabel) + " : " + ((_value == null) ? "" : _value);
		}

		public void UpdateParam()
		{
			XmlTagMap map = _param.getTags();
			map.setActualTag(_id, _value);
		}
	}

	class EntityListElement extends SingleValueListElement {

		String _id;

		EntityListElement(String id, String desc, String entityname)
		{
			super(entityname);

			_id = id;
			setListLabel(desc);
			setTextLabel(desc);
		}

		public String description()
		{
			if (_value.equals("")) {
				return translate(_listLabel) + " : " + translate("not defined");
			}
			else {
				return translate(_listLabel) + " : &" + _value + ";";
			}
		}

		public void UpdateParam()
		{
			XmlTagMap map = _param.getTags();
			map.setActualTag(_id, _value.trim());
		}
	}

	class DtdnameListElement extends SingleValueListElement {

		DtdnameListElement(String dtdname)
		{
			super(dtdname);

			setListLabel(_res.getString("dtd_name_label"));
			setTextLabel(_res.getString("dtd_name_label"));
		}

		public String description()
		{
			return _listLabel + _value;
		}

		public void UpdateParam()
		{
			_param.setDtdName(_value);
		}
	}

	class DtdsystemidListElement extends SingleValueListElement {

		DtdsystemidListElement(String dtdsystemid)
		{
			super(dtdsystemid);

			setListLabel(_res.getString("dtd_sysid_label"));
			setTextLabel(_res.getString("dtd_sysid_label"));
		}

		public String description()
		{
			return _listLabel + _value;
		}

		public void UpdateParam()
		{
			_param.setDtdSystemId(_value);
		}
	}

	class DtdpublicidListElement extends SingleValueListElement {

		DtdpublicidListElement(String dtdpublicid)
		{
			super(dtdpublicid);

			setListLabel(_res.getString("dtd_pubid_label"));
			setTextLabel(_res.getString("dtd_pubid_label"));
		}

		public String description()
		{
			return _listLabel + _value;
		}

		public void UpdateParam()
		{
			_param.setDtdPublicId(_value);
		}
	}

	class BooleanListElement 
			extends AbstractListElement2{

		private JPanel     _panel = null;
		private JCheckBox  _checkBox = null;

		private String _listLabel;
		private String _textLabel;
		private String _value;

		private JButton _button;

		BooleanListElement(String value)
		{
			_listLabel = "";
			_textLabel = "";
			_value = value;
		}

		BooleanListElement(String listLabel, String textLabel, String value)
		{
			_listLabel = listLabel;
			_textLabel = textLabel;
			_value = value;
		}

		public String description()
		{
			return translate(_textLabel) + " : " + translate(_value);
		}

		public void setListLabel(String label)
		{
			_listLabel = label;
		}

		public void setTextLabel(String label)
		{
			_textLabel = label;
		}

		public void startEdit(Object context)
		{
            _box = Box.createVerticalBox();
            
			_checkBox = new JCheckBox(translate(_textLabel));
			if (_value.equals("true")) {
				_checkBox.setSelected(true);
			}

			_button = new JButton(_res.getString("validate_button"));
			_button.addActionListener(this);
			_button.setActionCommand("validate");

			_box.add(toLeft(_checkBox));
			_box.add(toLeft(_button));

			_elementEditionPanel.add(_box);

            pack();
			validate();
		}

		public void cancelEdit(Object context)
		{
			_elementEditionPanel.remove(_box);

			_box = null;
			_checkBox = null;
			_button = null;

			validate();
		}

		public void validateEdit(Object context)
		{
			boolean state = _checkBox.isSelected();
			_value = String.valueOf(state);
			
			setChanged(true);
		}

		public void UpdateParam()
		{
			XmlTagMap map = _param.getTags();
			map.setAttributes(_listLabel, _value);
		}
	}
}