//===============================================================
// package : com.tetrasix.majix.xml
// class : com.tetrasix.majix.XmlGenerator
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix.xml;

import java.io.File;
import java.util.Vector;

import com.tetrasix.majix.rtf.RtfBlock;
import com.tetrasix.majix.rtf.RtfCell;
import com.tetrasix.majix.rtf.RtfCellProperties;
import com.tetrasix.majix.rtf.RtfCompoundObject;
import com.tetrasix.majix.rtf.RtfDocument;
import com.tetrasix.majix.rtf.RtfFieldInstance;
import com.tetrasix.majix.rtf.RtfHyperLink;
import com.tetrasix.majix.rtf.RtfLineBreak;
import com.tetrasix.majix.rtf.RtfSectionBreak;
import com.tetrasix.majix.rtf.RtfPageBreak;
import com.tetrasix.majix.rtf.RtfParagraph;
import com.tetrasix.majix.rtf.RtfParagraphProperties;
import com.tetrasix.majix.rtf.RtfPicture;
import com.tetrasix.majix.rtf.RtfRow;
import com.tetrasix.majix.rtf.RtfRowProperties;
import com.tetrasix.majix.rtf.RtfStyleSheet;
import com.tetrasix.majix.rtf.RtfSymbol;
import com.tetrasix.majix.rtf.RtfTabDefList;
import com.tetrasix.majix.rtf.RtfTabDef;
import com.tetrasix.majix.rtf.RtfTab;
import com.tetrasix.majix.rtf.RtfText;
import com.tetrasix.majix.rtf.RtfTextProperties;
import com.tetrasix.util.Configuration;
import com.tetrasix.util.PictWriter;

public class XmlGenerator {
	public static final boolean OUTPUT_TEXTCOLOR = Configuration.getProperty("majix.generate.colorelements","1").equals("1");
	public static final boolean USE_CHAR_STYLES = Configuration.getProperty("majix.generate.stylesheetproperties","0").equals("1");
	
	private ConversionTemplate     _template = null;
	private RtfDocument            _doc;
	private XmlGeneratorState      _start;
	private XmlGeneratorState      _current;
	private XmlGeneratorStateStack _stack;
	private XmlGenerator           _top;
	private boolean                _again;
	private boolean                _debug;
	private String                 _directory;
	private String                 _filename;

	private String				   RE;

	private int                    _cell_count;
	private Vector                 _cell_vector;
	private boolean			       _skip_next_row;

	// for avoidance of redundant tag-generation
	private RtfTextProperties _lastProperties = null;
    private boolean _openTextPropertiesTags = false;
	
	public XmlGenerator(ConversionTemplate template)
	{
		RE = System.getProperty("line.separator");
		_template = template;
		reset();
	}

	public void reset()
	{
		_doc       = null;
		_stack     = new XmlGeneratorStateStack();
		_top       = null;
		_start     = _template.getGeneratorAutomaton().getStartState();
		_current   = _start;
		_again     = false;
		_debug     = false;
		_directory = ".";
		_filename  = null;
		_cell_count = 0;
		_cell_vector = new Vector();
		_skip_next_row = false;
	}
	
	public void setTemplate(ConversionTemplate template)
	{
		_template = template;
	}

	public ConversionTemplate getTemplate()
	{
		return _template;
	}

	public void setDirectory(String directory)
	{
		_directory = directory;
		if (_directory.equals("")) {
			_directory = ".";
		}
	}

	public void setFileName(String filename)
	{
		_filename = filename;
	}

	public void generate(RtfDocument theDoc, XmlWriter out, XmlGeneratorContext context)
	{
		_doc = theDoc;
		theDoc.generate(this, out, context);
	}

	public void rtfgenerate(RtfDocument theDoc, XmlWriter out, XmlGeneratorContext context)
	{
		_again = true;

		XmlTagTemplate.setDocument(theDoc);
		while (_again) {
//			performTransition("*DOC*", (RtfParagraph)null, out);
			performTransition("*DOC*", (RtfParagraph)null, out, new RootFunctor(theDoc));
		}

		rtfgenerate((RtfCompoundObject) theDoc, out, null);

		_again = true;

		while (_again) {
			performTransition("*EOF*", (RtfParagraph)null, out);
		}
	}

	public void rtfgenerate(RtfCompoundObject objects, XmlWriter out, XmlGeneratorContext context)
	{
		for (int ii = 0; ii < objects.size(); ii++) {
			if (objects.getObject(ii) == null) {
				System.out.println("bad !");
				continue;
			}
			objects.getObject(ii).generate(this, out, context);
		}
	}

	public void rtfgenerate(RtfParagraph theParagraph, XmlWriter out, XmlGeneratorContext context)
	{
		String tag = "p";
		
		// a hack to process the table of contents
		if (theParagraph.size() == 1) {
			Object obj = theParagraph.getObject(0);
			if (obj instanceof RtfFieldInstance) {
				RtfFieldInstance field = (RtfFieldInstance) obj;
				String fieldname = field.getFieldName();
				if (fieldname.equals("TOC")) {
					rtfgenerate(field, out, context);
					return;
				}
			}
		}


		RtfStyleSheet ssheet = _doc.getStyleSheet();
		RtfParagraphProperties properties = theParagraph.getProperties();
		int style = properties.getStyle();
		String stylename = ssheet.getStyleName(style);

		if (stylename != null) {
			String newtag = _template.getStyleMap().getAbstractStyleName(stylename, true);
			if (newtag != null) {
				tag = newtag;
			}
		}

		if (properties.getNumStyle() > 0 && tag.equals("p")) {
			String newtag = null;
			switch (properties.getNumStyle()) {
			case RtfParagraphProperties.STYLE_NUMBERED_DEC:
				newtag = _template.getStyleMap().getAbstractStyleName("*Numbered paragraph", true);
				break;
			case RtfParagraphProperties.STYLE_NUMBERED_ALPHA:
				newtag = _template.getStyleMap().getAbstractStyleName("*Numbered paragraph alphanumeric", true);
				break;
			case RtfParagraphProperties.STYLE_NUMBERED_ALPHA_UP:
				newtag = _template.getStyleMap().getAbstractStyleName("*Numbered paragraph alphanumeric uppercase", true);
				break;
			case RtfParagraphProperties.STYLE_NUMBERED_ROMAN:
				newtag = _template.getStyleMap().getAbstractStyleName("*Numbered paragraph roman", true);
				break;
			case RtfParagraphProperties.STYLE_NUMBERED_ROMAN_UP:
				newtag = _template.getStyleMap().getAbstractStyleName("*Numbered paragraph roman uppercase", true);
				break;
			case RtfParagraphProperties.STYLE_BULLET:
				newtag = _template.getStyleMap().getAbstractStyleName("*Bulleted paragraph", true);
				break;
			}
			if (newtag != null) {
				tag = newtag;
			}
		}

		_again = true;
		while (_again) {
			//System.out.println("FOO<" + tag + "> : " + stylename);
//			performTransition(tag, theParagraph, out);
			performTransition(tag, theParagraph, out, new ParagraphFunctor(theParagraph));
		}
	}

	public void rtfgenerate(RtfRow theRow, XmlWriter out, XmlGeneratorContext context)
	{
		String tag = "row";

		if (_skip_next_row) {
			_skip_next_row = false;
			return;
		}
		
		_cell_count = 0;
		_again = true;
		while (_again) {
//			performTransition(tag, theRow, out);
			performTransition(tag, theRow, out, new RowFunctor(theRow));
		}
	}

	void performTransition(String key, RtfBlock theBlock, XmlWriter out)
	{
		performTransition(key, theBlock, out, null);
	}

	void performTransition(String key, RtfBlock theBlock, XmlWriter out, XmlGeneratorFunctor functor)
	{
		if (key.equals("meta") && ! Configuration.getProperty("enable.meta").equals("")) {
			processMeta(theBlock, out);
			_again = false;
			return;
		}

		if (_debug) {
			out.print("<!--");
			if ((_current != null) && (_current.getName() != null)) {
				out.print(_current.getName());
			}
			out.print(":");
		}

		XmlGeneratorTransition transition = null;
		if (_current != null) {
			transition = _current.next(key);
			if (transition == null) {
				transition = _current.getDefaultTransition();
			}
		}

		if (transition == null) {
			transition = _template.getGeneratorAutomaton().getDefaultTransition();
		}
		
		// Debug-output
		if (_debug) {
			if (key != null) {
				out.print(key);
			}
			out.print(":");
			if (transition.getEndState() != null) {
				out.print(transition.getEndState().getName());
			}
			if (transition.getAgainFlag())
				out.print(" again");
			if (transition.getPushFlag())
				out.print(" push ->");
			if ((transition.getEndState() == null) && (_stack != null))
				out.print(" <- pop");
			out.print(" stack: ");
			out.println(_stack);
			XmlTagTemplate tag = transition.getGenerateBefore();
			if ((tag != null) && (! tag.equals(""))) {
				out.println(" -- before: "+tag.toString(functor));
			}
			tag = transition.getGenerateAfter();
			if ((tag != null) && (! tag.equals(""))) {
				out.print(" -- after: "+tag.toString(functor));
			}
			out.print("-->" + RE);
		}

		XmlTagTemplate tag = transition.getGenerateBefore();
		if ((tag != null) && (! tag.equals(""))) {
		    // set the bookmark-identifier
			if ( (theBlock instanceof RtfParagraph)
				&& (transition.getGenerateParagraphFlag())){
				RtfParagraph thePara = (RtfParagraph) theBlock;
				tag.setIdentifiers(thePara.getIdentifiers());
				thePara.resetIdentifiers();
			}
			// print the tag and its attributes
			out.print(tag.toString(functor));
		}

		if (   (theBlock != null)
			&& (transition.getGenerateParagraphFlag())) {

			boolean is_definition = false;

			//@@@ UGLY
			// find out if its an definition list
			XmlGeneratorContext context = null;
			if (theBlock instanceof RtfParagraph) {
				RtfParagraph theParagraph = (RtfParagraph) theBlock;
				context = new XmlGeneratorContext(theParagraph.getTextProperties());

				RtfParagraphProperties properties = theParagraph.getProperties();
				int stylecode = properties.getStyle();
				if (stylecode >= 0) {
					RtfStyleSheet ssheet = _doc.getStyleSheet();
					String ename = ssheet.getStyleName(stylecode);
					String iname = _template.getStyleMap().getAbstractStyleName(ename, true);

					if ((iname != null) && (iname.equals("dl1"))) {
						is_definition = true;
					}
				}
			}
			else {
				context = new XmlGeneratorContext(null);
			}
			
			// handle the definition list
			if (_template.getGeneratorParam().isDefined("dt") &&
					_template.getGeneratorParam().isDefined("dd") &&
					is_definition) {
				out.print("<" + _template.getGeneratorParam().getActualTag("dt", true) + ">");
				boolean in_term = true;
				for (int ii = 0; ii < theBlock.size(); ii++) {
					if ((theBlock.getObject(ii) instanceof RtfTab) && in_term) {
						out.print("</" + _template.getGeneratorParam().getActualTag("dt", false) + ">" + RE);
						out.print("<" + _template.getGeneratorParam().getActualTag("dd", true) + ">" + RE);
						in_term = false;
					}
					else {
						theBlock.getObject(ii).generate(this, out, context);
					}
				}
				if (in_term) {
					out.print("</" + _template.getGeneratorParam().getActualTag("dt", false) + ">" + RE);
					out.print("<" + _template.getGeneratorParam().getActualTag("dd", true) + ">" + RE);
				}
				out.print("</" + _template.getGeneratorParam().getActualTag("dd", false) + ">" + RE);
			}
			else {
			    // split the paragraph into tabdeflist and paragraph content
			    boolean generateParContent = false;
			    // generate the tabdeflist if present (hardcoded)
			    // TODO integrate tabdeflist-generation into DFA
			    // first check for existance of the elements and attributes
		        boolean tdl_isdefined = _template.getGeneratorParam().isDefined("tdl");
		        boolean td_isdefined = _template.getGeneratorParam().isDefined("td");
		        boolean pac_isdefined = _template.getGeneratorParam().isDefined("pac");
		        String tdl = _template.getGeneratorParam().getActualTag("tdl", false);
			    String td  = _template.getGeneratorParam().getActualTag("td", false);
			    String pac = _template.getGeneratorParam().getActualTag("pac", false);
			    String attr_tabtype = _template.getGeneratorParam().getAttributes("tabtype"); 
			    String attr_tabalign = _template.getGeneratorParam().getAttributes("tabalign"); 
			    String attr_tabpos = _template.getGeneratorParam().getAttributes("tabpos"); 
			    if (theBlock instanceof RtfParagraph) {
			        // only generate if the elements are defined
			    	// and if its a RtfParagraph
			        generateParContent = (tdl_isdefined && td_isdefined && pac_isdefined);

			        StringBuffer tabstring = new StringBuffer();
			        RtfParagraph para = (RtfParagraph) theBlock;
			        // standard value in mm for RtfTabDef.TAB_DEFAULT_WIDTH
			        double tabpos_default_mm = convertTwipsToMM(RtfTabDef.TAB_DEFAULT_WIDTH, 2);
			        
			        if (generateParContent) {
	                // <tabdeflist>
					out.print(RE + "<" + tdl + ">" + RE);
			        if (para.getProperties().getTabDefList() != null) {
			            RtfTabDefList tabdeflist = para.getProperties().getTabDefList();
			            RtfTabDef defaultTab = tabdeflist.getDefaultTabDef();
				        double tabpos_default_doc_mm = convertTwipsToMM(defaultTab.getTabPosition(), 2);
			            if (defaultTab != null) {
				            // generate the default tab entry
							// <tabdef>
		                    tabstring.append("<" + td);
							// the tab attributes
							// type="default"
								tabstring.append(" " + attr_tabtype + "=\"default\"");
							// align
								tabstring.append(" " + attr_tabalign + "=\"left\"");
							// default position
								tabstring.append(" " + attr_tabpos + "=\"" + tabpos_default_doc_mm + "mm\"");
							// close the tag
		                    tabstring.append("/>" + RE);
			            } else {
				            // generate the default tab entry (RtfTabDef.TAB_DEFAULT_WIDTH)
							// <tabdef>
		                    tabstring.append("<" + td);
							// the tab attributes
							// type="normal"
								tabstring.append(" " + attr_tabtype + "=\"default\"");
							// align
								tabstring.append(" " + attr_tabalign + "=\"left\"");
							// default position
								tabstring.append(" " + attr_tabpos + "=\"" + tabpos_default_mm + "mm\"");
							// close the tag
		                    tabstring.append("/>" + RE);
			            }
	                    double tabpos_mm = 0f;
						// there are tabs defined, so generate them.
		                for (int tabindex = 0; tabindex < tabdeflist.size(); tabindex++) {
		                    RtfTabDef tabdef = (RtfTabDef) tabdeflist.getObject(tabindex);
		                    tabpos_mm = this.convertTwipsToMM(tabdef.getTabPosition(), 2);
							// <tabdef>
		                    tabstring.append("<" + td);
							// the tab attributes
							// type="normal"
								tabstring.append(" " + attr_tabtype + "=\"normal\"");
							// align
								tabstring.append(" " + attr_tabalign + "=\"" + tabdef.getTabAlign() + "\"");
							// position
								tabstring.append(" " + attr_tabpos + "=\"" + tabpos_mm + "mm\"");
							// close the tag
		                    tabstring.append("/>" + RE);
		                }
			        } else {
			            // generate the default tab entry (RtfTabDef.TAB_DEFAULT_WIDTH)
						// <tabdef>
	                    tabstring.append("<" + td);
						// the tab attributes
						// type="normal"
							tabstring.append(" " + attr_tabtype + "=\"default\"");
						// align
							tabstring.append(" " + attr_tabalign + "=\"left\"");
						// default position
							tabstring.append(" " + attr_tabpos + "=\"" + tabpos_default_mm + "mm\"");
						// close the tag
	                    tabstring.append("/>" + RE);
			        }
			        
	                // </tabdeflist>
			        tabstring.append("</" + tdl + ">" + RE);
			        // send the tablist to the writer
			        out.print(tabstring.toString());
			        // generate the parcontent-element if its a paragraph
					out.print("<" + pac + ">" + RE);
			    }
			    }
			    
			    // generate the block-content of normal paragraphs paragraph-generating
			    // blocks
				for (int ii = 0; ii < theBlock.size(); ii++) {
					theBlock.getObject(ii).generate(this, out, context);
				}

				// if the inline elements are not closed, then do so.
			    if (_openTextPropertiesTags && _lastProperties != null) {
			        generateInlineElements(this._lastProperties, out, context, false);
			    }
			    // reset for next paragraph
			    _lastProperties = null;
			    
			    // close the parcontent-element if its a paragraph
			    if (generateParContent) {
					out.print(RE + "</" + pac + ">" + RE);
			    }
			}
		}

		tag = transition.getGenerateAfter();
		if ((tag != null) && ! tag.equals("")) {
			out.print(tag.toString(functor));
		}

		_again = transition.getAgainFlag();

		if (transition.getPushFlag()) {
			_stack.push(_current);
			_current = null;
		}

		_current = transition.getEndState();
		if ((_current == null) && (_stack != null)) {
			_current = _stack.pop();
		}
	}

	void processMeta(RtfBlock theBlock, XmlWriter out)
	{
		String data = theBlock.getData();
		if ((data != null) && data.length() > 0) {
			//System.out.println("data is <" + data + ">");
			if (data.charAt(0) == '@') {
				XmlGeneratorParam param = _template.getGeneratorParam();
				XmlTagMap map = param.getTags();
				String id = data.substring(1, 3);
				String tag = data.substring(4);
				map.setActualTag(id, tag);
				//map.setAttributes(_id, _attrs);
			}
			if (data.charAt(0) == '#') {
				XmlGeneratorParam param = _template.getGeneratorParam();
				XmlTagMap map = param.getTags();
				String id = data.substring(1, 3);
				String attrs = data.substring(4);
				map.setAttributes(id, attrs);
			}
			if (data.charAt(0) == '-') {
				_cell_vector = new Vector();
			}
			if (data.charAt(0) == '$') {
				_cell_vector.addElement(data.substring(1));
			}
			if (data.charAt(0) == '*') {
				_skip_next_row = true;
			}
			if (data.charAt(0) == '!') {
				out.print(data.substring(1));
			}
		}
	}

	/**
	 * This method encapsulates the generation of the inline-elements and is called
	 * by other rtfgenerate-methods that generate the actual content.
	 * 
	 * @param textProperties the properties of the content to generate
	 * @param out the outputwriter for the xml-file
	 * @param context the properties of the paragraph
	 * @param startElements true for the starting elements, false for the ending elements
	 *        (you have to generate the content inbetween).
	 */
	private void generateInlineElements(RtfTextProperties textProperties,
	        XmlWriter out,
	        XmlGeneratorContext context,
	        boolean startElements) {
	    
		RtfTextProperties contextProperties = context.getTextProperties();
		
		boolean isOtherFontSize = textProperties.changedFontSize();
		boolean isBold = textProperties.isBold();
		boolean isItalic = textProperties.isItalic();
		boolean isHidden = textProperties.isHidden();
		boolean isUnderlined = textProperties.isUnderlined();
		boolean isDoubleUnderlined = textProperties.isDoubleUnderlined();
		boolean isSuper = textProperties.isSuper();
		boolean isSub = textProperties.isSub();
		boolean isDeleted = textProperties.isDeleted();
		boolean isExpanded = textProperties.getExpanded() != 0;
		int color = textProperties.getColor();
		int style = textProperties.getStyle();
		String ename = "";
		String iname = "";

		String ctagname = null;

	    if (startElements) {
	        // if USE_CHAR_STYLES is true, then the styletags of the templatedefiniton
	        // will be used, and only additional character properties will be written
	        // if USE_CHAR_STYLES is false, then the styletags will not be used and thus all character-
	        // properties will be written, including those of the style.
			if (USE_CHAR_STYLES && (style != -1) 
					&& (contextProperties.getStyle() != style)) {
	
				RtfStyleSheet ssheet = _doc.getStyleSheet();
				RtfTextProperties styleProps = ssheet.getTextProperties(style);
	
				ename = ssheet.getStyleName(style);
				iname = _template.getStyleMap().getAbstractStyleName(ename, false);
	
				if (iname != null) {
					ctagname = _template.getGeneratorParam().getActualTag(iname, true);
					if (ctagname != null &&
						_template.getGeneratorParam().isDefined(iname) &&
						!ctagname.equals("cs") &&
						!ctagname.equals("")) {
						out.print("<" + ctagname + ">");
					}
				}
	
				if (styleProps.changedFontSize()) {
					isOtherFontSize = false;
				}
				if (styleProps.isBold()) {
					isBold = false;
				}
				if (styleProps.isItalic()) {
					isItalic = false;
				}
				if (styleProps.isHidden()) {
					isHidden = false;
				}
				if (styleProps.isUnderlined()) {
					isUnderlined = false;
				}
				if (styleProps.isDoubleUnderlined()) {
					isDoubleUnderlined = false;
				}
				if (styleProps.isSuper()) {
					isSuper = false;
				}
				if (styleProps.isSub()) {
					isSub = false;
				}
				if (styleProps.isDeleted()) {
					isDeleted = false;
				}
				if (styleProps.getExpanded() != 0) {
					isExpanded = false;
				}
				if (styleProps.getColor() == color) {
					color = -1;
				}
			}
	
			if (_template.getGeneratorParam().isDefined("f") &&
				isOtherFontSize &&
				!contextProperties.changedFontSize()) {
			    // font element and fontsize attribute
			    String actualtag = _template.getGeneratorParam().getActualTag("f", true);
			    String attribute = new FontFunctor(textProperties).doit("f");
			    String element = "<" + actualtag + attribute + ">";
		    	out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("bt") &&
				isBold &&
				!contextProperties.isBold()) {
				String element = _template.getGeneratorParam().getStartTag("bt");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("it") &&
				isItalic &&
				! contextProperties.isItalic()) {
				String element = _template.getGeneratorParam().getStartTag("it");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("vt") &&
				isHidden &&
				! contextProperties.isHidden()) {
				String element = _template.getGeneratorParam().getStartTag("vt");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("ut") &&
				isUnderlined &&
				!contextProperties.isUnderlined()) {
				String element = _template.getGeneratorParam().getStartTag("ut");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("utd") &&
				isDoubleUnderlined &&
				! contextProperties.isDoubleUnderlined()) {
				String element = _template.getGeneratorParam().getStartTag("utd");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("super") &&
				isSuper &&
				! contextProperties.isSuper()) {
				String element = _template.getGeneratorParam().getStartTag("super");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("sub") &&
				isSub &&
				! contextProperties.isSub()) {
				String element = _template.getGeneratorParam().getStartTag("sub");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("del") &&
				isDeleted &&
				! contextProperties.isDeleted()) {
				String element = _template.getGeneratorParam().getStartTag("del");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("exp") &&
				isExpanded &&
				! (contextProperties.getExpanded() != 0)) {
			    // expanded attribute
			    String actualtag = _template.getGeneratorParam().getActualTag("exp", true);
			    String attribute = new CharacterFunctor(textProperties).doit("exp");
			    String element = "<" + actualtag + attribute + ">";
		    	out.print(element);
			}
	
			if (OUTPUT_TEXTCOLOR &&
					_template.getGeneratorParam().isDefined(textProperties.getColorCode()) &&
					(color != -1) 
					&& (contextProperties.getColor() != textProperties.getColor())) {
				if (textProperties.getColorCode() != null)
					out.print(_template.getGeneratorParam().getStartTag(textProperties.getColorCode()));
			}
	    } else { // startElements == false -> endElements
			if (OUTPUT_TEXTCOLOR &&
					_template.getGeneratorParam().isDefined(textProperties.getColorCode()) &&
					(color != -1) 
					&& (contextProperties.getColor() != textProperties.getColor())) {
				if (textProperties.getColorCode() != null)
					out.print(_template.getGeneratorParam().getEndTag(textProperties.getColorCode()));
			}

			if (_template.getGeneratorParam().isDefined("exp") &&
				isExpanded && ! (contextProperties.getExpanded() != 0)) {
				String element = _template.getGeneratorParam().getEndTag("exp");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("del") &&
				isDeleted &&
				! contextProperties.isDeleted()) {
				String element = _template.getGeneratorParam().getEndTag("del");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("sub") &&
				isSub &&
				! contextProperties.isSub()) {
				String element = _template.getGeneratorParam().getEndTag("sub");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("super") &&
				isSuper &&
				! contextProperties.isSuper()) {
				String element = _template.getGeneratorParam().getEndTag("super");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("utd") &&
				isDoubleUnderlined &&
				! contextProperties.isDoubleUnderlined()) {
				String element = _template.getGeneratorParam().getEndTag("utd");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("ut") &&
				isUnderlined &&
				! contextProperties.isUnderlined()) {
				String element = _template.getGeneratorParam().getEndTag("ut");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("vt") &&
				isHidden && ! contextProperties.isHidden()) {
				String element = _template.getGeneratorParam().getEndTag("vt");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("it") &&
				isItalic &&
				! contextProperties.isItalic()) {
				String element = _template.getGeneratorParam().getEndTag("it");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("bt") &&
				isBold &&
				! contextProperties.isBold()) {
				String element = _template.getGeneratorParam().getEndTag("bt");
				out.print(element);
			}
			if (_template.getGeneratorParam().isDefined("f") &&
				isOtherFontSize && !contextProperties.changedFontSize()) {
//			if (true) { // always output font size
				String element = _template.getGeneratorParam().getEndTag("f");
				out.print(element);
			}

			if (USE_CHAR_STYLES && ctagname != null) {
				if ((style != -1) 
						&& (contextProperties.getStyle() != style)) {
					ctagname = _template.getGeneratorParam().getActualTag(iname, false);
					if (ctagname != null &&
						_template.getGeneratorParam().isDefined(iname) &&
						!ctagname.equals("cs") && !ctagname.equals("")) {
						out.print("</" + ctagname + ">");
					}
				}
			}
	    }
	}

	public void rtfgenerate(RtfText theText, XmlWriter out, XmlGeneratorContext context)
	{
	    boolean nothingChanged = theText.getProperties().equals(_lastProperties);

	    // TODO gr�ndlich testen!!!!!
	    // generate end elements for the previous textproperties, if they exist
	    if (!nothingChanged && _lastProperties != null) {
	        generateInlineElements(_lastProperties, out, context, false);
	    }

	    // generate start elements for the textproperties
	    if (!nothingChanged) {
	        generateInlineElements(theText.getProperties(), out, context, true);
	        _lastProperties = theText.getProperties();
	        _openTextPropertiesTags = true;
	    }
	    
		out.printQuote(theText.getData());
	}

	public void rtfgenerate(RtfPicture thePicture, XmlWriter out, XmlGeneratorContext context)
	{
	    // close RtfTextProperties-Tags
	    if (this._openTextPropertiesTags && this._lastProperties != null) {
	        this.generateInlineElements(_lastProperties, out, context, false);
	        this._lastProperties = null;
	        this._openTextPropertiesTags = false;
	    }

		XmlGeneratorParam param = _template.getGeneratorParam();

		String imagesdir = param.getActualTag("gd", false);
		if (! imagesdir.equals("")) {
			File file = new File(_directory + File.separator + imagesdir);
			file.mkdir();
			imagesdir += File.separator;
		}

		if (_template.getGeneratorParam().isDefined("gr"))
		    out.print("<" + param.getActualTag("gr", false));
		String attribute2 = param.getAttribute2("gr");
		String attribute3 = param.getAttribute3("gr");
		if (! attribute2.equals("")) {
			out.print(" " );
			out.print(attribute2 + "='");
			if (thePicture.getCounter() == -1) {
				out.print(thePicture.getData());
			}
			else {  
				String path = imagesdir + _filename + "_";
				String counter = String.valueOf(thePicture.getCounter());
				if (counter.length() < 3) {
					counter = "000" + counter;
					counter = counter.substring(counter.length() - 3, counter.length());
				}
				path += counter;
				String extension = param.getActualTag("pe", true);
				if (extension.equals("")) {
					extension = "wmf";
				}
				path += "." + extension;
				if (thePicture.getPictdata() != null) {
					String graphicfilename = _directory + File.separator + path;
					PictWriter.dumpWMF(
						thePicture.getPictdata(),
						graphicfilename,
						thePicture.getPicw(),
						thePicture.getPich(),
						thePicture.getPicwg(),
						thePicture.getPichg()
						);
				}
                path = path.replace('\\', '/');
                out.print(path);
			}
			out.print("'");
		}
		if (! attribute3.equals("")) {
			String counter = String.valueOf(thePicture.getCounter());
			if (counter.length() < 3) {
				counter = "000" + counter;
				counter = counter.substring(counter.length() - 3, counter.length());
			}
			out.print(" " + attribute3 + "='" + param.getActualTag("gp", false)
						+ counter + "'");
		}
		String attributes = param.getAttributes("gr");
		if (attributes != null && !attributes.equals("")) {
			out.print(" ");
			out.print(attributes);
		}
		out.print("/>");
	}

	public void rtfgenerate(RtfSymbol theSymbol, XmlWriter out, XmlGeneratorContext context)
	{
		if (theSymbol.getFont().equals("Symbol")) {
			String symbolname = getSymbolName(theSymbol.getCode());
			if (symbolname.equals("")) {
				out.print("?");
			}
			else {
				out.print("&" + getSymbolName(theSymbol.getCode()) + ";");
			}
		}
	}


	public void rtfgenerate(RtfCell theCell, XmlWriter out, XmlGeneratorContext context)
	{
		String attr_rowspan = _template.getGeneratorParam().getAttributes("rowspan");
		
	    // only generate this cell if its not vertically merged and thus visible
	    if (attr_rowspan != null &&
	       !attr_rowspan.equals("") &&
	        theCell.getProperties().getVerticalMerged())
	        return;
	    
	    RtfCellProperties props = theCell.getProperties();
		_stack.push(_current);
		_current = _template.getGeneratorAutomaton().getStartCellState();

		// Debug-output
		if (_debug) {
			out.print("<!--");
			if ((_current != null) && (_current.getName() != null)) {
				out.print(_current.getName());
			}

			out.print(" ,stack: ");
			out.print(_stack);
			out.print("-->" + RE);
		}

		if (_cell_vector.size() > _cell_count) {
			XmlGeneratorParam param = _template.getGeneratorParam();
			XmlTagMap map = param.getTags();
			map.setActualTag("ce", _cell_vector.elementAt(_cell_count).toString());
		}
		
		_again = true;
		while (_again) {
			performTransition("cell", (RtfParagraph)null, out, new CellFunctor(theCell));
		}

		rtfgenerate((RtfCompoundObject) theCell, out, null);

		_again = true;
		while (_again) {
			performTransition("*EOC*", (RtfParagraph)null, out);
		}

		_cell_count = _cell_count + 1;
	}

	void performTransition(String key, RtfCell theCell, XmlWriter out)
	{
		if (_debug) {
			out.print("<!--");
			if ((_current != null) && (_current.getName() != null)) {
				out.print(_current.getName());
			}
			out.print(":");
		}

		XmlGeneratorTransition transition = null;
		if (_current != null) {
			transition = _current.next(key);
			if (transition == null) {
				transition = _current.getDefaultTransition();
			}
		}

		if (transition == null) {
			transition = _template.getGeneratorAutomaton().getDefaultTransition();
		}
		
		if (_debug) {
			if (key != null) {
				out.print(key);
			}
			out.print(":");
			if (transition.getEndState() != null) {
				out.print(transition.getEndState().getName());
			}
			out.print(" ");
			out.print(_stack);
			out.print("-->");
		}

		XmlTagTemplate tag = transition.getGenerateBefore();
		if (tag != null) {
			out.print(tag.toString());
		}

		if (   (theCell != null)
			&& (transition.getGenerateParagraphFlag())) {

			XmlGeneratorContext context = new XmlGeneratorContext(null);

			for (int ii = 0; ii < theCell.size(); ii++) {
				theCell.getObject(ii).generate(this, out, context);
			}
		}

		tag = transition.getGenerateAfter();
		if (tag != null) {
			out.print(tag);
		}

		_again = transition.getAgainFlag();

		if (transition.getPushFlag()) {
			_stack.push(_current);
			_current = null;
		}

		_current = transition.getEndState();
		if ((_current == null) && (_stack != null)) {
			_current = _stack.pop();
		}
	}

	public void setDebugFlag(boolean flag)
	{
		_debug = flag;
	}

	static String _greek_symbol_names[] = null;

	static String getSymbolName(int code)
	{
		if (_greek_symbol_names == null) {
			_greek_symbol_names = new String[255];

			_greek_symbol_names[ 65] = "Agr";
			_greek_symbol_names[ 97] = "agr";
			_greek_symbol_names[ 66] = "Bgr";
			_greek_symbol_names[ 98] = "bgr";
			_greek_symbol_names[ 71] = "Ggr";
			_greek_symbol_names[103] = "ggr";
			_greek_symbol_names[ 68] = "Dgr";
			_greek_symbol_names[100] = "dgr";
			_greek_symbol_names[ 69] = "Egr";
			_greek_symbol_names[101] = "egr";
			_greek_symbol_names[ 90] = "Zgr";
			_greek_symbol_names[122] = "zgr";
			_greek_symbol_names[ 72] = "EEgr";
			_greek_symbol_names[104] = "eegr";
			_greek_symbol_names[ 81] = "THgr";
			_greek_symbol_names[113] = "thgr";
			_greek_symbol_names[ 73] = "Igr";
			_greek_symbol_names[105] = "igr";
			_greek_symbol_names[ 75] = "Kgr";
			_greek_symbol_names[107] = "kgr";
			_greek_symbol_names[ 76] = "Lgr";
			_greek_symbol_names[108] = "lgr";
			_greek_symbol_names[ 77] = "Mgr";
			_greek_symbol_names[109] = "mgr";
			_greek_symbol_names[ 78] = "Ngr";
			_greek_symbol_names[110] = "ngr";
			_greek_symbol_names[ 88] = "Xgr";
			_greek_symbol_names[120] = "xgr";
			_greek_symbol_names[ 79] = "Ogr";
			_greek_symbol_names[111] = "ogr";
			_greek_symbol_names[ 80] = "Pgr";
			_greek_symbol_names[112] = "pgr";
			_greek_symbol_names[ 82] = "Rgr";
			_greek_symbol_names[114] = "rgr";
			_greek_symbol_names[ 83] = "Sgr";
			_greek_symbol_names[115] = "sgr";
			_greek_symbol_names[ 84] = "Tgr";
			_greek_symbol_names[116] = "tgr";
			_greek_symbol_names[ 85] = "Ugr";
			_greek_symbol_names[117] = "ugr";
			_greek_symbol_names[ 70] = "PHgr";
			_greek_symbol_names[102] = "phgr";
			_greek_symbol_names[ 67] = "KHgr";
			_greek_symbol_names[99] = "khgr";
			_greek_symbol_names[ 89] = "PSgr";
			_greek_symbol_names[121] = "psgr";
			_greek_symbol_names[ 87] = "OHgr";
			_greek_symbol_names[119] = "ohgr";
			_greek_symbol_names[118] = "ohgr";
		}
		String name = _greek_symbol_names[code];
		if (name != null) {
			return name;
		}
		else {
			return "";
		}
	}

	public void rtfgenerate(RtfFieldInstance theFieldInstance, XmlWriter out, XmlGeneratorContext context)
	{
	    // close RtfTextProperties-Tags
	    if (this._openTextPropertiesTags && this._lastProperties != null) {
	        this.generateInlineElements(_lastProperties, out, context, false);
	        this._lastProperties = null;
	        this._openTextPropertiesTags = false;
	    }

		String fieldname = theFieldInstance.getFieldName();

		if (_template.getGeneratorParam().isDefined("tc") && fieldname.equals("TOC")) {
			out.print("<" + _template.getGeneratorParam().getActualTag("tc", true) + "/>" + RE);
		}
	}

	public void rtfgenerate(RtfHyperLink theLink, XmlWriter out, XmlGeneratorContext context)
	{
	    if (_template.getGeneratorParam().isDefined("hl"))
	        out.print("<" + _template.getGeneratorParam().getActualTag("hl", false));
		String attributes = _template.getGeneratorParam().getAttributes("hl");
		if ((attributes != null) && (! attributes.equals(""))) {
			out.print(" " );
			out.print(attributes);
		}
		String attribute2 = _template.getGeneratorParam().getAttribute2("hl");
		String attribute3 = _template.getGeneratorParam().getAttribute3("hl");
		String refid = theLink.getRefid();
		String url;
		if ((attribute3 == null) || (attribute3.equals(""))) {
			if (! refid.equals("")) {
				refid = '#' + refid;
			}
			url = theLink.getUrl() + refid;
		}
		else {
			url = theLink.getUrl();
		}
		if ((attribute2 != null) && (! attribute2.equals("")) && (! url.equals(""))) {
			out.print(" " );
			out.print(attribute2);
			out.print("='" );
			out.printQuote(url);
			out.print("'" );
		}
		if ((attribute3 != null) && (! attribute3.equals("")) && (! refid.equals(""))) {
			out.print(" " );
			out.print(attribute3);
			out.print("='" );
			out.printQuote(theLink.getRefid());
			out.print("'" );
		}
		if (_template.getGeneratorParam().isDefined("hl")) {
			out.print(">");
			out.printQuote(theLink.getText());
			out.print("</" + _template.getGeneratorParam().getActualTag("hl", false) + ">");
		}
	}

	public void rtfgenerate(RtfTab theTab, XmlWriter out, XmlGeneratorContext context)
	{
		String element = _template.getGeneratorParam().getActualTag("tb", true);
	    int type = _template.getGeneratorParam().getTags().getType("tb");
		if (type == XmlTagMap.TC_TYPE_ENT && !element.equals("")) {
		    //redirect to RtfText.generate
		    new RtfText("&" + element + ";", theTab.getTextProperties()).generate(this, out, context);
		} else if (type == XmlTagMap.TC_TYPE_TAG && !element.equals("")) {
		    // tab attributes
		    String attribute = new TabFunctor(theTab.getTextProperties()).doit("tb");
		    String elementwithattr = element + attribute;
		    
		    // close RtfTextProperties-Tags
		    if (this._openTextPropertiesTags && this._lastProperties != null) {
		        this.generateInlineElements(_lastProperties, out, context, false);
		        this._lastProperties = null;
		        this._openTextPropertiesTags = false;
		    }

		    if (_template.getGeneratorParam().isDefined("tb"))
		        out.print("<"+elementwithattr+"/>");
		}
	}

	public void rtfgenerate(RtfPageBreak theBreak, XmlWriter out, XmlGeneratorContext context)
	{
	    // close RtfTextProperties-Tags
	    if (this._openTextPropertiesTags && this._lastProperties != null) {
	        this.generateInlineElements(_lastProperties, out, context, false);
	        this._lastProperties = null;
	        this._openTextPropertiesTags = false;
	    }

	    String element = _template.getGeneratorParam().getActualTag("pb", false);
	    int type = _template.getGeneratorParam().getTags().getType("pb");
		if (type == XmlTagMap.TC_TYPE_ENT && _template.getGeneratorParam().isDefined("pb") && !element.equals("")) {
			out.print("&" + element + ";");
		} else if (type == XmlTagMap.TC_TYPE_TAG && _template.getGeneratorParam().isDefined("pb") && !element.equals("")) {
		    out.print("<"+element+"/>");
		}
	}

	public void rtfgenerate(RtfSectionBreak theSectionBreak, XmlWriter out, XmlGeneratorContext context)
	{
	    // close RtfTextProperties-Tags
	    if (this._openTextPropertiesTags && this._lastProperties != null) {
	        this.generateInlineElements(_lastProperties, out, context, false);
	        this._lastProperties = null;
	        this._openTextPropertiesTags = false;
	    }

	    String element = _template.getGeneratorParam().getActualTag("sb", false);
	    int type = _template.getGeneratorParam().getTags().getType("sb");
		if (type == XmlTagMap.TC_TYPE_ENT && _template.getGeneratorParam().isDefined("sb") && !element.equals("")) {
			out.print("&" + element + ";");
		} else if (type == XmlTagMap.TC_TYPE_TAG && _template.getGeneratorParam().isDefined("sb") && !element.equals("")) {
		    out.print("<"+element+"/>");
		}
	}

	public void rtfgenerate(RtfLineBreak theLine, XmlWriter out, XmlGeneratorContext context)
	{
	    // close RtfTextProperties-Tags
	    if (this._openTextPropertiesTags && this._lastProperties != null) {
	        this.generateInlineElements(_lastProperties, out, context, false);
	        this._lastProperties = null;
	        this._openTextPropertiesTags = false;
	    }

	    String element = _template.getGeneratorParam().getActualTag("lb", false);
	    int type = _template.getGeneratorParam().getTags().getType("lb");
		if (type == XmlTagMap.TC_TYPE_ENT && _template.getGeneratorParam().isDefined("lb") && !element.equals("")) {
			out.print("&" + element + ";");
		} else if (type == XmlTagMap.TC_TYPE_TAG && _template.getGeneratorParam().isDefined("lb") && !element.equals("")) {
		    out.print("<"+element+"/>");
		}
	}

	/**
	 * This method converts twips (a twentieth of a point) to millimeters.
	 * The second parameter specifies the digit-count to round the result.
	 * 
	 * @param twips the twips-value to convert.
	 * @param roundDigits round to #roundDigits digits after the decimal-separator.
	 * @return the result in millimeters, rounded to #roundDigits digits.
	 */
	public double convertTwipsToMM(int twips, int roundDigits) {
        double result = (twips == 0 ? 0f : twips * 0.0176388888889f);
        // shortcut
        if (result == 0) return 0;
		// round the result
        double factor = Math.pow(10, roundDigits);
		result = Math.round( result * factor ) / factor;
		return result;
	}
	
	/**
	 * This method converts twips (a twentieth of a point) to points.
	 * The second parameter specifies the digit-count to round the result.
	 * 
	 * @param twips the twips-value to convert.
	 * @param roundDigits round to #roundDigits digits after the decimal-separator.
	 * @return the result in points, rounded to #roundDigits digits.
	 */
	public double convertTwipsToPoints(int twips, int roundDigits) {
        double result = (twips == 0 ? 0f : twips * 0.05f);
        // shortcut
        if (result == 0) return 0;
		// round the result
        double factor = Math.pow(10, roundDigits);
		result = Math.round( result * factor ) / factor;
		return result;
	}
	
	/**
	 * Encodes the characters in this string between 0x80 and
	 * 0xFF (both exclusive) to character entities (hex-encoded).
	 * Other characters will be in the string untouched.
	 * @param data the String to encode
	 * @return the encoded String
	 */
	public static String convertCharacter(String data) {
	    if (data == null) return null;
	    StringBuffer buffer = new StringBuffer();
	    for (int i=0;i<data.length(); i++) {
	        if (data.charAt(i) > 0x80 && data.charAt(i) < 0xFF)
	            buffer.append(XmlWriter.quote(data.charAt(i)));
	        else if (data.charAt(i) < 0xFF)
	            buffer.append(data.charAt(i));
	    }
	    
	    return buffer.toString();
	}
	
	class FontFunctor implements XmlGeneratorFunctor {

	    RtfTextProperties _theTextProperties;

	    FontFunctor(RtfTextProperties theTextProperties)
		{
	        _theTextProperties = theTextProperties;
		}

		public String doit(String tag)
		{
		    String result = "";
		    if (tag.equals("f")) {
				// fontsize of the Text
				int fontsize = _theTextProperties.getFontSize();
				int fontsize_pt = fontsize / 2; 
				String attr_fontsize = _template.getGeneratorParam()
						.getAttributes("fs");
				if (attr_fontsize != null && !attr_fontsize.equals("")) {
					result += " " + attr_fontsize + "=\"" + fontsize_pt + "pt\"";
				}
		    }
			
			return result;
		}
	}
	
	class CharacterFunctor implements XmlGeneratorFunctor {

	    RtfTextProperties _theTextProperties;

	    CharacterFunctor(RtfTextProperties theTextProperties)
		{
	        _theTextProperties = theTextProperties;
		}

		public String doit(String tag)
		{
		    String result = "";
		    if (tag.equals("exp")) {
				// expansion / compression of the Text
				int expansion = _theTextProperties.getExpanded();
				double expansion_pt = convertTwipsToPoints(expansion, 1);
				String attr_expansion = _template.getGeneratorParam()
						.getAttributes("expval");
				if (attr_expansion != null && !attr_expansion.equals("") && expansion_pt != 0) {
					result += " " + attr_expansion + "=\"" + expansion_pt + "pt\"";
				}
		    }
			
			return result;
		}
	}
	
	class TabFunctor implements XmlGeneratorFunctor {

	    RtfTextProperties _theTextProperties;

	    TabFunctor(RtfTextProperties theTextProperties)
		{
	        _theTextProperties = theTextProperties;
		}

		public String doit(String tag)
		{
		    String result = "";
			// underlined tab
			boolean underlined = _theTextProperties.isUnderlined();
			boolean doubleunderlined = _theTextProperties.isDoubleUnderlined();
			String attr_underlined = _template.getGeneratorParam()
					.getAttributes("tut");
			String attr_doubleunderlined = _template.getGeneratorParam()
					.getAttributes("tutd");
			if (attr_underlined != null && !attr_underlined.equals("")) {
			    // cant be underlined AND doubleunderlined
			    if (underlined)
			        result += " " + attr_underlined + "=\"true\"";
			    else if (doubleunderlined)
			        result += " " + attr_doubleunderlined + "=\"true\"";
			}
			
			// deleted tab
			boolean deleted = _theTextProperties.isDeleted();
			String attr_deleted = _template.getGeneratorParam()
					.getAttributes("tud");
			if (attr_deleted != null && !attr_deleted.equals("") && deleted) {
		        result += " " + attr_deleted + "=\"true\"";
			}
			
			// hidden tab
			boolean hidden = _theTextProperties.isHidden();
			String attr_hidden = _template.getGeneratorParam()
					.getAttributes("tvt");
			if (attr_hidden != null && !attr_hidden.equals("") && hidden) {
		        result += " " + attr_hidden + "=\"true\"";
			}
			
			return result;
		}
	}
	
	class RowFunctor implements XmlGeneratorFunctor {

	    RtfRow _theRow;

		RowFunctor(RtfRow theRow)
		{
			_theRow = theRow;
		}

		public String doit(String tag)
		{
			String result = "";
			if (tag.equals("ro")) {
				// row borders
				boolean hasNoBorders = _theRow.getProperties().hasNoBorder();
				if (!hasNoBorders) {
					boolean hasTopBorder = _theRow.getProperties().hasTopBorder();
					boolean hasBottomBorder = _theRow.getProperties().hasBottomBorder();
					boolean hasLeftBorder = _theRow.getProperties().hasLeftBorder();
					boolean hasRightBorder = _theRow.getProperties().hasRightBorder();
					boolean hasHorizontalBorder = _theRow.getProperties().hasHorizontalBorder();
					boolean hasVerticalBorder = _theRow.getProperties().hasVerticalBorder();
					String attr_cborders_top = _template.getGeneratorParam().getAttributes("bortop");
					String attr_cborders_bottom = _template.getGeneratorParam().getAttributes("borbot");
					String attr_cborders_left = _template.getGeneratorParam().getAttributes("borle");
					String attr_cborders_right = _template.getGeneratorParam().getAttributes("borri");
					String attr_cborders_horizontal = _template.getGeneratorParam().getAttributes("borhor");
					String attr_cborders_vertical = _template.getGeneratorParam().getAttributes("borvert");
					
					if (attr_cborders_top != null && !attr_cborders_top.equals("") && hasTopBorder) {
					    double topWidth = convertTwipsToPoints(_theRow.getProperties().getBorderWidth(RtfRowProperties.ROWBORDER_TOP),1);
						result += " " + attr_cborders_top + "=\""+topWidth+"pt\"";
					}
					if (attr_cborders_bottom != null && !attr_cborders_bottom.equals("") && hasBottomBorder) {
					    double bottomWidth = convertTwipsToPoints(_theRow.getProperties().getBorderWidth(RtfRowProperties.ROWBORDER_BOTTOM),1);
						result += " " + attr_cborders_bottom + "=\""+bottomWidth+"pt\"";
					}
					if (attr_cborders_left != null && !attr_cborders_left.equals("") && hasLeftBorder) {
					    double leftWidth = convertTwipsToPoints(_theRow.getProperties().getBorderWidth(RtfRowProperties.ROWBORDER_LEFT),1);
						result += " " + attr_cborders_left + "=\""+leftWidth+"pt\"";
					}
					if (attr_cborders_right != null && !attr_cborders_right.equals("") && hasRightBorder) {
					    double rightWidth = convertTwipsToPoints(_theRow.getProperties().getBorderWidth(RtfRowProperties.ROWBORDER_RIGHT),1);
						result += " " + attr_cborders_right + "=\""+rightWidth+"pt\"";
					}
					if (attr_cborders_horizontal != null && !attr_cborders_horizontal.equals("") && hasHorizontalBorder) {
					    double horizontalWidth = convertTwipsToPoints(_theRow.getProperties().getBorderWidth(RtfRowProperties.ROWBORDER_HORIZONTAL),1);
						result += " " + attr_cborders_horizontal + "=\""+horizontalWidth+"pt\"";
					}
					if (attr_cborders_vertical != null && !attr_cborders_vertical.equals("") && hasVerticalBorder) {
					    double verticalWidth = convertTwipsToPoints(_theRow.getProperties().getBorderWidth(RtfRowProperties.ROWBORDER_VERTICAL),1);
						result += " " + attr_cborders_vertical + "=\""+verticalWidth+"pt\"";
					}
				}
				
				// row paddings/margins
				boolean hasTopMargin = _theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_TOP) > 0;
				boolean hasBottomMargin = _theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_BOTTOM) > 0;
				boolean hasLeftMargin = _theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_LEFT) > 0;
				boolean hasRightMargin = _theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_RIGHT) > 0;
				String attr_rmargin_top = _template.getGeneratorParam().getAttributes("martop");
				String attr_rmargin_bottom = _template.getGeneratorParam().getAttributes("marbot");
				String attr_rmargin_left = _template.getGeneratorParam().getAttributes("marle");
				String attr_rmargin_right = _template.getGeneratorParam().getAttributes("marri");
				
				if (attr_rmargin_top != null && !attr_rmargin_top.equals("") && hasTopMargin) {
				    double topMargin = convertTwipsToPoints(_theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_TOP),1);
					result += " " + attr_rmargin_top + "=\""+topMargin+"pt\"";
				}
				if (attr_rmargin_bottom != null && !attr_rmargin_bottom.equals("") && hasBottomMargin) {
				    double bottomMargin = convertTwipsToPoints(_theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_BOTTOM),1);
					result += " " + attr_rmargin_bottom + "=\""+bottomMargin+"pt\"";
				}
				if (attr_rmargin_left != null && !attr_rmargin_left.equals("") && hasLeftMargin) {
				    double leftMargin = convertTwipsToPoints(_theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_LEFT),1);
					result += " " + attr_rmargin_left + "=\""+leftMargin+"pt\"";
				}
				if (attr_rmargin_right != null && !attr_rmargin_right.equals("") && hasRightMargin) {
				    double rightMargin = convertTwipsToPoints(_theRow.getProperties().getRowPadding(RtfRowProperties.ROWBORDER_RIGHT),1);
					result += " " + attr_rmargin_right + "=\""+rightMargin+"pt\"";
				}
			}

			return result;
		}
	}

	class CellFunctor implements XmlGeneratorFunctor {

		RtfCell _theCell;

		CellFunctor(RtfCell theCell)
		{
			_theCell = theCell;
		}

		public String doit(String tag)
		{
			String result = "";
			if (tag.equals("ce")) {
			    // cell width in original weird point-like measurement?
			    if (!_template.getGeneratorParam().isDefined("cwu") ||
			            _template.getGeneratorParam().getActualTag("cwu", true).equals("false")) {
				    // old evaluation (sf)
					int width_twips = _theCell.getProperties().getWidth();
					if (width_twips > 0) {
						String attr = _template.getGeneratorParam().getAttributes("cw");
						if (! attr.equals("")) {
						    result += " " + _template.getGeneratorParam().getAttributes("cw")
								+ "=\"" + (width_twips + 10) / 20 + "\"";
						}
					}
				} else {
				    // cell width in millimeters
					int width_twips = _theCell.getProperties().getWidth();
					// conversion to millimeter and round to 4 digits after comma
					double width_mm = convertTwipsToMM(width_twips, 4);
					if (width_mm > 0) {
						String attr_width = _template.getGeneratorParam().getAttributes("cw");
		
						if (attr_width != null && !attr_width.equals("")) {
							result += " " + attr_width
								+ "=\"" + width_mm + "mm\"";
						}
					}
				}

				// vertical alignment of the cell
				int valign = _theCell.getProperties().getVerticalAlignment();
				String attr_valign = _template.getGeneratorParam().getAttributes("cv");
				String valign_text = null;
				switch (valign) {
					case RtfCellProperties.VERT_ALIGN_TOP:
						valign_text = "top";
						break;
					case RtfCellProperties.VERT_ALIGN_CENTER:
						valign_text = "center";
						break;
					case RtfCellProperties.VERT_ALIGN_BOTTOM:
						valign_text = "bottom";
						break;
					default:
						break;
				}
				
				if (attr_valign != null && !attr_valign.equals("")) {
					result += " " + attr_valign
						+ "=\"" + valign_text + "\"";
				}
				
				// cell borders
				boolean hasNoBorders = _theCell.getProperties().hasNoBorderDefined();
				if (!hasNoBorders) {
					boolean hasTopBorderDefined = _theCell.getProperties().hasTopBorderDefined();
					boolean hasBottomBorderDefined = _theCell.getProperties().hasBottomBorderDefined();
					boolean hasLeftBorderDefined = _theCell.getProperties().hasLeftBorderDefined();
					boolean hasRightBorderDefined = _theCell.getProperties().hasRightBorderDefined();
					String attr_cborders_top = _template.getGeneratorParam().getAttributes("bortop");
					String attr_cborders_bottom = _template.getGeneratorParam().getAttributes("borbot");
					String attr_cborders_left = _template.getGeneratorParam().getAttributes("borle");
					String attr_cborders_right = _template.getGeneratorParam().getAttributes("borri");
					
					if (attr_cborders_top != null && !attr_cborders_top.equals("") && hasTopBorderDefined) {
					    double topWidth = convertTwipsToPoints(_theCell.getProperties().getBorderWidth(RtfCellProperties.CELLBORDER_TOP),1);
						result += " " + attr_cborders_top + "=\""+topWidth+"pt\"";
					}
					if (attr_cborders_bottom != null && !attr_cborders_bottom.equals("") && hasBottomBorderDefined) {
					    double bottomWidth = convertTwipsToPoints(_theCell.getProperties().getBorderWidth(RtfCellProperties.CELLBORDER_BOTTOM),1);
						result += " " + attr_cborders_bottom + "=\""+bottomWidth+"pt\"";
					}
					if (attr_cborders_left != null && !attr_cborders_left.equals("") && hasLeftBorderDefined) {
					    double leftWidth = convertTwipsToPoints(_theCell.getProperties().getBorderWidth(RtfCellProperties.CELLBORDER_LEFT),1);
						result += " " + attr_cborders_left + "=\""+leftWidth+"pt\"";
					}
					if (attr_cborders_right != null && !attr_cborders_right.equals("") && hasRightBorderDefined) {
					    double rightWidth = convertTwipsToPoints(_theCell.getProperties().getBorderWidth(RtfCellProperties.CELLBORDER_RIGHT),1);
						result += " " + attr_cborders_right + "=\""+rightWidth+"pt\"";
					}
				}
				
				// row paddings/margins
				boolean hasTopMargin = _theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_TOP) > -1;
				boolean hasBottomMargin = _theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_BOTTOM) > -1;
				boolean hasLeftMargin = _theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_LEFT) > -1;
				boolean hasRightMargin = _theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_RIGHT) > -1;
				String attr_cmargin_top = _template.getGeneratorParam().getAttributes("martop");
				String attr_cmargin_bottom = _template.getGeneratorParam().getAttributes("marbot");
				String attr_cmargin_left = _template.getGeneratorParam().getAttributes("marle");
				String attr_cmargin_right = _template.getGeneratorParam().getAttributes("marri");
				
				if (attr_cmargin_top != null && !attr_cmargin_top.equals("") && hasTopMargin) {
				    double topMargin = convertTwipsToPoints(_theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_TOP),1);
					result += " " + attr_cmargin_top + "=\""+topMargin+"pt\"";
				}
				if (attr_cmargin_bottom != null && !attr_cmargin_bottom.equals("") && hasBottomMargin) {
				    double bottomMargin = convertTwipsToPoints(_theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_BOTTOM),1);
					result += " " + attr_cmargin_bottom + "=\""+bottomMargin+"pt\"";
				}
				if (attr_cmargin_left != null && !attr_cmargin_left.equals("") && hasLeftMargin) {
				    double leftMargin = convertTwipsToPoints(_theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_LEFT),1);
					result += " " + attr_cmargin_left + "=\""+leftMargin+"pt\"";
				}
				if (attr_cmargin_right != null && !attr_cmargin_right.equals("") && hasRightMargin) {
				    double rightMargin = convertTwipsToPoints(_theCell.getProperties().getCellPadding(RtfCellProperties.CELLBORDER_RIGHT),1);
					result += " " + attr_cmargin_right + "=\""+rightMargin+"pt\"";
				}

				// colspan
				int colspan = _theCell.getProperties().getColSpan();
				String attr_colspan = _template.getGeneratorParam().getAttributes("colspan");
				
				if (attr_colspan != null && !attr_colspan.equals("") && colspan > 1) {
					result += " " + attr_colspan + "=\""+colspan+"\"";
				}

				// rowspan
				int rowspan = _theCell.getProperties().getRowSpan();
				String attr_rowspan = _template.getGeneratorParam().getAttributes("rowspan");
				
				if (attr_rowspan != null && !attr_rowspan.equals("") && rowspan > 1) {
					result += " " + attr_rowspan + "=\""+rowspan+"\"";
				}
			}

			return result;
		}
	}

	class ParagraphFunctor implements XmlGeneratorFunctor {

		RtfParagraph _theParagraph;

		ParagraphFunctor(RtfParagraph theParagraph)
		{
			_theParagraph = theParagraph;
		}

		public String doit(String tag) {
			String result = "";
			if (tag.equals("pa") ||
			    tag.equals("h1") ||
			    tag.equals("h2") ||
			    tag.equals("h3") ||
			    tag.equals("h4") ||
			    tag.equals("h5") ||
			    tag.equals("h6")
			    ) {
				// alignment of the paragraph
				int alignment = _theParagraph.getProperties()
						.getParagraphAlignment();
				String attr_align = _template.getGeneratorParam()
						.getAttributes("pal");
				String align_text = RtfParagraphProperties.ALIGNMENT_CODES[alignment];

				if (attr_align != null && !attr_align.equals("")) {
					result += " " + attr_align + "=\"" + align_text + "\"";
				}
				
				// first line indentation
				int first_indent = _theParagraph.getProperties().getFirstIndent();
				
				double first_mm = convertTwipsToMM(first_indent, 2);
				if (first_indent != 0) {
					String attr_fin = _template.getGeneratorParam().getAttributes("fin");
	
					if (attr_fin != null && !attr_fin.equals("")) {
						result += " " + attr_fin
							+ "=\"" + first_mm + "mm\"";
					}
				}

				// left indentation
				int left_indent = _theParagraph.getProperties().getLeftIndent();
				
				double left_mm = convertTwipsToMM(left_indent, 2);
				if (left_indent != 0) {
					String attr_lin = _template.getGeneratorParam().getAttributes("lin");
	
					if (attr_lin != null && !attr_lin.equals("")) {
						result += " " + attr_lin
							+ "=\"" + left_mm + "mm\"";
					}
				}

				// right indentation
				int right_indent = _theParagraph.getProperties().getRightIndent();
				
				double right_mm = convertTwipsToMM(right_indent, 2);
				if (right_indent != 0) {
					String attr_rin = _template.getGeneratorParam().getAttributes("rin");
	
					if (attr_rin != null && !attr_rin.equals("")) {
						result += " " + attr_rin
							+ "=\"" + right_mm + "mm\"";
					}
				}

				// space before lines
				int spacebefore_twips = _theParagraph.getProperties().getSpaceBefore();
				
				double space_mm = convertTwipsToMM(spacebefore_twips, 2);
				if (spacebefore_twips > 0) {
					String attr_sb = _template.getGeneratorParam().getAttributes("sbl");
	
					if (attr_sb != null && !attr_sb.equals("")) {
						result += " " + attr_sb
							+ "=\"" + space_mm + "mm\"";
					}
				}

				// space after lines
				int spaceafter_twips = _theParagraph.getProperties().getSpaceAfter();
				
				space_mm = convertTwipsToMM(spaceafter_twips, 2);
				if (spaceafter_twips > 0) {
					String attr_sa = _template.getGeneratorParam().getAttributes("sal");
	
					if (attr_sa != null && !attr_sa.equals("")) {
						result += " " + attr_sa
							+ "=\"" + space_mm + "mm\"";
					}
				}
				
				// line spacing
				String attr_ls = _template.getGeneratorParam().getAttributes("ls");
				String attr_lst = _template.getGeneratorParam().getAttributes("lst");
				int line_space_twips = _theParagraph.getProperties().getLineSpacing();
				boolean exactSpacing = (line_space_twips < 0); 
				
				double space_pt = convertTwipsToPoints(Math.abs(line_space_twips), 1);
				if (line_space_twips != 0) {
					if (!exactSpacing) {
					    // "at least"-spacing
						if (! attr_ls.equals("") && ! attr_lst.equals("")) {
							result += " " + attr_ls
								+ "=\"" + space_pt + "pt\" "
								+ attr_lst + "=\"atleast\"";
						}
					} else {
					    // exact spacing
						if (attr_ls != null && attr_lst != null && !attr_ls.equals("") && !attr_lst.equals("")) {
							result += " " + attr_ls
								+ "=\"" + space_pt + "pt\" "
								+ attr_lst + "=\"exact\"";
						}
					}
				}
			} else if (tag.equals("ol") || tag.equals("ul")) {
			    boolean orderedlist = _theParagraph.getProperties().getNumStyle()
			    					  >= RtfParagraphProperties.STYLE_NUMBERED_DEC;

			    // tab size between listbullet and itemparagraph
			    RtfTabDef listTabDef = _theParagraph.getProperties().getTabDefList().getListTabDef();
			    RtfTabDef defaultTabDef = _theParagraph.getProperties().getTabDefList().getDefaultTabDef();
			    // sorry for the ugly ternary double-expression... :-)
				int listtab_pos = (listTabDef == null ?
				                      (defaultTabDef == null ?
				                       RtfTabDef.TAB_DEFAULT_WIDTH :
				                       defaultTabDef.getTabPosition()) :
				                   listTabDef.getTabPosition());
                double listtabpos_mm = convertTwipsToMM(listtab_pos, 2);
                
				String attr_listtab = _template.getGeneratorParam()
						.getAttributes("ltd");
				if (attr_listtab != null && !attr_listtab.equals("")) {
					result += " " + attr_listtab + "=\"" + listtabpos_mm + "mm\"";
				}

				if (orderedlist) {
					// ordering type
					int order_type = _theParagraph.getProperties().getNumStyle();
					
					String attr_type = _template.getGeneratorParam()
							.getAttributes("olt");
					String type_text = RtfParagraphProperties.NUMBERTYPE_CODES[order_type-2];
					if (attr_type != null && !attr_type.equals("")) {
						result += " " + attr_type + "=\"" + type_text + "\"";
					}
					
					// ordering start
					int order_start = _theParagraph.getProperties().getOrderingStart();
					
					String attr_start = _template.getGeneratorParam()
							.getAttributes("ols");
					if (attr_start != null && !attr_start.equals("") && order_start > 1) {
						result += " " + attr_start + "=\"" + order_start + "\"";
					}
					
					// text before
					String textbefore = convertCharacter(_theParagraph.getProperties().getTextBefore());
					
					String attr_before = _template.getGeneratorParam()
							.getAttributes("ltb");
					if (attr_before != null && !attr_before.equals("") && textbefore != null) {
						result += " " + attr_before + "=\"" + textbefore + "\"";
					}
					
					// text after
					String textafter = convertCharacter(_theParagraph.getProperties().getTextAfter());
					
					String attr_after = _template.getGeneratorParam()
							.getAttributes("lta");
					if (attr_after != null && !attr_after.equals("") && textafter != null) {
						result += " " + attr_after + "=\"" + textafter + "\"";
					}
				} else {
				    // unordered list
				    // bullet
					String bullet = convertCharacter(_theParagraph.getProperties().getBullet());
					
					String attr_bullet = _template.getGeneratorParam()
							.getAttributes("ulb");
					if (attr_bullet != null && !attr_bullet.equals("") && bullet != null) {
						result += " " + attr_bullet + "=\"" + bullet + "\"";
					}
				}
			} else if (tag.equals("li")) {
				// dont show numbers
				boolean dontshow = _theParagraph.getProperties().getDontShowNumbering();
				
				String attr_dontshow = _template.getGeneratorParam()
						.getAttributes("old");
				if (attr_dontshow != null && !attr_dontshow.equals("") && dontshow) {
					result += " " + attr_dontshow + "=\"true\"";
				}
					
			    // bullettext in list items (workaround for revision changed orderedlists)
				String bullet = convertCharacter(_theParagraph.getProperties().getBulletText());
				
				String attr_bullettext = _template.getGeneratorParam()
						.getAttributes("ibt");
				if (attr_bullettext != null && !attr_bullettext.equals("") && bullet != null) {
					result += " " + attr_bullettext + "=\"" + bullet + "\"";
				}
			}
			return result;
		}
	}

	class RootFunctor implements XmlGeneratorFunctor {

		RtfDocument _theDocument;

		RootFunctor(RtfDocument theDocument)
		{
			_theDocument = theDocument;
		}

		public String doit(String tag) {
			String result = "";
			if (tag.equals("re")) {
				// columncount of the document
				int colcount = _theDocument.getColumnCount();
				// attribute name
				String attr_colcount = _template.getGeneratorParam()
						.getAttributes("colcount");

				if (attr_colcount != null && !attr_colcount.equals("")) {
					result += " " + attr_colcount
						+ "=\"" + colcount + "\"";
				}
			}
			return result;
		}
	}
}