/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.address.ejb.entity.beans;

import java.io.Serializable;

import javax.persistence.Entity;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.core.utils.GECAMedUtils;

/**
 * Entity Bean that maps the <b>address.zip</b> table
 * which stores a list of all available zip codes for luxembourg
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: Zip.java,v $
 * <br>Revision 1.7  2008-09-25 09:42:27  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.6  2008-01-15 09:29:39  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.5  2007-12-12 15:10:23  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */

@javax.persistence.NamedQueries({

    @javax.persistence.NamedQuery(name="findAllZip",
        query="SELECT OBJECT(o) FROM Zip o ORDER BY o.zip ASC"),
    @javax.persistence.NamedQuery(name="findAllZipByZip",
        query="SELECT OBJECT(o) FROM Zip o WHERE o.zip=:zip"),
    @javax.persistence.NamedQuery(name="findAllZipByLocality",
      query="SELECT OBJECT(o) FROM Zip o WHERE o.localityId=:localityId"),
    @javax.persistence.NamedQuery(name="findAllZipByStreet",
        query="SELECT OBJECT(o) FROM Zip o WHERE o.street LIKE :street"),
    @javax.persistence.NamedQuery(name="findAllZipByStreetAndLocality",
        query="SELECT OBJECT(o) FROM Zip o WHERE o.street LIKE :street AND o.localityId = :localityId")
	})

@Entity
@Table(name="zip", schema="address")
public class Zip extends GECAMedEntityBean implements Serializable {

	private static final long serialVersionUID = 1L;
	/** Regular field. */
	private Integer 	zip;
	private Integer 	localityId;
	private String 		street;
	private Character	parity;
	private Integer		first;
	private Integer		last;
	
	private transient String key = null;
	
	public final static Character ODD  = Character.valueOf('O');
	public final static Character EVEN = Character.valueOf('E');
	public final static Character NONE = Character.valueOf('N');
	
	
//---------------------------------------------------------------------------

	
@Transient
public String generateZipKey ()
	{
	StringBuffer l_Key;
	
	if (key != null) return key;
	
	l_Key = new StringBuffer ();
	l_Key.append (this.getZip()).append("_");
	l_Key.append (this.getStreet()).append("_");
	l_Key.append (this.getLocalityId());
	l_Key.append ("_").append (this.getParity());
	l_Key.append ("_").append (this.getFirst());
	l_Key.append ("_").append (this.getLast());
	
	key = l_Key.toString(); 
	
	return key;
	}

//---------------------------------------------------------------------------
/**
 * Overrides java.lang.Object equals method in order to make sure that
 * equality means same object ID.
 * @return <b>true</b> if the two objects have the same ID, <b>false</b> otherwise
 */
//---------------------------------------------------------------------------	 

@Override
public boolean equals (Object p_OtherZip)	
	{
	Zip l_OtherZip;
		
	if (p_OtherZip == null) return false;
	
	if (! (p_OtherZip instanceof Zip)) return false;

	l_OtherZip = (Zip) p_OtherZip;
	return l_OtherZip.generateZipKey().equals(this.generateZipKey());
	}

//---------------------------------------------------------------------------
/**
 * Overrides java.lang.Object hashCode method. Algorithm taken from 
 * Effective Java Book, Chapter 3. Overriding hashCode method is a MUST
 * when you override equals method.
 */
//---------------------------------------------------------------------------

@Override
public int hashCode ()
	{
	return this.generateZipKey().hashCode();
	}

//---------------------------------------------------------------------------
	
	/**
	 * @return Returns the localityId.
	 */
	@javax.persistence.Column(name = "locality_id")
	public Integer getLocalityId() {
		return localityId;
	}

	/**
	 * @param localityId The localityId to set.
	 */
	public void setLocalityId(Integer localityId) {
		this.localityId = localityId;
		this.key = null;
	}

	/**
	 * @return Returns the street.
	 */
	@javax.persistence.Column(name = "street")
	public String getStreet() {
		return street;
	}

	/**
	 * @param street The street to set.
	 */
	public void setStreet(String street) {
		this.street = street;
		this.key = null;
	}

	/**
	 * Get the zip.
	 *
	 * @return The zip.
	 */
	@javax.persistence.Column(name = "zip")
	public Integer getZip() {
		return zip;
	}

	/**
	 * Set the zip.
	 *
	 * @param zip The zip.
	 */
	public void setZip(Integer zip) {
		this.zip = zip;
		this.key = null;
	}

	
	/**
	 * sets the parity 
	 * 
	 * @return the parity
	 */
	@javax.persistence.Column(name = "parity")
	public Character getParity() {
		return (parity!=null)?parity:Zip.NONE;
	}

	/**
	 * Set the value.
	 *
	 * @param value The value.
	 */
	public void setParity(Character parity) {
		this.parity = parity;
		this.key = null;
	}

	/**
	 * gets the first.
	 *
	 * @return  The first.
	 */
	@javax.persistence.Column(name = "first")
	public Integer getFirst() {
		return (first!=null)?first:Integer.valueOf(0);
	}

	/**
	 * Set the first.
	 *
	 * @param value The first.
	 */
	public void setFirst(Integer first) {
		this.first = first;
		this.key = null;
	}
	
	/**
	 * the last.
	 *
	 * @return The last.
	 */
	@javax.persistence.Column(name = "last")
	public Integer getLast() {
		return (last!=null)?last:Integer.valueOf(0);
	}

	/**
	 * Set the value.
	 *
	 * @param value The value.
	 */
	public void setLast(Integer last) {
		this.last = last;
		this.key = null;
	}
	
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
        return String.valueOf(getZip());
    }

//	@Override
//	public boolean equals(Object obj) {
//		try {
//			return this.getZip().equals(((Zip)obj).getZip());
//		} catch (Exception e) {
//			return false;
//		}		
//	}
    
	@Override
	public int compareTo (GECAMedEntityBean o)
	{
		if (o != null && o instanceof Zip)
		{
			int i = GECAMedUtils.compareTo(this.getZip(), ((Zip)o).getZip());
			if (i != 0)
				return i;
		}
		return super.compareTo(o);
	}
}
