/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.agenda.ejb.session.helper;

import java.io.Serializable;
import java.util.Date;

import bizcal.util.DateUtil;

/**
 * Class that holds start time, endtime and day of week.
 * The times are specified as simple integer values:<br>
 * <b>e.g.</b> <ul> <li> 8:00 -> 800</li>
 * 					<li>12:34 -> 1234</li>
 * 			</ul>
 * The dayOfWeek are the constants from the standard java Calendar class.
 *
 * 
 *  can be moved to gui
 * 
 * @author martin.heinemann@tudor.lu
 * 10.07.2007
 * 10:35:23
 *
 *
 * @version
 * <br>$Log: TimeInterval.java,v $
 * <br>Revision 1.5  2008-09-25 09:42:27  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.4  2008-01-18 16:09:05  heinemann
 * <br>code cleanup and java doc
 * <br>
 * <br>Revision 1.3  2007/08/09 14:10:34  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.2  2007/07/19 14:09:47  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.1  2007/07/16 09:12:45  heinemann
 * <br>*** empty log message ***
 * <br>
 *   
 */
public class TimeInterval implements Serializable, Comparable<TimeInterval> {
	
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private int starttime 	= 0;
	private int endtime 	= 0;
	private int dayOfWeek 	= -1;
	
	
	
	
	
	
	/**
	 * @param starttime
	 * @param endtime
	 */
	public TimeInterval(int starttime, int endtime) {
		this.starttime  = adaptTime(starttime);
		this.endtime 	= adaptTime(endtime);
	}

	/**
	 * @param starttime
	 * @param endtime
	 * @param dayOfWeek
	 */
	public TimeInterval(int starttime, int endtime, int dayOfWeek) {
		/* ================================================== */
		this.starttime 	= adaptTime(starttime);
		this.endtime 	= adaptTime(endtime);
		this.dayOfWeek 	= dayOfWeek;
		/* ================================================== */
	}
	
	
	/**
	 * Creates a TimeInterval object and extracts only the
	 * time of the dates.
	 * 
	 * @param startDate
	 * @param endDate
	 */
	public TimeInterval (Date startDate, Date endDate) {
		/* ================================================== */
		this.starttime = DateUtil.getHourOfDay(startDate)*100 + DateUtil.getMinuteOfHour(startDate);
		this.endtime   = DateUtil.getHourOfDay(endDate)*100   + DateUtil.getMinuteOfHour(endDate);
		/* ================================================== */
	}
	
	public TimeInterval(TimeInterval oldInterval) {
		/* ================================================== */
		this.setDayOfWeek(oldInterval.getDayOfWeek());
		this.setStarttime(oldInterval.getStarttime());
		this.setEndtime(oldInterval.getEndtime());
		/* ================================================== */
	}
	

	/**
	 * @return the dayOfWeek
	 */
	public int getDayOfWeek() {
		return dayOfWeek;
	}

	/**
	 * @param dayOfWeek the dayOfWeek to set
	 */
	public void setDayOfWeek(int dayOfWeek) {
		this.dayOfWeek = dayOfWeek;
	}

	/**
	 * @return the endtime
	 */
	public int getEndtime() {
		return endtime;
	}

	/**
	 * @param endtime the endtime to set
	 */
	public void setEndtime(int endtime) {
		this.endtime = adaptTime(endtime);
	}

	/**
	 * @return the starttime
	 */
	public int getStarttime() {
		return starttime;
	}

	/**
	 * @param starttime the starttime to set
	 */
	public void setStarttime(int starttime) {
		this.starttime = adaptTime(starttime);
	}
	
	
	/**
	 * Adapts the time in case the values are out of reach, like 32000.
	 * 
	 * @param time
	 * @return
	 */
	private int adaptTime(int time) {
		/* ================================================== */
		if (time < 1)
			time = 0;
		/* ------------------------------------------------------- */
		else
		if (time < 100) {
			int rest = (time % 60);
			time = 100+rest;
		} else
			if (time < 2400) {
				// get the "hours"
				int hours = time/100;
				// minutes
				int minutes = time - (hours*100);
				if (minutes >= 60) {
					int rest = minutes % 60;
					// rest hin, eins im Sinn
					hours = hours + 1;
					minutes = rest;
				}
				if (hours > 23)
					hours = hours - 23;
				/* ------------------------------------------------------- */
				time = (hours*100)+minutes;
			}
		return time;
		/* ================================================== */
	}
	
	
	public String toString() {
		/* ================================================== */
		return "TimeInterval ["+getStarttime()+", "+getEndtime()+"] - WeekDay "+ getDayOfWeek();
		/* ================================================== */
	}

	
	/**
	 * @param date
	 * @return
	 */
	public Date getStartDate(Date date) {
		/* ================================================== */
		try {
			Date d = DateUtil.round2Hour(date, (getStarttime()/100));
			d = DateUtil.round2Minute(d, (getStarttime() - (getStarttime()/100*100)));
			return d;
		} catch (Exception e) {
			e.printStackTrace();
			return date;
		}
		/* ================================================== */
	}
	
	/**
	 * @param date
	 * @return
	 */
	public Date getEndDate(Date date) {
		/* ================================================== */
		try {
			Date d = DateUtil.round2Hour(date, (getEndtime()/100));
			d = DateUtil.round2Minute(d, (getStarttime() - (getStarttime()/100*100)));
			return d;
		} catch (Exception e) {
			e.printStackTrace();
			return date;
		}
		/* ================================================== */
	}
	
	
	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(TimeInterval o) {
		/* ====================================================== */
		// sorting by start time and dow
		if (o.getDayOfWeek() > dayOfWeek)
			return -1;
		if (o.getDayOfWeek() < dayOfWeek)
			return 1;
		/* ------------------------------------------------------- */
		// else, if dow == dow
		if (o.getStarttime() > starttime)
			return -1;
		if (o.getStarttime() < starttime)
			return 1;
		return 0;
		/* ====================================================== */
	}
	
	
	
	
	
}
