/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.agenda.ejb.session.interfaces;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Remote;
import javax.jms.JMSException;

import lu.tudor.santec.gecamed.agenda.ejb.entity.beans.AgendaCalendar;
import lu.tudor.santec.gecamed.agenda.ejb.entity.beans.Appointment;
import lu.tudor.santec.gecamed.agenda.ejb.entity.beans.AppointmentType;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;

/**
 * @author martin.heinemann@tudor.lu
 * 15.03.2007
 * 14:56:46
 *
 *
 * @version
 * <br>$Log: AppointmentManager.java,v $
 * <br>Revision 1.25  2012-06-28 15:57:19  troth
 * <br>fix bug: Ticket #878  Existing recurring appointments not taken into account by NAF (first test version).
 * <br>
 * <br>Revision 1.24  2010-07-08 08:24:20  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.23  2009-04-03 13:22:05  heinemann
 * <br>fix for: Ticket #266 (new enhancement)
 * <br>
 * <br>If Color of a physician is changed, it will not change the color of his Calendar
 * <br>
 * <br>Revision 1.22  2008-09-30 14:53:48  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.21  2008-09-25 09:42:27  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.20  2008-04-08 09:52:33  heinemann
 * <br>moved creation of new calendar from OfficeBean to AgendaModule
 * <br>
 * <br>Revision 1.19  2008-02-27 08:21:24  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.18  2008-02-11 16:45:52  heinemann
 * <br>removed waitingroom methods
 * <br>
 * <br>Revision 1.17  2008-01-18 16:09:05  heinemann
 * <br>code cleanup and java doc
 * <br>
 * <br>Revision 1.16  2007-09-17 13:06:03  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.15  2007/08/28 11:34:36  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.14  2007/08/28 11:29:59  hermen
 * <br>do not delete appointment types, set them deprecated
 * <br>
 * <br>Revision 1.10  2007/07/16 12:06:38  hermen
 * <br>added search for Appointments
 * <br>
 * <br>Revision 1.3  2007/06/04 12:03:34  hermen
 * <br>load and store selected calendars
 * <br>
 * <br>Revision 1.2  2007/05/25 13:50:25  heinemann
 * <br>pres-weekend checkin
 * <br>
 * <br>Revision 1.1  2007/03/27 05:55:22  heinemann
 * <br>initial checkin
 * <br>
 *
 */
@Remote
public interface AppointmentManager {


	public static final Integer TYPE_PATIENT 	 = 0;
	public static final Integer TYPE_APPOINTMENT = 1;

	// ======================================================
	// constants for JMS messages
	// ======================================================
	public static final Integer MSG_UPDATE_CALENDAR_EVENTS 	= 1;
	public static final Integer MSG_REMOVED_CALENDAR 		= 2;
	public static final Integer MSG_ADDED_CALENDAR 			= 3;
	public static final Integer MSG_UPDATE_CALENDAR 		= 4;

	public static final String TOPIC_NAME = "topic/GECAMed/calendarUpdateTopic";
	
	
	/**
	 * Search direction -> into the future
	 */
	public static final int FORWARD   = 1;
	/**
	 * Search direction -> into the past 
	 */
	public static final int BACKWARDS = 2;
	
	/* ##########################################################
	 *
	 * Appointment methods
	 *
	 */


	/**
	 *
	 * Save an Appointment object.
	 * created and createdBy are filled in by the session bean
	 * @param app the object to save
	 * @return the merged object
	 */
	@RolesAllowed("gecam")
	public Appointment saveAppointment(Appointment app);
	
	/**
	 *
	 * Save an Appointment object.
	 * created and createdBy are filled in by the session bean
	 * @param app the object to save
	 * @return the merged object
	 */
	@RolesAllowed("gecam")
	public Appointment saveAppointment(Appointment app, boolean resetNotified);

	/**
	 * Returns all appointments for the given calendar id
	 *
	 * @param calenderId
	 * @return
	 */
	@RolesAllowed("gecam")
	public List<Appointment> getAppointments(Integer calenderId, Date from, Date to, String client);
	
	
	public List<Appointment> getAppointments(Integer calendarId, Date from, Date to, Date modifiedAfter, boolean recuring, boolean debug);
	
	/**
	 * Returns a list of all upcoming appointment of a patient
	 * 
	 * @param patientId
	 * @param limit max amoaunt of appointments
	 * @return
	 */
	@RolesAllowed("gecam")
	public List<Appointment> getAppointments(Integer patientId, Integer limit);
	
	/**
	 * returns the first appointment matching the given Calendar ID and Desription,
	 * else null
	 * 
	 * @param calId
	 * @param description
	 * @return
	 */
	public Appointment getAppointmentByCalIDandDesc(Integer calId, String description);
	
	/* ##########################################################
	 *
	 * AppointmentType methods
	 *
	 */

	/**
	 * @return
	 */
	@RolesAllowed("gecam")
	public Collection<AppointmentType> getAppointmentTypes(Integer calendarId, boolean deprecated);

	/**
	 * @param type
	 * @return
	 */
	@RolesAllowed("gecam")
	public AppointmentType saveAppointmentType(String clientId, AppointmentType type);


	/* ##########################################################
	 *
	 * AgendaCalendar methods
	 *
	 */

	/**
	 * Returns all calendars in the system.
	 *
	 * @return
	 */
	/**
	 * @return
	 */
	@RolesAllowed("gecam")
	public Collection<AgendaCalendar> getCalendars();

	/**
	 * @param calendar
	 * @return
	 */
	@RolesAllowed("gecam")
	public AgendaCalendar saveCalendar(String clientId, AgendaCalendar calendar);

	/**
	 * Remove the calendar and all its appointments
	 *
	 * @param calendar
	 */
	@RolesAllowed("gecam")
	public void removeCalendar(String clientId, AgendaCalendar calendar);

	/**
	 * @param userId
	 * @return
	 */
	@RolesAllowed("gecam")
	public Collection<AgendaCalendar> getCalendarsByUser(Integer userId);

	/**
	 * @param physicianId
	 * @return
	 */
	@RolesAllowed("gecam")
	public Collection<AgendaCalendar> getCalendarsByPhysician(Integer physicianId);

	
	
	/**
	 * @param app
	 * @param direction
	 * @return
	 */
	@RolesAllowed("gecam")
	public Appointment checkFreeAppointment(Appointment app, int direction, Locale locale);
	
	
	/**
	 * @param patientId
	 * @return
	 */
	@RolesAllowed("gecam")
	public Appointment getNextAppointmentOfPatient(Integer patientId);
	
	
	

	@RolesAllowed("gecam")
	public Collection<AgendaCalendar> getCalendarSelectionsByUser(Integer userId);

	@RolesAllowed("gecam")
	public void addUserCalendarSelection(AgendaCalendar cal);

	@RolesAllowed("gecam")
	public void deleteUserCalendarSelection(AgendaCalendar cal);

	@RolesAllowed("gecam")
	public void deleteAppointment(String clientId, Appointment ap);
	
	@RolesAllowed("gecam")
	public void deleteAppointmentType(String clientId, AppointmentType at);
	
	@RolesAllowed("gecam")
	public Collection<AppointmentType> getAppointmentTypesByID(Integer type, Integer calendarId, boolean deprecated);

	@RolesAllowed("gecam")
	public AgendaCalendar getCalendar(Integer calendarId);
	
	@RolesAllowed("gecam")
	public void sendCalendarUpdateMessage(String clientId, Integer type, Integer calendarId, Date changedDate) throws JMSException;

	@RolesAllowed("gecam")
	public List<Appointment> searchAppointments(Collection<AgendaCalendar> calendars, String searchString);

	/**
	 * returns the Appointment object. Null for nothing found
	 * @param appointmentId
	 * @return
	 */
	public Appointment getAppointment(Integer appointmentId);
	
	/**
	 * returns the color of the calendar
	 * 
	 * @param calendarId
	 * @return
	 */
	@RolesAllowed("gecam")
	public Integer getCalendarColor(Integer calendarId);
	
	/**
	 * Returns the appointment for the patient that takes place today.
	 * 
	 * @param patientId
	 * @return
	 */
	@RolesAllowed("gecam")
	public Appointment getCurrentAppointment(Integer patientId, Date date);
	
	/**
	 * Creates an initial calendar for the physician, will do nothing if there is already a calendar for this physcian.
	 * 
	 * @param physician
	 */
	@RolesAllowed("gecamed")
	public Boolean createNewCalendar(Physician physician);

	
	/**
	 * Create a default calendar
	 */
	@RolesAllowed("gecamed")
	public AgendaCalendar createDefaultCalendar();


}
