/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.agenda.gui.widgets;

import java.awt.CardLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSeparator;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.TitledBorder;

import lu.tudor.santec.gecamed.agenda.ejb.entity.beans.Appointment;
import lu.tudor.santec.gecamed.agenda.gui.AgendaModule;
import lu.tudor.santec.gecamed.agenda.gui.widgets.recurrence.AbstractFrequencePanel;
import lu.tudor.santec.gecamed.agenda.gui.widgets.recurrence.DailyFreqPanel;
import lu.tudor.santec.gecamed.agenda.gui.widgets.recurrence.MonthlyFreqPanel;
import lu.tudor.santec.gecamed.agenda.gui.widgets.recurrence.WeeklyFreqPanel;
import lu.tudor.santec.gecamed.agenda.gui.widgets.recurrence.YearlyFreqPanel;
import lu.tudor.santec.gecamed.agenda.utils.RRule;
import lu.tudor.santec.gecamed.core.gui.utils.GECAMedGuiUtils;
import lu.tudor.santec.i18n.Translatrix;
import bizcal.util.DateUtil;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;
import com.toedter.calendar.JDateChooser;

/**
 * 
 * 
 * Panel that extends abstract AppointmentEditPanel to edit the recurring information of
 * an Appointment. Possible recurrances are daily, weekly and weekly per day, monthly and annual.
 * As well as an end date and an end recurring number.
 * 
 * 
 * @author martin.heinemann@tudor.lu
 * 18.01.2008
 * 08:30:41
 *
 *
 * @version
 * <br>$Log: RecurrencePanel.java,v $
 * <br>Revision 1.30  2012-06-28 15:32:44  troth
 * <br>Move the java class RRule from the package 'lu.tudor.santec.gecamed.agenda.gui.widgets.recurrence' to 'lu.tudor.santec.gecamed.agenda.utils' because the class is need in the agenda beans.
 * <br>
 * <br>Revision 1.29  2011-11-28 16:11:49  troth
 * <br>code clearup
 * <br>
 * <br>Revision 1.28  2011-11-09 14:42:59  troth
 * <br>1. to translation for end date
 * <br>2. fix bug: the time interval of the different periods was rolled on the calendar and not add.
 * <br>
 * <br>Revision 1.27  2011-08-05 15:06:07  troth
 * <br>fix ticket #891
 * <br>
 * <br>Revision 1.26  2011-08-01 14:58:44  troth
 * <br>Fix tickets #874 #839.
 * <br>
 * <br>Revision 1.25  2011-06-28 13:48:27  troth
 * <br>testing agenda
 * <br>
 * <br>Revision 1.23  2011-06-16 15:13:14  troth
 * <br>fix Ticket #863 Dialog 'Wiederholungsoption' beh�lt alte Werte zur�ck
 * <br><http://santec.tudor.lu/trac/gecamed/ticket/863>
 * <br>
 * <br>Revision 1.22  2011-05-12 12:49:14  troth
 * <br>Fix little Bug - Dialog was not reset to init values when create a new appointment.
 * <br>
 * <br>Revision 1.21  2011-05-11 15:07:08  troth
 * <br>fix followup bug of Ticket  839 http://santec.tudor.lu/trac/gecamed/ticket/839
 * <br>
 * <br>Revision 1.20  2011-05-10 16:50:52  troth
 * <br>code clean up
 * <br>
 * <br>Revision 1.19  2011-05-09 14:30:20  troth
 * <br>Fix Ticket 839: "End after [X] occurrences" not taken into account
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/839
 * <br>
 * <br>Revision 1.18  2011-04-05 08:47:46  troth
 * <br>1. Add new icon for the button 'create new appointment'
 * <br>2. Add some translations
 * <br>3. Fix some warnings
 * <br>
 * <br>Revision 1.17  2011-04-04 11:38:30  troth
 * <br>Redesign of the appointment create dialog and the right navi bar.
 * <br>
 * <br>Revision 1.16  2008-09-25 09:42:27  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.15  2008-05-30 11:46:43  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.14  2008-05-29 12:43:39  heinemann
 * <br>fixed recurrence bug for appointments that recurr per week.
 * <br>The days in the current week are now also treated.
 * <br>
 * <br>Revision 1.13  2008-01-18 16:09:05  heinemann
 * <br>code cleanup and java doc
 * <br>
 *   
 */
public class RecurrencePanel extends AppointmentEditPanel implements ActionListener{

	private static final long serialVersionUID = 1L;

	private Icon icon = AgendaModule.getMediumIcon(AgendaModule.ICON_NEW_RECUREVENT);

	private JCheckBox enableCheckBox;

	private JLabel summaryLabel;

	private JPanel repeatPanel;

	private JPanel periodPanel;

	private ButtonGroup freqGroup;

	private JRadioButton rDaily;

	private JRadioButton rWeekly;

	private JRadioButton rMonthly;

	private JRadioButton rYearly;

	private CellConstraints cc = new CellConstraints();

	private JPanel cardPanel;

	private CardLayout cardLayout;

	private TitledBorder summaryLabelBorder;

	private TitledBorder repeatPanelBorder;

	private TitledBorder periodPanelBorder;

	private JSeparator seperator;

	private LinkedHashMap<String, AbstractFrequencePanel> freqPanels = new LinkedHashMap<String, AbstractFrequencePanel>();
	
	private LinkedHashMap<String, JRadioButton> freqButtons = new LinkedHashMap<String, JRadioButton>();
	
	private boolean isenabled;

	private AbstractFrequencePanel currFreqPanel;
	
	private DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT);

	private ButtonGroup periodGroup;

	private JRadioButton periodNoEnd;

	private JRadioButton periodCountEnd;

	private JRadioButton periodFixedEnd;

	private JDateChooser periodEndDate;

	private JSpinner periodCountSpinner;

	private JLabel periodsummary;
	
	private JLabel periodCountLabel;
	
	private AbstractFrequencePanel dayPanel;
	
	private AbstractFrequencePanel weekPanel;
	
	private AbstractFrequencePanel monthPanel;
	
	private AbstractFrequencePanel yearPanel;
	//private boolean withoutSummary = false;

	private HashMap<Integer, Boolean> enableConfiguration = new HashMap<Integer, Boolean>();
	
	
	RecurrencePanel() {
		/* ================================================== */
		this.initComponents();
		this.buildPanel();
		this.initRecurrencePanel(false);
		/* ================================================== */
	}
	
	/**
	 * init the components of the recurrence panel
	 * 
	 * @param withoutSummaryPanel
	 */
	public void initRecurrencePanel(boolean enable) {
		
		this.rWeekly.doClick();
		this.periodNoEnd.doClick();
		this.periodCountSpinner.setEnabled(false);
		this.periodCountLabel.setEnabled(false);
		this.periodEndDate.setEnabled(false);
		this.enablePanel(enable);	
	}
	
	
	private void buildPanel() {
		
		FormLayout fl = new FormLayout(
				"3dlu, pref, 3dlu, fill:155dlu:grow, 3dlu",
				"3dlu, top:25dlu, 3dlu, fill:pref:grow, 3dlu, fill:pref:grow, 3dlu");
		
		this.setLayout(fl);
		this.setOpaque(false);
		this.add(this.enableCheckBox, cc.xy(2,2));
		this.add(this.summaryLabel, cc.xy(4,2));
		this.add(this.repeatPanel, cc.xyw(2,4,3));
		this.add(this.periodPanel, cc.xyw(2,6,3));
	}

	private void initComponents() {
		this.enableCheckBox = new JCheckBox(Translatrix.getTranslationString("calendar.recurrence.activate"));
		this.enableCheckBox.addActionListener(this);
		this.enableCheckBox.setOpaque(false);
		this.summaryLabel = new JLabel();
		this.summaryLabelBorder = new TitledBorder(Translatrix.getTranslationString("calendar.recurrence.summary"));
		this.summaryLabel.setBorder(summaryLabelBorder);
		
		
		// --- REPEAT PANEL --------------------------------------------------------------------------------------------------------------------
		this.repeatPanel = new JPanel();
		this.repeatPanel.setOpaque(false);
		this.repeatPanelBorder = new TitledBorder(Translatrix.getTranslationString("calendar.recurrence.frequence"));
		this.repeatPanel.setBorder(repeatPanelBorder);
		this.repeatPanel.setLayout(new FormLayout(
				"3dlu, fill:pref, right:10dlu, 3dlu, fill:pref:grow, 3dlu",
				"3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu"));
		
		this.freqGroup = new ButtonGroup();
		
		this.rDaily = new JRadioButton(Translatrix.getTranslationString("calendar.recurrence.daily"));
		this.rDaily.setOpaque(false);
		this.rDaily.setActionCommand(Appointment.DAILY + "");
		this.rDaily.addActionListener(this);
		this.freqButtons.put(Appointment.DAILY + "", this.rDaily);
		this.freqGroup.add(rDaily);
		this.repeatPanel.add(rDaily, cc.xy(2, 2));
		
		this.rWeekly = new JRadioButton(Translatrix.getTranslationString("calendar.recurrence.weekly"));
		this.rWeekly.setOpaque(false);
		this.rWeekly.setActionCommand(Appointment.WEEKLY + "");
		this.rWeekly.addActionListener(this);
		this.freqButtons.put(Appointment.WEEKLY + "", this.rWeekly);
		this.freqGroup.add(rWeekly);
		this.repeatPanel.add(rWeekly, cc.xy(2, 4));
		
		this.rMonthly = new JRadioButton(Translatrix.getTranslationString("calendar.recurrence.monthly"));
		this.rMonthly.setOpaque(false);
		this.rMonthly.setActionCommand(Appointment.MONTHLY + "");
		this.rMonthly.addActionListener(this);
		this.freqButtons.put(Appointment.MONTHLY + "", this.rMonthly);
		this.freqGroup.add(rMonthly);
		this.repeatPanel.add(rMonthly, cc.xy(2, 6));
		
		this.rYearly = new JRadioButton(Translatrix.getTranslationString("calendar.recurrence.yearly"));
		this.rYearly.setOpaque(false);
		this.rYearly.setActionCommand(Appointment.YEARLY + "");
		this.rYearly.addActionListener(this);
		this.freqButtons.put(Appointment.YEARLY + "", this.rYearly);
		this.freqGroup.add(rYearly);
		this.repeatPanel.add(rYearly, cc.xy(2, 8));
	
		seperator = new JSeparator(JSeparator.VERTICAL);
		this.repeatPanel.add(seperator, cc.xywh(3, 2, 1, 7));
		
		this.cardLayout = new CardLayout();
		this.cardPanel = new JPanel(cardLayout);
		this.cardPanel.setOpaque(false);
		this.repeatPanel.add(cardPanel, cc.xywh(5, 2, 1, 7));
		
		this.dayPanel = new DailyFreqPanel();
		this.cardPanel.add(dayPanel, Appointment.DAILY + "");
		freqPanels.put(Appointment.DAILY + "", dayPanel);
		
		this.weekPanel = new WeeklyFreqPanel();
		this.cardPanel.add(weekPanel, Appointment.WEEKLY + "");
		freqPanels.put(Appointment.WEEKLY + "", weekPanel);
		
		this.monthPanel = new MonthlyFreqPanel();
		this.cardPanel.add(monthPanel, Appointment.MONTHLY + "");
		freqPanels.put(Appointment.MONTHLY + "", monthPanel);
		
		this.yearPanel = new YearlyFreqPanel();
		this.cardPanel.add(yearPanel, Appointment.YEARLY + "");
		freqPanels.put(Appointment.YEARLY + "", yearPanel);
		

		// --- PERIOD PANEL --------------------------------------------------------------------------------------------------------------------
		this.periodPanel = new JPanel(new FormLayout(
				"pref, 3dlu, 30dlu, 3dlu, 40dlu, 3dlu, pref:grow",
				"pref, 3dlu, pref, 3dlu, pref, 3dlu , pref"));
		this.periodPanel.setOpaque(false);
		this.periodPanelBorder = new TitledBorder(Translatrix.getTranslationString("calendar.recurrence.period"));
		this.periodPanel.setBorder(periodPanelBorder);
		
		this.periodsummary = new JLabel();
		
		this.periodPanel.add(periodsummary, cc.xyw(1,1, 5 ));
		
		this.periodGroup = new ButtonGroup();
		
		this.periodNoEnd = new JRadioButton(Translatrix.getTranslationString("calendar.recurrence.periodNoEnd"));
		this.periodNoEnd.setOpaque(false);
		this.periodNoEnd.addActionListener(this);
		this.periodGroup.add(periodNoEnd);
		this.periodPanel.add(periodNoEnd, cc.xy(1,3 ));
		
		this.periodCountEnd = new JRadioButton(Translatrix.getTranslationString("calendar.recurrence.periodCountEnd"));
		this.periodCountEnd.setOpaque(false);
		this.periodCountEnd.addActionListener(this);
		this.periodGroup.add(periodCountEnd);
		this.periodPanel.add(periodCountEnd, cc.xy(1,5 ));
		this.periodCountSpinner = new JSpinner(new SpinnerNumberModel(1, 1, 200, 1));
		this.periodPanel.add(periodCountSpinner, cc.xy(3,5 ));
		this.periodCountLabel = new JLabel(Translatrix.getTranslationString("calendar.recurrence.recurrences"));
		this.periodPanel.add(periodCountLabel, cc.xyw(5,5,3 ));
		
		this.periodFixedEnd = new JRadioButton(Translatrix.getTranslationString("calendar.recurrence.periodFixedEnd"));
		this.periodFixedEnd.setOpaque(false);
		this.periodFixedEnd.addActionListener(this);
		this.periodGroup.add(periodFixedEnd);
		this.periodPanel.add(periodFixedEnd, cc.xy(1,7 ));
		
		this.periodEndDate = GECAMedGuiUtils.getDateChooser(false);
		this.periodPanel.add(periodEndDate, cc.xyw(3,7, 3 ));
		
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.gecamed.agenda.gui.widgets.AppointmentEditPanel#getAppointment(lu.tudor.santec.gecamed.agenda.ejb.entity.beans.Appointment)
	 */
	@Override
	public void getAppointment(Appointment appointment) {
		
		if (RecurrenceOptionDialog.getInstance().wasOpen)
		{
			if (isenabled) {
				int freq = Integer.parseInt(freqGroup.getSelection().getActionCommand());
				appointment.setFrequency(freq);
				RRule r = new RRule();
				currFreqPanel.getRule(r);
				appointment.setRRule(r.getRule());
			
				if (this.periodNoEnd.isSelected()) {
					appointment.setUntil(null);
				} else if (this.periodCountEnd.isSelected()) {
					Calendar c = new GregorianCalendar();
					c.setTime(appointment.getStartDate());
					for (int i = 0; i < ((Integer) periodCountSpinner.getValue()); i++) {
						if (freq == Appointment.DAILY) {
							c.add(Calendar.DAY_OF_WEEK, currFreqPanel.getRecurrsEvery());
						} else if (freq == Appointment.WEEKLY) {
							c.add(Calendar.WEEK_OF_YEAR, currFreqPanel.getRecurrsEvery());
						} else if (freq == Appointment.MONTHLY) {
							c.add(Calendar.MONTH, currFreqPanel.getRecurrsEvery());
						} else if (freq == Appointment.YEARLY) {
							c.add(Calendar.YEAR, currFreqPanel.getRecurrsEvery());
						} 
					}
					/* ------------------------------------------------------- */
					// set hours and minutes of the appointment to the until date
					c.set(Calendar.HOUR_OF_DAY, DateUtil.getHourOfDay(   appointment.getStartDate()));
					c.set(Calendar.MINUTE,      DateUtil.getMinuteOfHour(appointment.getStartDate()));
					/* ------------------------------------------------------- */
					
					appointment.setUntil(c.getTime());
				} else if (this.periodFixedEnd.isSelected()) {
					appointment.setUntil(periodEndDate.getDate());
				}
			} else {
				appointment.setFrequency(Appointment.NO_RECUR);
				appointment.setRRule("");
			}
		}
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.gecamed.agenda.gui.widgets.AppointmentEditPanel#setAppointment(lu.tudor.santec.gecamed.agenda.ejb.entity.beans.Appointment)
	 */
	@Override
	public void setAppointment(Appointment appointment) {
		
		if(appointment.getUntil() != null)
			this.summaryLabel.setText(
				df.format(appointment.getStartDate()) + 
				" - " +
				df.format(appointment.getUntil()));
		else
			this.summaryLabel.setText(
					df.format(appointment.getStartDate()) + 
					" - " + 
					Translatrix.getTranslationString("Agenda.Recurrence.noEnddate"));
		
		this.periodsummary.setText(Translatrix.getTranslationString("calendar.recurrence.starts") + " " +
				df.format(appointment.getStartDate()));
		// set the current Day as default in the weekly recurrence panel (C
		Calendar cal = new GregorianCalendar();
		cal.setTime(appointment.getStartDate());
		int dayNumber = cal.get(Calendar.DAY_OF_WEEK);
		((WeeklyFreqPanel) freqPanels.get(Appointment.WEEKLY + "")).setWeekday(dayNumber);
		
		if (! appointment.getFrequency().equals(Appointment.NO_RECUR))
		{
			// enable the panel
			enableCheckBox.setSelected(true);
			this.enablePanel(true);
			// set frequency
			freqButtons.get(appointment.getFrequency() + "").doClick();
			// set the rule to the panels
			RRule rule = new RRule(appointment.getRRule());
			for (Iterator<AbstractFrequencePanel> iter = freqPanels.values().iterator(); iter.hasNext();) {
				AbstractFrequencePanel panel = (AbstractFrequencePanel) iter.next();
				panel.setRule(rule);
			}
			if (appointment.getUntil() != null) {
				periodFixedEnd.doClick();
				periodEndDate.setDate(appointment.getUntil());
			} else {
				periodNoEnd.doClick();
				periodEndDate.setDate(new Date());
			}
		}	else {
			enableCheckBox.setSelected(false);
			this.enablePanel(false);
		}
		
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.gecamed.agenda.gui.widgets.AppointmentEditPanel#getIcon()
	 */
	@Override
	public Icon getIcon() {
		/* ====================================================== */
		return this.icon;
		/* ====================================================== */
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		/* ================================================== */
		// select the check box to enable recurrence panel
		if (e.getSource().equals(this.enableCheckBox))
		{
			this.enablePanel(this.enableCheckBox.isSelected());

			if(this.enableCheckBox.isSelected())
			{
				if (this.periodNoEnd.isSelected())
				{
					this.periodCountSpinner.setVisible(false);
					this.periodCountLabel.setVisible(false);
					this.periodEndDate.setVisible(false);
				}
				if (this.periodCountEnd.isSelected())
				{
					this.periodCountSpinner.setValue(1);
					this.periodCountSpinner.setVisible(true);
					this.periodCountLabel.setVisible(true);
					this.periodEndDate.setVisible(false);	
				}
				if (this.periodFixedEnd.isSelected())
				{
					this.periodCountSpinner.setVisible(false);
					this.periodCountLabel.setVisible(false);
					this.periodEndDate.setVisible(true);
				}
			}
		} else if (e.getSource() instanceof JRadioButton) {
			if(
					e.getSource().equals(this.rDaily) ||
					e.getSource().equals(this.rWeekly) ||
					e.getSource().equals(this.rMonthly) ||
					e.getSource().equals(this.rYearly)
			)
			{
				cardLayout.show(cardPanel, e.getActionCommand());
				currFreqPanel = freqPanels.get(e.getActionCommand());
			}
			if (e.getSource().equals(this.periodNoEnd))
			{
				this.periodCountSpinner.setVisible(false);
				this.periodCountLabel.setVisible(false);
				this.periodEndDate.setVisible(false);
			}
			if (e.getSource().equals(this.periodCountEnd))
			{
				this.periodCountSpinner.setValue(1);
				this.periodCountSpinner.setVisible(true);
				this.periodCountLabel.setVisible(true);
				this.periodEndDate.setVisible(false);	
			}
			if (e.getSource().equals(this.periodFixedEnd))
			{
				this.periodCountSpinner.setVisible(false);
				this.periodCountLabel.setVisible(false);
				this.periodEndDate.setVisible(true);
			}
		}
		/* ================================================== */
	}

	/**
	 * @param b
	 */
	private void enablePanel(boolean b) {
		/* ================================================== */
		this.isenabled = b;
		this.summaryLabel.setEnabled(b);
		RecurrencePanel.enableBorder(summaryLabelBorder, b);
		RecurrencePanel.enablePanel(this.repeatPanel, b);
		RecurrencePanel.enableBorder(repeatPanelBorder, b);
		RecurrencePanel.enablePanel(this.periodPanel, b);
		RecurrencePanel.enableBorder(periodPanelBorder, b);
		seperator.setEnabled(b);
		for (AbstractFrequencePanel panel : freqPanels.values()) {
			RecurrencePanel.enablePanel(panel, b);
		}
		if (b)
			applyEnableConfig();
		/* ================================================== */
	}
	
	/**
	 * @param panel
	 * @param enabled
	 */
	public static void enablePanel(JPanel panel, boolean enabled) {
		/* ================================================== */
		panel.setEnabled(enabled);
		Component[] components = panel.getComponents();
		if (components != null && components.length > 0) {
			/* ------------------------------------------------------- */
			int count = components.length;
			for (int i = 0; i < count; i++)
				components[i].setEnabled(enabled);
		      /* ------------------------------------------------------- */
		   }
		   /* ================================================== */
	}
	
	/** 
	 * Enables/Disables the recurrance panels by their identifier
	 * @param key see Appointment.DAILY,WEEKLY,MONTHLY,YEARLY
	 * @param enabled
	 */
	public void enablePanel(int key, boolean enabled) {
		/* ================================================== */
		// store the setting in a hashmap
		/* ------------------------------------------------------- */
		this.enableConfiguration.put(key, enabled);
		/* ------------------------------------------------------- */
		// if the panel is already enabled, we can pipe the command
		// directly to the component
		/* ------------------------------------------------------- */
		if (this.enableCheckBox.isSelected()) {
			/* ------------------------------------------------------- */
			applyEnableConfig();
			/* ------------------------------------------------------- */
		}
		/* ================================================== */
	}
	
	
	private void applyEnableConfig() {
		/* ================================================== */
		boolean first = false;
		/* ------------------------------------------------------- */
		for (int key : this.enableConfiguration.keySet()) {
			/* ------------------------------------------------------- */
			boolean value = enableConfiguration.get(key);
			
			freqPanels.get(key + "") .setEnabled(value);
			JRadioButton button = freqButtons.get(key + "");
			button.setEnabled(value);
			if (!value) 
				button.setSelected(false);
			else
				if (!first) {
					button.setSelected(true);
					first = true;
				}
			
			/* ------------------------------------------------------- */
		}
		/* ================================================== */
	}
	
	
	/**
	 * @param border
	 * @param enabled
	 */
	public static void enableBorder(TitledBorder border, boolean enabled) {
		if (enabled) {
			border.setTitleColor(BorderFactory.createTitledBorder("").getTitleColor());
		} else
			border.setTitleColor(Color.LIGHT_GRAY);
	}

	public void setIsEnabled (boolean isenabled)
	{
		this.isenabled = isenabled;
	}
	
	public boolean getIsEnabled ()
	{
		return this.isenabled;
	}
	/* (non-Javadoc)
	 * @see lu.tudor.santec.gecamed.agenda.gui.widgets.AppointmentEditPanel#setRecurrenceStatus(java.lang.String)
	 */
	@Override
	public void setRecurrenceStatus(String string) {}
	
	
	/**
	 * reset the panel components dayPanel and weekPanel. 
	 */
	public void resetPanel()
	{
		((DailyFreqPanel) this.dayPanel).resetDailyFreqPanel();
		((WeeklyFreqPanel) this.weekPanel).resetWeeklyFreqPanelPanel();
	}
}