/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;
import java.util.Date;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.patient.utils.PatientNameFormatter;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The Activity Bean is a composite view of performed acts, containing also
 * data about the patient act was performed for. Activity Beans are mainly
 * intended for statistics. 
 * @author nico.mack@tudor.lu
  */

@Entity
@Table(name = "activity", schema = "billing")

public class Activity extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;
	
	private Integer		m_InvoiceId;
	private Integer		m_PhysicianId;
	private Integer		m_PatientId;
	private String  	m_PatientFirstName;
	private String		m_PatientMaidenName;
	private String  	m_PatientName;
	private String		m_PatientSSN;
	private String		m_AccessionNumber;
	private Date		m_PerformedDate;
	private String		m_Code;
	private Integer		m_Quantity;
	private Double		m_Coefficient;
	private Double		m_KeyValue;
	private Integer		m_FractionDigit;
	private String		m_Suffixes;

	private Double		m_Majoration;
	private Double		m_BaseValue;
	private Double		m_MonetaryValue;
	
	private static PatientNameFormatter m_Formatter = new PatientNameFormatter ();
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	
//---------------------------------------------------------------------------

public Activity ()
	{
	m_AccessionNumber   	= new String();
	m_PerformedDate 		= new Date ();
	m_Code		   			= new String ();
	m_Quantity	   			= Integer.valueOf (0);
	m_Coefficient  			= new Double (0);
	m_KeyValue				= new Double (0);
	m_FractionDigit			= new Integer (1);
	m_Suffixes				= new String ();
	
	m_Majoration			= Invoice.c_DefaultMajoration;
	m_BaseValue				= new Double (0);
	m_MonetaryValue			= new Double (0);
	}	

//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives				                                              	*
//***************************************************************************	

//***************************************************************************
//* Class Body				                                              	*
//***************************************************************************	

//---------------------------------------------------------------------------
/**
 * Computes and returns the actual monetary value of this activity.
 */
//---------------------------------------------------------------------------

@Transient
public double monetize ()
	{
  	Act	l_Act;
  	
  	l_Act = new Act ();
  	
  	l_Act.setCoefficient	(this.getCoefficient());
  	l_Act.setKeyValue 		(this.getKeyValue(), this.getFractionDigits());
   	
  	this.setBaseValue (l_Act.monetize ());
  	
  	l_Act.setSuffixes		(this.getSuffixes());
  	l_Act.setQuantity		(this.getQuantity());
  	
  	this.setMonetaryValue (l_Act.monetize ());		
  	
  	return m_MonetaryValue;
	}

//---------------------------------------------------------------------------

@Transient
public static void setMaidenNameIsMarriedName (Boolean p_MaidenNameIsMarriedName)
	{
	m_Formatter.setMaidenNameIsMarriedName(p_MaidenNameIsMarriedName);
	}

//---------------------------------------------------------------------------
/**
 * returns the full name of the patient.
 * @return full name of patient.
 */
//---------------------------------------------------------------------------

@Transient
public String getPatientFullName() 
	{
	return m_Formatter.getPatientFullName (m_PatientFirstName, m_PatientMaidenName, m_PatientName);
	}

//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	
//---------------------------------------------------------------------------
/**
 * Returns the ID of the invoice associated with this activity.
 * @return The ID of the invoice associated with this activity.
 */
//---------------------------------------------------------------------------

@Column(name = "invoice_id", updatable=false, insertable=false)
public Integer getInvoiceId() 
	{
	return m_InvoiceId;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified invoice with this activity
 * @param p_InvoiceId specifies the ID of the invoice to be associated with this
 * activity
 */
//---------------------------------------------------------------------------

public void setInvoiceId (Integer p_InvoiceId)
	{
	m_InvoiceId = p_InvoiceId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the ID of the patient associated with this activity.
 * @return The ID of the patient associated with this activity.
 */
//---------------------------------------------------------------------------

@Column(name = "patient_id", insertable =false, updatable = false)

public Integer getPatientId() 
	{
	return m_PatientId;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified patient with this activity
 * @param p_PatientId specifies the ID of the patient to be associated with this
 * activity
 */
//---------------------------------------------------------------------------

public void setPatientId (Integer p_PatientId) 
	{
	m_PatientId = p_PatientId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the ID of the physician associated with this activity
 * @return The ID of the physician associated with this activity
 */
//---------------------------------------------------------------------------

@Column(name = "physician_id", insertable =false, updatable = false)

public Integer getPhysicianId() 
	{
	return m_PhysicianId;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified physician with this activity
 * @param p_PhysicianId specifies the ID of the physician to be associated with this
 * activity
 */
//---------------------------------------------------------------------------

public void setPhysicianId (Integer p_PhysicianId) 
	{
	m_PhysicianId = p_PhysicianId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the first name of the patient this activity was performed on
 * @return first name of patient this activity was performed on
 */
//---------------------------------------------------------------------------

@Column(name = "patient_first_name", insertable =false, updatable = false)

public String getPatientFirstName ()
	{
	return m_PatientFirstName;
	}

//---------------------------------------------------------------------------
/**
 * Sets the first name of the patient this activity was performed on
 * @param p_PatientFirstName specifies the first name of the patient
 */
//---------------------------------------------------------------------------

public void setPatientFirstName (String p_PatientFirstName)
	{
	m_PatientFirstName = p_PatientFirstName;
	}

//---------------------------------------------------------------------------
/**
 * Returns the maiden/married name of the patient associated with this activity 
  * @return maiden/married name of patient this activity is associated with
 */
//---------------------------------------------------------------------------

@Column(name = "patient_maiden_name", insertable =false, updatable = false)

public String getPatientMaidenName ()
	{
	return m_PatientMaidenName;
	}

//---------------------------------------------------------------------------
/**
 * Sets the maiden/married name of the patient associated with this activity
 * @param p_PatientMaidenName specifies the maiden/married name of the patient 
 * this activity is associated with
 */
//---------------------------------------------------------------------------

public void setPatientMaidenName (String p_PatientMaidenName)
	{
	m_PatientMaidenName = p_PatientMaidenName;
	}

//---------------------------------------------------------------------------
/**
 * Returns the name of the patient this activity was performed on
 * @return name of patient this activity was performed on
 */
//---------------------------------------------------------------------------

@Column(name = "patient_name", insertable =false, updatable = false)

public String getPatientName ()
	{
	return m_PatientName;
	}

//---------------------------------------------------------------------------
/**
 * Sets the name of the patient this activity was performed on
 * @param p_PatientName specifies the name of the patient
 */
//---------------------------------------------------------------------------

public void setPatientName (String p_PatientName)
	{
	m_PatientName = p_PatientName;
	}

//---------------------------------------------------------------------------
/**
 * Returns the social security number of the patient this activity was performed on
 * @return social security number of patient this activity was performed on
 */
//---------------------------------------------------------------------------

@Column(name = "patient_ssn", insertable =false, updatable = false)

public String getPatientSSN ()
	{
	return m_PatientSSN;
	}

//---------------------------------------------------------------------------
/**
 * Sets the social security number of the patient this activity was performed on
 * @param p_PatientSSN specifies the social security number of the patient
 */
//---------------------------------------------------------------------------

public void setPatientSSN (String p_PatientSSN)
	{
	m_PatientSSN = p_PatientSSN;
	}

//---------------------------------------------------------------------------
/**
 * Returns the accession number of this activity. An accession numbers is a  unique 
 * number or combination of letters and numbers assigned to each activity in hospitals.
 * @return accession number of activity if defined, <code>NULL</code> otherwise
 */
//---------------------------------------------------------------------------

@Column (name ="accession_number", insertable =false, updatable = false)

public String getAccessionNumber() 
	{
	return m_AccessionNumber;
	}

//---------------------------------------------------------------------------
/**
 * Sets the accession number of this activity
 * @param p_AccessionNumber specifies the accession number for this activity
 */
//---------------------------------------------------------------------------

public void setAccessionNumber (String p_AccessionNumber)
	{
	m_AccessionNumber = p_AccessionNumber;
	}

//---------------------------------------------------------------------------
/**
 * Returns the date and time this activity was performed
 * @return date and time this activity was performed
 */
//---------------------------------------------------------------------------

@Column (name ="performed", insertable =false, updatable = false)

public Date getPerformedDate() 
	{
	return m_PerformedDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the date and time this activity was performed
 * @param p_PerformedDate specifies the date this activity was performed on.
 */
//---------------------------------------------------------------------------

public void setPerformedDate (Date p_PerformedDate) 
	{
	m_PerformedDate = (p_PerformedDate != null) ? new Date (p_PerformedDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the code associated with this activity
 * @return The code of this activity
 */
//---------------------------------------------------------------------------

@Column (name ="code", insertable =false, updatable = false)

public String getCode() 
	{
	return m_Code;
	}

//---------------------------------------------------------------------------
/**
 * Sets the code of this activity
 * @param p_Code specifies the new code to be set
 */
//---------------------------------------------------------------------------

public void setCode (String p_Code) 
	{
	m_Code = p_Code;
	}

//---------------------------------------------------------------------------
/**
 * Returns the number of times this activity was performed
 * @return This activity's quantity
 */
//---------------------------------------------------------------------------

@Column (name ="quantity", insertable =false, updatable = false)

public Integer getQuantity() 
	{
	return m_Quantity;
	}

//---------------------------------------------------------------------------
/**
 * Sets the number of times this activity was performed
 * @param p_Quantity specifies the new quantity
 */
//---------------------------------------------------------------------------

public void setQuantity (Integer p_Quantity) 
	{
	m_Quantity = p_Quantity;
	}

//---------------------------------------------------------------------------
/**
 * Returns the coefficient associated with this activity
 * @return This activity's coefficient
 */
//---------------------------------------------------------------------------

@Column (name ="coefficient", insertable =false, updatable = false)

public Double getCoefficient() 
	{
	return m_Coefficient;
	}

//---------------------------------------------------------------------------
/**
 * Sets this activity's coefficient
 * @param p_Coefficient specifies the new coefficient
 */
//---------------------------------------------------------------------------

public void setCoefficient (Double p_Coefficient) 
	{
	m_Coefficient = p_Coefficient;
	}

//---------------------------------------------------------------------------
/**
 * Returns the key value that was applicable when this activity was performed
 * @return applicable key value
 */
//---------------------------------------------------------------------------

@Column (name = "key_value", insertable =false, updatable = false)

public Double getKeyValue() 
	{
	return m_KeyValue;
	}

//---------------------------------------------------------------------------
/**
 * Sets the key value to be applied on this activity
 * @param p_KeyValue specifies the new key value
 */
//---------------------------------------------------------------------------

public void setKeyValue (Double p_KeyValue) 
	{
	m_KeyValue = p_KeyValue;
	}

//---------------------------------------------------------------------------
/**
* Returns the number of digits to round to
* @return the number of digits to round to
*/
//---------------------------------------------------------------------------

@Column (name = "fraction_digits", insertable =false, updatable = false)

public Integer getFractionDigits () 
	{
	return m_FractionDigit;
	}

//---------------------------------------------------------------------------
/**
* Sets the number of digits to round to
* @param p_KeyValue the number of digits to round to
*/
//---------------------------------------------------------------------------

public void setFractionDigits (Integer p_FractionDigits) 
	{
	m_FractionDigit = p_FractionDigits;
	}

//---------------------------------------------------------------------------
/**
 * Returns the suffixes set for this activity
 * @return the suffixes set for this activity
 */
//---------------------------------------------------------------------------

@Column (name ="suffixes", insertable =false, updatable = false)

public String getSuffixes () 
	{
	return m_Suffixes;
	}

//---------------------------------------------------------------------------
/**
 * Sets the suffixes for this activity
 * @param p_Suffixes the new suffixes to be set for this activity
 */
//---------------------------------------------------------------------------

public void setSuffixes (String p_Suffixes) 
	{
	m_Suffixes = p_Suffixes;
	}

//---------------------------------------------------------------------------
/**
 * Returns the current majoration applicable for this activity
 * @return the applicable majoration
 */
//---------------------------------------------------------------------------

@Column (name ="majoration", insertable =false, updatable = false)

public Double getMajoration() 
	{
	return m_Majoration;
	}

//---------------------------------------------------------------------------
/**
 * Sets the majoration applicable for this activity
 * @param p_Majoration the new majoration to apply for this activity
 */
//---------------------------------------------------------------------------

public void setMajoration (Double p_Majoration) 
	{
	m_Majoration = p_Majoration;
	}

//---------------------------------------------------------------------------
/**
 * Returns the base value of this activity. Base value is no intrinsic property
 * of activity bean, i.e. it must be either explicitely set to be defined or
 * the monetize has be called first.
 * @return the base value of this activity
 */
//---------------------------------------------------------------------------

@Transient
public Double getBaseValue() 
	{
	return m_BaseValue;
	}

//---------------------------------------------------------------------------
/**
 * Explicitely sets the base value of this activity.
 * @param p_BaseValue specifies the new base value for this activity
 */
//---------------------------------------------------------------------------

@Transient
public void setBaseValue (Double p_BaseValue) 
	{
	m_BaseValue = p_BaseValue;
	
	}

//---------------------------------------------------------------------------
/**
 * Returns the monetary value of this activity. Monetary value is no intrinsic property
 * of activity bean, i.e. it must be either explicitely set to be defined or
 * the monetize has be called first.
 * @return the monetary value of this activity
 */
//---------------------------------------------------------------------------

@Transient
public Double getMonetaryValue() 
	{
	return m_MonetaryValue;
	}

//---------------------------------------------------------------------------
/**
 * Explicitely sets the monetary value of this activity.
 * @param p_MonetaryValue specifies the new monetary value for this activity
 */
//---------------------------------------------------------------------------

@Transient
public void setMonetaryValue (Double p_MonetaryValue) 
	{
	m_MonetaryValue = p_MonetaryValue;
	}

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
