/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.Table;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The Suffix class represents the available suffixes for medical acts as defined
 * by the UCM (Union des Caisses de Maladies). Suffixes are idenfied by single
 * letters and represent a multiplication factor for act coefficient they are applied
 * to. Multiplication factors may be greater or lower than 1, i.e. they may increase
 * increase or lower the value of an act. Certain Suffixes have minimum threshold values.
 * If suffix factor results in an act value lower than minimum threshold value, then
 * minimum value should be used.
 * 
 * @author nico.mack@tudor.lu
 * @since 06/06/08
 */

@Entity
@Table(name = "suffix", schema = "billing")

@NamedQueries 
	({
	@NamedQuery(name = Suffix.c_AllSuffixes,    query = "SELECT OBJECT(o) FROM Suffix o"),
	@NamedQuery(name = Suffix.c_SuffixByLetter, query = "SELECT OBJECT(o) FROM Suffix o WHERE UPPER(o.letter) = UPPER(:letter)")
	})

public class Suffix extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;

	private Character	m_Letter;
	private String		m_Meaning;
	private Double		m_Factor;
	private Double		m_Minimum;
	
//***************************************************************************
//* Constants	                                                            *
//***************************************************************************	

	public static final transient String c_AllSuffixes    = "getAllSuffixes";
	public static final transient String c_SuffixByLetter = "getSuffixByLetter";
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	

public Suffix ()
	{
	m_Letter 	= Character.valueOf (' ');
	m_Meaning	= new String ();
	m_Factor  	= new Double (0.0);
	m_Minimum	= new Double (0.0);
	}	

//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	

//---------------------------------------------------------------------------
/**
 * Returns the identifying letter of this suffix
 * @return identifying letter of suffix
 */
//---------------------------------------------------------------------------

@Column (name = "letter")

public Character getLetter() 
	{
	return m_Letter;
	}

//---------------------------------------------------------------------------
/**
 * Sets the identifying letter for this suffix
 * @param p_Letter specifies the new identifying letter for this suffix
 */
//---------------------------------------------------------------------------

public void setLetter (Character p_Letter) 
	{
	m_Letter = p_Letter;
	}

//---------------------------------------------------------------------------
/**
 * Returns a description of this suffix.
 * @return the meaning of this suffix.
 */
//---------------------------------------------------------------------------

@Column (name ="meaning")

public String getMeaning() 
	{
	return m_Meaning;
	}

//---------------------------------------------------------------------------
/**
 * Sets the description for this suffix
 * @param p_Meaning specifies the new description for this suffix
 */
//---------------------------------------------------------------------------

public void setMeaning (String p_Meaning) 
	{
	m_Meaning = p_Meaning;
	}

//---------------------------------------------------------------------------
/**
 * Returns the multiplication factor for this suffix
 * @return This suffixs' multiplication factor
 */
//---------------------------------------------------------------------------

@Column (name ="factor")

public Double getFactor() 
	{
	return m_Factor;
	}

//---------------------------------------------------------------------------
/**
 * Sets this suffixs' multiplication factor
 * @param p_Factor specifies the multiplication factor
 */
//---------------------------------------------------------------------------

public void setFactor (Double p_Factor) 
	{
	m_Factor = p_Factor;
	}

//---------------------------------------------------------------------------
/**
 * Returns the minimum coefficient of this suffix
 * @return This suffixs' minimum coefficient
 */
//---------------------------------------------------------------------------

@Column (name ="minimum")

public Double getMinimum() 
	{
	return m_Minimum;
	}

//---------------------------------------------------------------------------
/**
 * Sets this suffixs' minimum coefficient
 * @param p_Minimum specifies the new minimum coefficient
 */
//---------------------------------------------------------------------------

public void setMinimum (Double p_Minimum) 
	{
	m_Minimum = p_Minimum;
	}

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
