/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;
import java.util.Date;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The TrashedAct bean is intended to hold data of deleted acts. The TrashedAct bean 
 * is an exact clone (property wise) of the Act bean. The only difference is that it 
 * is mapped to a different database table. Doing so, allows deleted acts to be saved
 * for later restore if necessary.
 * @author nico.mack@tudor.lu
 * @since 08/01/11
 */

@Entity
@Table(name = "trashed_act", schema = "billing")

@NamedQueries 
	({
	 @NamedQuery(name = TrashedAct.c_TrashedActsByInvoiceId, 		query = "SELECT OBJECT(o) FROM TrashedAct o WHERE o.invoiceId = :invoiceId ORDER BY o.performedDate"),
	 @NamedQuery(name = TrashedAct.c_TrashedActByAccessionNumber, 	query = "SELECT OBJECT(o) FROM TrashedAct o WHERE o.accessionNumber = :accessionNumber")
	})

public class TrashedAct extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;
	
	private Integer		m_InvoiceId;
	private Integer		m_PhysicianId;
	private String		m_AccessionNumber;
	private Date		m_PerformedDate;
	private Boolean		m_ShowTime;
	private String		m_Code;
	private String		m_Label;
	private Integer		m_Quantity;
	private Double		m_Coefficient;
	private Double		m_OrgCoefficient;
	private Double		m_KeyValue;
	private Integer		m_FractionDigits;
	private Boolean		m_CAT;
	private Boolean  	m_CAC;
	private Boolean  	m_APCM;
	private Boolean  	m_ACM;
	private String		m_Suffixes;
	private String 		m_HospitalisationClass;
	private Double 		m_FixAmount;
	private Double		m_Majoration;
	private Double		m_Amount;
	private String		m_MedPrescCode;
	private String		m_AdditionalActInfo;
	private String		m_Username;
	private Integer		m_Adjustment;
//	private Integer		m_OrderNo;

//***************************************************************************
//* Constants	                                                            *
//***************************************************************************	

	public static final transient String c_TrashedActsByInvoiceId      = "getTrashedActsByInvoiceId";
	public static final transient String c_TrashedActByAccessionNumber = "getTrashedActByAccessionNumber";
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	
//---------------------------------------------------------------------------

public TrashedAct ()
	{
	this.init();
	}	

//---------------------------------------------------------------------------
/**
 * Creates a new instance of TrashAct initialized with the properties of the
 * specified act.
 * @param p_Act specifies the act to get data from to initialize the newly
 * created TrashedAct.
 */
//---------------------------------------------------------------------------

public TrashedAct (Act p_Act)
	{	
	if (p_Act != null) 
		{
		this.setInvoiceId 		(p_Act.getInvoiceId());
		this.setPhysicianId		(p_Act.getPhysicianId());
		this.setAccessionNumber	(p_Act.getAccessionNumber());
		this.setPerformedDate   (p_Act.getPerformedDate());
		this.setShowTime		(p_Act.getShowTime());
		this.setCode			(p_Act.getCode());
		this.setLabel			(p_Act.getLabel());
		this.setQuantity		(p_Act.getQuantity());
		this.setCoefficient		(p_Act.getCoefficient());
		this.setKeyValue		(p_Act.getKeyValue(), p_Act.getFractionDigits());
		this.setCAT				(p_Act.getCAT());
		this.setCAC				(p_Act.getCAC());
		this.setAPCM			(p_Act.getAPCM());
		this.setACM				(p_Act.getACM());
		this.setSuffixes		(p_Act.getSuffixes());
		this.setMajoration		(p_Act.getMajoration());
		this.setOrgCoefficient	(p_Act.getOrgCoefficient());
		this.setAmount			(p_Act.getAmount());
		this.setFixAmount		(p_Act.getFixAmount());
		this.setHospitalisationClass(p_Act.getHospitalisationClass());
		this.setMedPrescCode	(p_Act.getMedPrescCode());
		this.setUsername		(p_Act.getUsername());
		this.setAdditionalActInfo(p_Act.getAdditionalActInfo());
		this.setAdjustment		(p_Act.getAdjustment());
//		this.setOrderNo			(p_Act.getOrderNo());
		}
	else this.init();
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives				                                              	*
//***************************************************************************	
//---------------------------------------------------------------------------

private void init ()
	{
	m_AccessionNumber   	= new String();
	m_PerformedDate 		= new Date ();
	m_ShowTime	   			= Boolean.valueOf (false);
	m_Code		   			= new String ();
	m_Label		   			= new String ();
	m_Quantity	   			= Integer.valueOf (0);
	m_Coefficient  			= new Double (0);
	m_KeyValue				= new Double (0);
	m_FractionDigits		= Integer.valueOf(1);
	m_CAT   				= Boolean.valueOf (false);
	m_CAC	   				= Boolean.valueOf (false);
	m_APCM	   				= Boolean.valueOf (false);
	m_ACM	   				= Boolean.valueOf (false);
	m_Suffixes				= new String ();
	}	

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body				                                              	*
//***************************************************************************	

//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	

//---------------------------------------------------------------------------
/**
 * Returns the ID of the trashed invoice this act is associated with
 * @return The ID of the trashed invoice this act is associated with
 */
//---------------------------------------------------------------------------

@Column(name = "invoice_id", updatable=false, insertable=false)
public Integer getInvoiceId() 
	{
	return m_InvoiceId;
	}

//---------------------------------------------------------------------------
/**
 * Sets the ID of the trashed invoice this act is to be associated with
 * @param p_InvoiceId specifies the trashed invoice to associate this act with
 */
//---------------------------------------------------------------------------

public void setInvoiceId (Integer p_InvoiceId)
	{
	m_InvoiceId = p_InvoiceId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the ID of the physician this act is associated with
 * @return The ID of the physician this act is associated with
 */
//---------------------------------------------------------------------------

@Column(name = "physician_id")
public Integer getPhysicianId() 
	{
	return m_PhysicianId;
	}

//---------------------------------------------------------------------------
/**
 * Sets the ID of the physician this act is to be associated with
 * @param p_PhysicianId specifies the new physician to associate this act with
 */
//---------------------------------------------------------------------------

public void setPhysicianId (Integer p_PhysicianId) 
	{
	m_PhysicianId = p_PhysicianId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the accession number of this act. An accession numbers is a  unique 
 * number or combination of letters and numbers assigned to each act in hospitals.
 * @return accession number of act if defined, <code>NULL</code> otherwise
 */
//---------------------------------------------------------------------------

@Column (name ="accession_number")

public String getAccessionNumber() 
	{
	return m_AccessionNumber;
	}

//---------------------------------------------------------------------------
/**
 * Sets the accession number of this act
 * @param p_AccessionNumber specifies the accession number for this act
 */
//---------------------------------------------------------------------------

public void setAccessionNumber (String p_AccessionNumber)
	{
	m_AccessionNumber = p_AccessionNumber;
	}

//---------------------------------------------------------------------------
/**
 * Returns the date and time this act was performed
 * @return date and time this act was performed
 */
//---------------------------------------------------------------------------

@Column (name ="performed")

public Date getPerformedDate() 
	{
	return m_PerformedDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the date and time this act was performed
 * @param p_PerformedDate specifies the date this act was performed on.
 */
//---------------------------------------------------------------------------

public void setPerformedDate (Date p_PerformedDate) 
	{
	m_PerformedDate = (p_PerformedDate != null) ? new Date (p_PerformedDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the state of the show performed time flag of this act
 * @return <b>true</b> if performed time is to be shown, <b>false</b> otherwise
 */
//---------------------------------------------------------------------------

@Column (name ="show_time")

public Boolean getShowTime() 
	{
	return m_ShowTime;
	}

//---------------------------------------------------------------------------
/**
 * Sets set state of the show performed time flag
 * @param p_ShowTime specifies state to set show performed time flag to
 */
//---------------------------------------------------------------------------

public void setShowTime (Boolean p_ShowTime) 
	{
	m_ShowTime = p_ShowTime;
	}

//---------------------------------------------------------------------------
/**
 * Returns the code associated with this act
 * @return The code of this act
 */
//---------------------------------------------------------------------------

@Column (name ="code")

public String getCode() 
	{
	return m_Code;
	}

//---------------------------------------------------------------------------
/**
 * Sets the code of this act
 * @param p_Code specifies the new code to set
 */
//---------------------------------------------------------------------------

public void setCode (String p_Code) 
	{
	m_Code = p_Code;
	}

//---------------------------------------------------------------------------
/**
 * Returns the descriptive label of this act
 * @return The descriptive label of this act
 */
//---------------------------------------------------------------------------

@Column (name ="label")

public String getLabel() 
	{
	return m_Label;
	}

//---------------------------------------------------------------------------
/**
 * Sets this acts descriptive label
 * @param p_Label specifies the new label
 */
//---------------------------------------------------------------------------

public void setLabel(String p_Label) 
	{
	m_Label = p_Label;
	}

//---------------------------------------------------------------------------
/**
 * Returns the number of times this act was performed
 * @return This acts quantity
 */
//---------------------------------------------------------------------------

@Column (name ="quantity")

public Integer getQuantity() 
	{
	return m_Quantity;
	}

//---------------------------------------------------------------------------
/**
 * Sets the number of times this act was performed
 * @param p_Quantity specifies the new quantity
 */
//---------------------------------------------------------------------------

public void setQuantity (Integer p_Quantity) 
	{
	m_Quantity = p_Quantity;
	}

//---------------------------------------------------------------------------
/**
 * Returns the coefficient associated with this act
 * @return This acts coefficient
 */
//---------------------------------------------------------------------------

@Column (name ="coefficient")

public Double getCoefficient() 
	{
	return m_Coefficient;
	}

//---------------------------------------------------------------------------
/**
 * Sets this acts' coefficient
 * @param p_Coefficient specifies the new coefficient
 */
//---------------------------------------------------------------------------

public void setCoefficient (Double p_Coefficient) 
	{
	m_Coefficient = p_Coefficient;
	}

//---------------------------------------------------------------------------
/**
 * Returns the key value that was applicable when this act was performed
 * @return applicable key value
 */
//---------------------------------------------------------------------------

@Column (name = "key_value")

public Double getKeyValue() 
	{
	return m_KeyValue;
	}

//---------------------------------------------------------------------------
/**
 * This method is required by hibernate to manage the database connection.<br>
 * Please use the method setKeyValue(Double, Integer) instead.
 * 
 * @param p_KeyValue specifies the new key value
 */
//---------------------------------------------------------------------------
@Deprecated
public void setKeyValue (Double p_KeyValue) 
	{
	m_KeyValue = p_KeyValue;
	}

//---------------------------------------------------------------------------
/**
 * Sets the key value to be applied on this act and defines, on which decimal
 * place the amount of this act is rounded to.
 * 
 * @param p_KeyValue The key value for this act
 * @param p_FractionDigits The decimal place to round to
 */
//---------------------------------------------------------------------------
@Transient
public void setKeyValue (Double p_KeyValue, Integer p_FractionDigits) 
	{
	m_KeyValue			= p_KeyValue;
	m_FractionDigits	= p_FractionDigits;
	}

/**
 * @return The decimal place this act is rounded to
 */
@Column(name="fraction_digits")
public Integer getFractionDigits ()
{
	return m_FractionDigits;
}


/**This method is required by hibernate to manage the database connection.<br>
 * Please use the method setKeyValue(Double, Integer) instead.
 * 
 * @param digits The decimal place to round to
 */
@Deprecated
public void setFractionDigits (Integer digits)
{
	this.m_FractionDigits = digits;
}

//---------------------------------------------------------------------------
/**
 * Returns the state of the CAT (Cumulable A plein Tarif) flag of this act
 * @return <b>true</b> if CAT is applicable, <b>false</b> otherwise
 */
//---------------------------------------------------------------------------

@Column (name ="cat")

public Boolean getCAT() 
	{
	return m_CAT;
	}

//---------------------------------------------------------------------------
/**
 * Sets set state of the CAT flag
 * @param p_CAT specifies state to set CAT flag to
 */
//---------------------------------------------------------------------------

public void setCAT(Boolean p_CAT) 
	{
	m_CAT = p_CAT;
	}

//---------------------------------------------------------------------------
/**
 * Returns the state of the CAC (Cumulable Avec Consultation) flag of this act
 * @return <b>true</b> if CAC is applicable, <b>false</b> otherwise
 */
//---------------------------------------------------------------------------

@Column (name ="cac")

public Boolean getCAC () 
	{
	return m_CAC;
	}

//---------------------------------------------------------------------------
/**
 * Sets set state of the CAC flag
 * @param p_CAC specifies state to set CAC flag to
 */
//---------------------------------------------------------------------------

public void setCAC (Boolean p_CAC) 
	{
	m_CAC = p_CAC;
	}

//---------------------------------------------------------------------------
/**
 * Returns the state of the APCM (Accord Prealable du Conseil Medical) flag of this act
 * @return <b>true</b> if APCM is required, <b>false</b> otherwise
 */
//---------------------------------------------------------------------------

@Column (name ="apcm")

public Boolean getAPCM () 
	{
	return m_APCM;
	}

//---------------------------------------------------------------------------
/**
 * Sets set state of the APCM flag
 * @param p_APCM specifies state to set APCM flag to
 */
//---------------------------------------------------------------------------

public void setAPCM (Boolean p_APCM) 
	{
	m_APCM = p_APCM;
	}


//---------------------------------------------------------------------------
/**
 * Returns the state of the ACM (Accord du Conseil Medical) flag of this act
 * @return <b>true</b> if ACM is required, <b>false</b> otherwise
 */
//---------------------------------------------------------------------------

@Column (name ="acm")

public Boolean getACM () 
	{
	return m_ACM;
	}

//---------------------------------------------------------------------------
/**
 * Sets set state of the ACM flag
 * @param p_ACM specifies state to set ACM flag to
 */
//---------------------------------------------------------------------------

public void setACM (Boolean p_ACM) 
	{
	m_ACM = p_ACM;
	}

//---------------------------------------------------------------------------
/**
 * Returns the suffixes set for this act
 * @return the suffixes set for this act
 */
//---------------------------------------------------------------------------

@Column (name ="suffixes")

public String getSuffixes () 
	{
	return m_Suffixes;
	}

//---------------------------------------------------------------------------
/**
 * Sets the suffixes for this act
 * @param p_Suffixes the new suffixes to be set for this act
 */
//---------------------------------------------------------------------------

public void setSuffixes (String p_Suffixes) 
	{
	m_Suffixes = p_Suffixes;
	}

@Column (name = "org_coefficient")
public Double getOrgCoefficient()
{
	return m_OrgCoefficient;
}

public void setOrgCoefficient(Double p_OrgCoefficient)
{
	this.m_OrgCoefficient = p_OrgCoefficient;
}

@Column (name = "hospitalisation_class")
public String getHospitalisationClass()
{
	return m_HospitalisationClass;
}

public void setHospitalisationClass(String p_HospitalisationClass)
{
	this.m_HospitalisationClass = p_HospitalisationClass;
}

@Column (name = "fix_amount")
public Double getFixAmount()
{
	return m_FixAmount;
}

public void setFixAmount(Double p_FixAmount)
{
	this.m_FixAmount = p_FixAmount;
}

@Column (name = "majoration")
public Double getMajoration()
{
	return m_Majoration;
}

public void setMajoration(Double p_Majoration)
{
	this.m_Majoration = p_Majoration;
}

@Column (name = "amount")
public Double getAmount()
{
	return m_Amount;
}


public void setAmount(Double p_Amount)
{
	this.m_Amount = p_Amount;
}


@Column(name = "med_presc_code")
public String getMedPrescCode ()
{
	return m_MedPrescCode;
}


public void setMedPrescCode (String p_MedPrescCode)
{
	m_MedPrescCode = p_MedPrescCode;
}

@Column(name = "additional_act_info")
public String getAdditionalActInfo ()
{
	return m_AdditionalActInfo;
}


public void setAdditionalActInfo (String p_AdditionalActInfo)
{
	m_AdditionalActInfo = p_AdditionalActInfo;
}


@Column (name ="username")
public String getUsername ()
{
	return m_Username;
}


public void setUsername (String p_Username)
{
	m_Username = p_Username;
}

@Column (name ="adjustment")
public Integer getAdjustment ()
{
	return m_Adjustment;
}


public void setAdjustment (Integer p_Adjustment)
{
	m_Adjustment = p_Adjustment;
}


//
//@Column (name = "order_no")
//public Integer getOrderNo()
//{
//	return m_OrderNo;
//}
//
//
//public void setOrderNo(Integer p_OrderNo)
//{
//	this.m_OrderNo = p_OrderNo;
//}

//---------------------------------------------------------------------------
//***************************************************************************
//* Recreation Methods														*
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * The method recreates a new instance of act from this trashed act instance.
 * @return a new act initialized with the properties of this trashed act.
 */
//---------------------------------------------------------------------------

@Transient
public Act recreateAct ()
	{	
	Act	l_Act;
	
	l_Act = new Act ();
	
	l_Act.setInvoiceId			(this.getInvoiceId());
	l_Act.setPhysicianId		(this.getPhysicianId());
	l_Act.setAccessionNumber	(this.getAccessionNumber());
	l_Act.setPerformedDate		(this.getPerformedDate());
	l_Act.setShowTime			(this.getShowTime());
	l_Act.setCode				(this.getCode());
	l_Act.setLabel				(this.getLabel());
	l_Act.setQuantity			(this.getQuantity());
	l_Act.setCoefficient		(this.getCoefficient());
	l_Act.setKeyValue			(this.getKeyValue(), this.getFractionDigits());
	l_Act.setCAT				(this.getCAT());
	l_Act.setCAC				(this.getCAC());
	l_Act.setAPCM				(this.getAPCM());
	l_Act.setACM				(this.getACM());
	l_Act.setSuffixes			(this.getSuffixes());
	l_Act.setOrgCoefficient		(this.getOrgCoefficient());
	l_Act.setMajoration			(this.getMajoration());
	l_Act.setAmount				(this.getAmount());
	l_Act.setFixAmount			(this.getFixAmount());
	l_Act.setMedPrescCode		(this.getMedPrescCode());
	l_Act.setUsername			(this.getUsername());
	l_Act.setAdditionalActInfo	(this.getAdditionalActInfo());
	l_Act.setAdjustment			(this.getAdjustment());
	/* It's OK to use this deprecated method here, 
	 * as giving the invoice as additional parameter,
	 * will additionally check and set the majoration,
	 * but this is done here anyway.
	 */
	l_Act.setHospitalisationClass(this.getHospitalisationClass());
//	l_Act.setOrderNo(this.getOrderNo());

	return l_Act;
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* End of Class															*
//***************************************************************************
//---------------------------------------------------------------------------
}
