/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Hospitalisation;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.HospitalisationClass;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Insurance;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.GecamedUser;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The UntrashedInvoice bean is intended to restore previously delete invoices.
 * The UntrashedInvoice bean is an exact clone (property wise) of the TrashedInvoice 
 * bean. The only difference is that it is mapped to the original invoice table.
 * The UntrashedInvoice bean class is one of the few examples of beans that do not 
 * inherit from GECAMedEntityBean. GECAMedEntityBean classes rely on a sequence 
 * generator to  generate object Ids. When restoring previously deleted invoices from 
 * the trash we'd like to preserve the original invoice Id, thus we're handling invoice 
 * Id just like any other property.
 * @author nico.mack@tudor.lu
 * @since 08/01/14
 */

@Entity
@Table(name = "invoice", schema = "billing")

public class UntrashedInvoice implements Serializable
	{
	private static final long serialVersionUID = 1L;

	private Integer					m_Id;
	
	private Patient  				m_Patient;
	private Physician				m_Physician;
	private Hospitalisation			m_Hospitalisation;
	private HospitalisationClass	m_HospitalisationClass;
	private Insurance				m_HealthInsurance;
	private Insurance				m_ThirdPartyPayer;
	private GecamedUser				m_Closer;
	private Date 					m_ClosureDate;
	private GecamedUser				m_Modifier;
	private Date 					m_ModificationDate;
//	private String					m_InvoiceNumber;
	private Boolean					m_FirstClassRequired;
	private Date 					m_InvoiceDate;
	private Date					m_DueDate;
	private Integer					m_State;
	private Integer					m_OldState;
	private String					m_AccidentNumber;
	private Date					m_AccidentDate;
	private Integer					m_Reminders;
	private Date					m_ReminderDate;
	private Double					m_Amount;
	private Double					m_Payment;
	private Double					m_Deduction;
	private Double					m_Balance;
	private Double					m_Majoration;
	private Settlement				m_Settlement;
	private Date					m_SettlementDate;
	private Statement				m_Statement;
	
	private Set <Act>				m_Acts;
	private Set <Memo>				m_Memos;
	
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************

	private static final transient Integer c_Zero	= Integer.valueOf (0);
		
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	

public UntrashedInvoice ()
	{
	this.init();
	}	

//---------------------------------------------------------------------------
/**
 * Creates a new instance of UntrashedInvoice initialized with the properties of the
 * specified trashed invoice. This is the prefered method of instantiation.
 * @param p_TrashedInvoice specifies the trashed invoice to get data from to 
 * initialize the newly created UntrashedInvoice.
 */
//---------------------------------------------------------------------------

public UntrashedInvoice (TrashedInvoice p_TrashedInvoice)
	{
	Iterator <TrashedAct>	l_TrashedActIterator;
	TrashedAct				l_TrashedAct;
	Set <Act>				l_UntrashedActs;
	Act						l_UntrashedAct;
	
	Iterator <TrashedMemo>	l_TrashedMemoIterator;
	TrashedMemo				l_TrashedMemo;
	Set <Memo>				l_UntrashedMemos;
	Memo					l_UntrashedMemo;
	
	if ((p_TrashedInvoice != null) && (p_TrashedInvoice.getId() != null)) 
		{
		this.setId						(p_TrashedInvoice.getId());
		this.setPatient					(p_TrashedInvoice.getPatient());
		this.setPhysician				(p_TrashedInvoice.getPhysician());
		this.setHospitalisation			(p_TrashedInvoice.getHospitalisation());
		this.setHospitalisationClass	(p_TrashedInvoice.getHospitalisationClass());
		this.setHealthInsurance			(p_TrashedInvoice.getHealthInsurance());
		this.setThirdPartyPayer			(p_TrashedInvoice.getThirdPartyPayer());
		this.setCloser					(p_TrashedInvoice.getCloser());
		this.setClosureDate				(p_TrashedInvoice.getClosureDate());
		this.setModifier				(p_TrashedInvoice.getModifier());
		this.setModificationDate		(p_TrashedInvoice.getModificationDate());
		this.setFirstClassRequired		(p_TrashedInvoice.getFirstClassRequired());
		this.setInvoiceDate				(p_TrashedInvoice.getInvoiceDate());
		this.setDueDate					(p_TrashedInvoice.getDueDate());
		this.setState					(p_TrashedInvoice.getState());
		this.setOldState				(p_TrashedInvoice.getOldState());
		this.setAccidentNumber			(p_TrashedInvoice.getAccidentNumber());
		this.setAccidentDate			(p_TrashedInvoice.getAccidentDate());
		this.setNumberOfReminders		(p_TrashedInvoice.getNumberOfReminders());
		this.setReminderDate			(p_TrashedInvoice.getReminderDate());
		this.setAmount					(p_TrashedInvoice.getAmount());
		this.setPayment					(p_TrashedInvoice.getPayment());
		this.setDeduction				(p_TrashedInvoice.getDeduction());
		this.setBalance					(p_TrashedInvoice.getBalance());
		this.setMajoration				(p_TrashedInvoice.getMajoration());
		this.setSettlement				(p_TrashedInvoice.getSettlement());
		this.setSettlementDate			(p_TrashedInvoice.getSettlementDate());
		this.setStatement				(p_TrashedInvoice.getStatement());

		//====================================================================
		//= Process Acts on invoice
		//====================================================================

		if (p_TrashedInvoice.getNumberOfActs() > 0)
			{
			l_UntrashedActs = new LinkedHashSet <Act> ();
			l_TrashedActIterator = p_TrashedInvoice.getActs().iterator();

			while (l_TrashedActIterator.hasNext())
				{
				l_TrashedAct = l_TrashedActIterator.next();
				l_UntrashedAct = l_TrashedAct.recreateAct();
				l_UntrashedActs.add (l_UntrashedAct);
				}			
			this.setActs (l_UntrashedActs);
			}
		
		//====================================================================
		//= Process Memos attached to invoice
		//====================================================================

		if (p_TrashedInvoice.getNumberOfMemos() > 0)
			{
			l_UntrashedMemos = new LinkedHashSet <Memo> ();
			l_TrashedMemoIterator = p_TrashedInvoice.getMemos().iterator();

			while (l_TrashedMemoIterator.hasNext())
				{
				l_TrashedMemo = l_TrashedMemoIterator.next();
				l_UntrashedMemo = l_TrashedMemo.recreateMemo();
				l_UntrashedMemos.add (l_UntrashedMemo);
				}			
			this.setMemos (l_UntrashedMemos);
			}
		}
	
	else this.init();
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives                                       						*
//***************************************************************************
//---------------------------------------------------------------------------

private void init ()
	{
 	m_FirstClassRequired		= Boolean.valueOf (false);
	m_State						= Integer.valueOf (0);
	m_OldState					= Integer.valueOf (0);
	m_Reminders					= Integer.valueOf (0);
	m_Amount					= new Double (0);
	m_Payment					= new Double (0);
	m_Deduction					= new Double (0);
	m_Balance					= new Double (0);
	m_Majoration				= Invoice.c_DefaultMajoration;
	}

//---------------------------------------------------------------------------
/**
 * the formatInvoiceNumber formates the invoice number either as a short
 * number or as a long number. 
 * @param p_Short specifies whether short format should be used for invoice
 * number or long format. Specify <code>true</code> for short format, <code>
 * false</code> for long format.
 * @return the formatted invoice number.
 * @see #Invoice.formatInvoiceNumber (Integer,Integer,Integer,boolean)
 */
//---------------------------------------------------------------------------

@Transient
public String formatInvoiceNumber (boolean p_Short, boolean p_LeadingZeros)
	{
	Integer l_PhysicianId;
	Integer l_PatientId;
	
	l_PhysicianId = (this.getPhysician() != null)?this.getPhysician().getId():null;
	l_PatientId   = (this.getPatient() != null)  ?this.getPatient().getId():null;
	
	return Invoice.formatInvoiceNumber(this.getId(), 
									   l_PhysicianId, 
									   l_PatientId, 
									   p_Short,
									   p_LeadingZeros);
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body                                                              *
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * Get the object ID.
 * this is the primary key of the mapped db table
 * @return The object ID.
 */
//---------------------------------------------------------------------------
	
@Id
@Column(name = "id")

public Integer getId() 
	{
	return m_Id;
	}

//---------------------------------------------------------------------------
/**
 * Sets the object ID
 * this is the primary key of the mapped db table
 * @param p_Id
 */
//---------------------------------------------------------------------------

public void setId (Integer p_Id) 
	{
	this.m_Id = p_Id;
	}

//---------------------------------------------------------------------------
/**
 * Returns the patient associated with this invoice.
 * @return The patient associated with this invoice.
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.LAZY)
@JoinColumn(name = "patient_id")

public Patient getPatient() 
	{
	return m_Patient;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified patient with this invoice
 * @param p_Patient specifies the patient to be associated with this
 * invoice
 */
//---------------------------------------------------------------------------

public void setPatient (Patient p_Patient) 
	{
	m_Patient = p_Patient;
	}

//---------------------------------------------------------------------------
/**
 * Returns the physician associated with this invoice
 * @return The physician associated with this invoice
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "physician_id")

public Physician getPhysician() 
	{
	return m_Physician;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified physician with this invoice
 * @param p_Physician specifies the physician to be associated with this
 * invoice
 */
//---------------------------------------------------------------------------

public void setPhysician (Physician p_Physician) 
	{
	m_Physician = p_Physician;
	}

//---------------------------------------------------------------------------
/**
 * Returns the hospitalisation associated with this invoice
 * @return The hospitalisation associated with this invoice
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.LAZY)
@JoinColumn(name = "hospitalisation_id")

public Hospitalisation getHospitalisation() 
	{
	return m_Hospitalisation;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified hospitalisation with this invoice
 * @param p_Hospitalisation specifies the hospitalisation to be associated with
 * this invoice
 */
//---------------------------------------------------------------------------

public void setHospitalisation (Hospitalisation p_Hospitalisation) 
	{
	m_Hospitalisation = p_Hospitalisation;
	}


//---------------------------------------------------------------------------
/**
 * Returns the patients' hospitalisation class associated with this invoice
 * @return The patients' hospitalisation class
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "class_id")

public HospitalisationClass getHospitalisationClass() 
	{
	return m_HospitalisationClass;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified hospitalisation class with this invoice
 * @param p_HospitalisationClass specifies the hospitalisation class to be 
 * associated with this invoice
 */
//---------------------------------------------------------------------------

public void setHospitalisationClass (HospitalisationClass p_HospitalisationClass) 
	{
	m_HospitalisationClass = p_HospitalisationClass;
	}

//---------------------------------------------------------------------------
/**
 * Returns the patients' health insurance associated with this invoice
 * @return The patients' health insurance
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "insurance_id")

public Insurance getHealthInsurance() 
	{
	return m_HealthInsurance;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified health insurance with this invoice
 * @param p_HealthInsurance specifies the health insurance to be 
 * associated with this invoice
 */
//---------------------------------------------------------------------------

public void setHealthInsurance (Insurance p_HealthInsurance) 
	{
	m_HealthInsurance = p_HealthInsurance;
	}

//---------------------------------------------------------------------------
/**
 * Returns the third party paying insurance covering this invoices' due amount
 * @return The third party paying insurance associated with this invoice
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "third_party_id")

public Insurance getThirdPartyPayer() 
	{
	return m_ThirdPartyPayer;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified insurance as the third party payer for this invoices'
 * due amount
 * @param p_ThirdPartyPayer specifies the third party paying insurance to be
 * associated with this invoice
 */
//---------------------------------------------------------------------------

public void setThirdPartyPayer (Insurance p_ThirdPartyPayer) 
	{
	m_ThirdPartyPayer = p_ThirdPartyPayer;
	}

//---------------------------------------------------------------------------
/**
 * Returns the user who closed this invoice
 * @return The closing user
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.LAZY)
@JoinColumn(name = "closer_id")

public GecamedUser getCloser() 
	{
	return m_Closer;
	}

//---------------------------------------------------------------------------
/**
 * Sets the user who closed this invoice
  * @param p_Closer specifies the closing user
 */
//---------------------------------------------------------------------------

public void setCloser (GecamedUser p_Closer) 
	{
	m_Closer = p_Closer;
	}

//---------------------------------------------------------------------------
/**
 * Returns this invoices' closure date
 * @return Invoice closure date
 */
//---------------------------------------------------------------------------

@Column(name = "closure_date")

public Date getClosureDate() 
	{
	return m_ClosureDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified date as this invoices' closure date
 * @param p_ClosureDate specifies the new invoice closure date
 */
//---------------------------------------------------------------------------

public void setClosureDate (Date p_ClosureDate) 
	{
	m_ClosureDate = (p_ClosureDate != null) ? new Date (p_ClosureDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the user who last modified this invoice
 * @return The modifying user
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.LAZY)
@JoinColumn(name = "modifier_id")

public GecamedUser getModifier() 
	{
	return m_Modifier;
	}

//---------------------------------------------------------------------------
/**
 * Sets the user who last modified this invoice
  * @param p_Modifier specifies the modifying user
 */
//---------------------------------------------------------------------------

public void setModifier (GecamedUser p_Modifier) 
	{
	m_Modifier = p_Modifier;
	}

//---------------------------------------------------------------------------
/**
 * Returns this invoices' modification date
 * @return Invoice modification date
 */
//---------------------------------------------------------------------------

@Column(name = "modification_date")

public Date getModificationDate() 
	{
	return m_ModificationDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified date as this invoices' modification date
 * @param p_ModificationDate specifies the new invoice modification date
 */
//---------------------------------------------------------------------------

public void setModificationDate (Date p_ModificationDate) 
	{
	m_ModificationDate = (p_ModificationDate != null) ? new Date (p_ModificationDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns this invoices fixed or imported invoice number.
 * @return The invoice number, if available, <code>null</code> otherwise
 */
//---------------------------------------------------------------------------

//@Column(name = "invoice_number")
//
//public String getInvoiceNumber() 
//	{
//	return m_InvoiceNumber;
//	}

//---------------------------------------------------------------------------
/**
 * Sets this invoices' fixed or imported invoice number.
 * @param p_InvoiceNumber specifies the fixed number to be associated 
 * with this invoice
 */
//---------------------------------------------------------------------------

//public void setInvoiceNumber (String p_InvoiceNumber) 
//	{
//	m_InvoiceNumber = p_InvoiceNumber;
//	}

//---------------------------------------------------------------------------
/**
 * Returns whether this invoice refers to a hospitalisation requiring a
 * first class. The requesting physician must provide a certificat.
 * @return TRUE if this invoice refers to a required first class hospitalistion,
 * FALSE otherwise.
 */
//---------------------------------------------------------------------------

@Column (name = "first_class_required")

public Boolean getFirstClassRequired() 
	{
	return m_FirstClassRequired;
	}

//---------------------------------------------------------------------------
/**
 * Specifies whether this invoice refers to a hospitalisation requiring a
 * first class.
 * @param p_FirstClassRequired set to <code>true</code> indicates that the hospitalisation
 * this invoice refers to requires a first class.
 */
//---------------------------------------------------------------------------

public void setFirstClassRequired (Boolean p_FirstClassRequired) 
	{
	m_FirstClassRequired = p_FirstClassRequired;
	}

//---------------------------------------------------------------------------
/**
 * Returns this invoices' creation date
 * @return Invoice creation date
 */
//---------------------------------------------------------------------------

@Column(name = "created")

public Date getInvoiceDate() 
	{
	return m_InvoiceDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified date as this invoices' creation number
 * @param p_InvoiceDate specifies the new invoice creation date
 */
//---------------------------------------------------------------------------

public void setInvoiceDate (Date p_InvoiceDate) 
	{
	m_InvoiceDate = (p_InvoiceDate != null) ? new Date (p_InvoiceDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns this invoices' due date
 * @return Invoice due date
 */
//---------------------------------------------------------------------------

@Column(name = "due")

public Date getDueDate() 
	{
	return m_DueDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified date as this invoices' due number
 * @param p_DueDate specifies the new invoice due date
 */
//---------------------------------------------------------------------------

public void setDueDate (Date p_DueDate) 
	{
	m_DueDate = (p_DueDate != null) ? new Date (p_DueDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the state this invoice is currently in
 * @return The invoice state
 */
//---------------------------------------------------------------------------

@Column(name = "state")

public Integer getState() 
	{
	return m_State;
	}

//---------------------------------------------------------------------------
/**
 * Sets the state this invoice is currently in
 * @param p_State specifies the new invoice state
 */
//---------------------------------------------------------------------------

public void setState (Integer p_State) 
	{
	m_State = p_State;
	}

//---------------------------------------------------------------------------
/**
 * Returns the state this invoice was previously in
 * @return The old invoice state
 */
//---------------------------------------------------------------------------

@Column(name = "old_state")

public Integer getOldState() 
	{
	return m_OldState;
	}

//---------------------------------------------------------------------------
/**
 * Sets the state this invoice was in
 * @param p_OldState specifies the old invoice state
 */
//---------------------------------------------------------------------------

public void setOldState (Integer p_OldState) 
	{
	m_OldState = p_OldState;
	}

//---------------------------------------------------------------------------
/**
 * In case this invoice is related to an accident, the getAccidentNumber method
 * will return the corresponding accident number
 * @return The accident number, if available
 */
//---------------------------------------------------------------------------

@Column(name = "accident_number")

public String getAccidentNumber() 
	{
	return m_AccidentNumber;
	}

//---------------------------------------------------------------------------
/**
 * Sets this invoices' accident number to the one specified
 * @param p_AccidentNumber specifies the accident number to be associated 
 * with this invoice
 */
//---------------------------------------------------------------------------

public void setAccidentNumber (String p_AccidentNumber) 
	{
	m_AccidentNumber = p_AccidentNumber;
	}

//---------------------------------------------------------------------------
/**
 * In case this invoice is related to an accident, the getAccidentDate method
 * will return the date the accident occured
 * @return The accident date, if available
 */
//---------------------------------------------------------------------------

@Column(name = "accident_date")

public Date getAccidentDate() 
	{
	return m_AccidentDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified date as the accident date
 * @param p_AccidentDate specifies the new accident date
 */
//---------------------------------------------------------------------------

public void setAccidentDate (Date p_AccidentDate) 
	{
	m_AccidentDate = (p_AccidentDate != null) ? new Date (p_AccidentDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the number of reminders already emitted for this invoice
 * @return The number of emitted reminders.
 */
//---------------------------------------------------------------------------

@Column(name = "reminders")

public Integer getNumberOfReminders() 
	{
	return m_Reminders;
	}

//---------------------------------------------------------------------------
/**
 * Sets the number of already emitted reminders
 * @param p_Reminders specifies the number of emitted reminders
 */
//---------------------------------------------------------------------------

public void setNumberOfReminders (Integer p_Reminders) 
	{
	m_Reminders = p_Reminders;
	}

//---------------------------------------------------------------------------
/**
 * Returns the date the latest reminder was emitted
 * @return Latest reminder date
 */
//---------------------------------------------------------------------------

@Column(name = "reminder_date")

public Date getReminderDate() 
	{
	return m_ReminderDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the date the latest reminder was emitted
 * @param p_ReminderDate specifies the latest reminder date
 */
//---------------------------------------------------------------------------

public void setReminderDate (Date p_ReminderDate) 
	{
	m_ReminderDate = (p_ReminderDate != null) ? new Date (p_ReminderDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the value this invoice amounts to
 * @return The invoice total amount
 */
//---------------------------------------------------------------------------

@Column(name = "amount")

public Double getAmount() 
	{
	return m_Amount;
	}

//---------------------------------------------------------------------------
/**
 * Sets the value this invoice amounts to.
 * @param p_Amount specifies the total amount to set this invoice to
 */
//---------------------------------------------------------------------------

public void setAmount (Double p_Amount) 
	{
	m_Amount = p_Amount;
	}

//---------------------------------------------------------------------------
/**
 * Returns the already paid amount for this invoice
 * @return The already paid amount
 */
//---------------------------------------------------------------------------

@Column(name = "payment")

public Double getPayment() 
	{
	return m_Payment;
	}

//---------------------------------------------------------------------------
/**
 * Sets the amount already paid for this invoice.
 * @param p_Payment specifies the already paid amount.
 */
//---------------------------------------------------------------------------

public void setPayment (Double p_Payment) 
	{
	m_Payment = p_Payment;
	}

//---------------------------------------------------------------------------
/**
 * Returns an eventual deduction applied to this invoice. Deductions may be
 * applied to invoice to compensate for non-reimbursed amounts.
 * @return The deduced amount
 */
//---------------------------------------------------------------------------

@Column(name = "deduction")

public Double getDeduction() 
	{
	return m_Deduction;
	}

//---------------------------------------------------------------------------
/**
 * Sets the amount to be deduced from this invoice
 * @param p_Deduction specifies the amount to be deduced
 */
//---------------------------------------------------------------------------

public void setDeduction (Double p_Deduction) 
	{
	m_Deduction = p_Deduction;
	}

//---------------------------------------------------------------------------
/**
 * Returns the amount that still needs to be paid to settle this invoice. The
 * balance is normaly obtained by subtracting payment and deduction from total
 * invoice amount.
 * @return The invoice balance
 */
//---------------------------------------------------------------------------

@Column(name = "balance")

public Double getBalance() 
	{
	return m_Balance;
	}

//---------------------------------------------------------------------------
/**
 * Sets the amount that still needs to paid to settle this invoice
 * @param p_Balance specifies the new settlement amount
 */
//---------------------------------------------------------------------------

public void setBalance (Double p_Balance) 
	{
	m_Balance = p_Balance;
	}

//---------------------------------------------------------------------------
/**
 * Depending on the context of the invoice, the total amount of the invoice
 * may be subject to a majoration coefficient, i.e. the total amount of the
 * invoice and thus every position on it will be multiplied it. This method
 * return the value of this coefficient which defaults to 1.
 * @return The invoice balance
 */
//---------------------------------------------------------------------------

@Column(name = "majoration")

public Double getMajoration() 
	{
	return m_Majoration;
	}

//---------------------------------------------------------------------------
/**
 * Specifies the majoration coefficient for this invoice. Majoration coefficient
 * defaults to one. Setting another value entails total amount and thus the amount
 * of every position on the invoice to be multiplied by the new value.
 * @param p_Majoration specifies the new majoration coefficient. Please remember
 * that 66% has to be specified as 1.66.
 */
//---------------------------------------------------------------------------

public void setMajoration (Double p_Majoration) 
	{
	m_Majoration = p_Majoration;
	}

//---------------------------------------------------------------------------
/**
 * Returns this invoice settlement. The returned settlement object holds
 * information about how this invoice was settled. 
 * @return This invoices' settlement, if available.
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "settlement_id")

public Settlement getSettlement() 
	{
	return m_Settlement;
	}

//---------------------------------------------------------------------------
/**
 * Sets this invoices' settlement
 * @param p_Settlement specifies the settlement for this invoice
 */
//---------------------------------------------------------------------------

public void setSettlement (Settlement p_Settlement) 
	{
	m_Settlement = p_Settlement;
	}

//---------------------------------------------------------------------------
/**
 * Returns this invoices' settlement date
 * @return Invoice settlement date
 */
//---------------------------------------------------------------------------

@Column(name = "settlement_date")

public Date getSettlementDate() 
	{
	return m_SettlementDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified date as this invoices' settlement number
 * @param p_SettlementDate specifies the new invoice settlement date
 */
//---------------------------------------------------------------------------

public void setSettlementDate (Date p_SettlementDate) 
	{
	m_SettlementDate = (p_SettlementDate != null) ? new Date (p_SettlementDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the statement this invoice is on
 * @return The statement this invoice is on, if available
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "statement_id")

public Statement getStatement() 
	{
	return m_Statement;
	}

//---------------------------------------------------------------------------
/**
 * Sets the statement this invoice is on
 * @param p_Statement specifies the statement this invoice is on
 */
//---------------------------------------------------------------------------

public void setStatement (Statement p_Statement) 
	{
	m_Statement = p_Statement;
	}

//---------------------------------------------------------------------------

@OneToMany(cascade = javax.persistence.CascadeType.ALL, fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "invoice_id")
@OrderBy("performedDate ASC")
public Set<Act> getActs ()
	{
	return m_Acts;
	}

//---------------------------------------------------------------------------

public void setActs (Set<Act> p_Acts)
	{
	m_Acts = p_Acts;
	}

//---------------------------------------------------------------------------

@OneToMany(cascade = javax.persistence.CascadeType.ALL, fetch = javax.persistence.FetchType.EAGER)
@JoinColumn(name = "invoice_id")
@OrderBy("creationDate ASC")

public Set<Memo> getMemos ()
	{
	return m_Memos;
	}

//---------------------------------------------------------------------------

public void setMemos (Set<Memo> p_Memos)
	{
	m_Memos = p_Memos;
	}

//---------------------------------------------------------------------------
/**
 * Returns the number of acts associated with this invoice
 * @return The number of invoice acts
 */
//---------------------------------------------------------------------------

@Transient
public Integer getNumberOfActs() 
	{
	if (this.getActs() != null) return Integer.valueOf (this.getActs().size());
						  else return c_Zero;
	
	}

//---------------------------------------------------------------------------
/**
 * Returns the number of memos associated with this invoice
 * @return The number of invoice memos
 */
//---------------------------------------------------------------------------

@Transient
public Integer getNumberOfMemos ()
	{
	if (this.getMemos() != null) return Integer.valueOf (this.getMemos().size());
	  else return c_Zero;
	}

//---------------------------------------------------------------------------

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
