/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.session.beans;

import java.util.Collection;
import java.util.LinkedHashSet;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.InvoiceStub;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.SearchInvoiceStubInterface;
import lu.tudor.santec.gecamed.core.ejb.session.beans.GECAMedSessionBean;
import lu.tudor.santec.gecamed.core.utils.querybuilder.HibernateQueryFactory;
import lu.tudor.santec.gecamed.core.utils.querybuilder.WhereClause;

//***************************************************************************
//* Interface Definition and Members                                        *
//***************************************************************************

/**
 * The SearchInvoiceStubBean implements functionality required to search for
 * Invoices. The bean relies on Invoice stubs to improve performance of
 * transfering large numbers of invoices.
 * @author nico.mack@tudor.lu
 */

@Stateless
@Remote (SearchInvoiceStubInterface.class)
public class SearchInvoiceStubBean extends GECAMedSessionBean implements SearchInvoiceStubInterface
	{
	private static final long serialVersionUID = 1L;

//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * Returns the number of invoices matching the specified where clause.
 * @param p_Clause specifies the where clause to use for the search query.
 * @return the number of invoices matching the specified where clause.
 */
//---------------------------------------------------------------------------

@RolesAllowed("gecam")
@SuppressWarnings("unchecked")
public Long getInvoiceStubCountByWhereClause (WhereClause p_Clause) throws Exception
	{
	Query					l_Query;
	String					l_QueryString;

	Long					l_Count;
	
	if (p_Clause == null) return 0L;

	l_QueryString = "SELECT COUNT(o) FROM Invoice o";

	try	{
		l_Query = HibernateQueryFactory.buildQueryFromWhereClause (m_EntityManager,l_QueryString, p_Clause);
		l_Count = (Long) l_Query.getSingleResult();
		}
	catch (NoResultException p_Exception)
		{
		l_Count = 0L;
		}
	
	return l_Count;
	}

//---------------------------------------------------------------------------
/**
 * Returns the invoice stubs matching the specified where clause.
 * @param p_Clause specifies the where clause to use for the search query.
 * @return a collection holding all the invoice stubs for matching the
 * specified where clause.
 */
//---------------------------------------------------------------------------

@RolesAllowed("gecam")
@SuppressWarnings("unchecked")
public Collection <InvoiceStub> getInvoiceStubsByWhereClause (WhereClause p_Clause) throws Exception
	{
	Query						l_Query;
	String						l_QueryString;

	Collection <InvoiceStub>	l_InvoiceStubs;

	l_InvoiceStubs = new LinkedHashSet <InvoiceStub> ();
	
	if (p_Clause == null) return l_InvoiceStubs;

	l_QueryString = "SELECT OBJECT(s) FROM InvoiceStub s WHERE s.id IN " +
					"(SELECT o.id FROM Invoice o $WHERECLAUSE)";

	try	{
		l_Query = HibernateQueryFactory.buildQueryFromNestedWhereClause (m_EntityManager,l_QueryString, p_Clause);
		l_InvoiceStubs = l_Query.getResultList();
		}
	catch (NoResultException p_Exception)
		{
		// Simply Return empty Collection
		}
	
	return l_InvoiceStubs;
	}

//---------------------------------------------------------------------------
/**
 * Returns the last invoice by date, for the specified patient, if any
 * @param p_PatientId specifies the Id of the patient to get the last invoice
 * stub for.
 * @return the latest invoice stub for the specified patient, if an invoice
 * exists, <code>null</code> otherwise.
 */
//---------------------------------------------------------------------------

@RolesAllowed("gecam")
public InvoiceStub getLastInvoiceStubForPatient (Integer p_PatientId) throws Exception
	{
	InvoiceStub					l_InvoiceStub;

	l_InvoiceStub = null;
	
	if (p_PatientId == null) return l_InvoiceStub;

	try	{
		l_InvoiceStub = (InvoiceStub) m_EntityManager.createNamedQuery (InvoiceStub.c_LastInvoiceForPatient)
													  .setParameter(InvoiceStub.c_PatientIdParameter, p_PatientId)
													  .setMaxResults(1)
													  .getSingleResult();
		}
	catch (NoResultException p_Exception)
		{
		}
	
	return l_InvoiceStub;
	}
//***************************************************************************
//* End of Class															*
//***************************************************************************
}
