/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.invoice;

import java.awt.Component;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;

import javax.swing.ButtonGroup;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JToggleButton;
import javax.swing.SwingUtilities;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.EtchedBorder;

import lu.tudor.santec.gecamed.billing.gui.BillingModule;
import lu.tudor.santec.gecamed.billing.utils.InvoiceWorkflow;
import lu.tudor.santec.i18n.Translatrix;

/**
 * @author nmac
 */

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class StateChooser extends JPanel implements  ActionListener,
													 KeyListener
    {     
	private JPopupMenu							m_Popup = null; 
	private ButtonGroup							m_StateGroup  = null;
	private Hashtable <Integer, JToggleButton> 	m_StateLookup = null;
	
	private static final long serialVersionUID = 1L;
 	
	public static final String c_ChosenStatesChanged = "ChosenStatesChanged";
	
	public static final Insets c_Insets = new Insets (2,2,2,2);
	
	public static final boolean c_SingleSelection   = false;
	public static final boolean c_MultipleSelection = true;
	
//***************************************************************************
//* Constants	                                                            *
//***************************************************************************

	public static final int c_States [] = { InvoiceWorkflow.c_OpenState,
											InvoiceWorkflow.c_VerifiedState,
											InvoiceWorkflow.c_ClosedState,
											InvoiceWorkflow.c_PrintedState,
											InvoiceWorkflow.c_RemindedState,
											InvoiceWorkflow.c_PaidState,
											InvoiceWorkflow.c_CanceledState };
												
	public static final String c_Icons [] = { "invoice_open_state.png",
										      "invoice_verified_state.png",
										      "invoice_closed_state.png",
											  "invoice_printed_state.png",
											  "invoice_reminded_state.png",
											  "invoice_paid_state.png",
											  "invoice_canceled_state.png" };
			
	public static final String c_ToolTips [] = { "StateChooser.OPN",
			 									 "StateChooser.VER",
			 									 "StateChooser.CLS",
			 									 "StateChooser.PRN",
			 									 "StateChooser.REM",
			 									 "StateChooser.PAI",
			 									 "StateChooser.CAN" };

													
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

public StateChooser (boolean p_MultipleSelection) 
    {
	this.setLayout(new GridLayout(0,c_States.length));
 	
	if (!p_MultipleSelection) m_StateGroup = new ButtonGroup ();
	
	m_StateLookup  = new Hashtable <Integer, JToggleButton> ();
	this.populateStateChooser();
	
	m_Popup = new JPopupMenu(); 
	m_Popup.setLightWeightPopupEnabled(true);
	m_Popup.add(this);
    }
    
//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************

public void updateUI() 
	{
	super.updateUI();
	setEnabled(isEnabled());

	if (m_Popup != null) SwingUtilities.updateComponentTreeUI(m_Popup);
	}

//---------------------------------------------------------------------------

private void resetButtons ()
	{
	Enumeration <Integer>	l_Keys;
	Integer					l_State;
	JToggleButton			l_Button;
		
	l_Keys = m_StateLookup.keys();	
	
	while (l_Keys.hasMoreElements())
		{
		l_State = l_Keys.nextElement();
		l_Button = m_StateLookup.get(l_State);
		l_Button.setSelected(false);
		}
	}

//---------------------------------------------------------------------------

public void populateStateChooser ()
	{
	EmptyBorder			l_Padding;
	CompoundBorder		l_Border;
	EtchedBorder		l_Frame;
	
	JToggleButton		l_Button;
	ImageIcon			l_Icon;
	int					l_Index;
		
	l_Padding = new EmptyBorder (2,2,2,2);
	l_Frame   = new EtchedBorder (EtchedBorder.LOWERED);	
	l_Border = new CompoundBorder (l_Frame,l_Padding);
	
	for (l_Index = 0; l_Index < c_States.length; l_Index++)
		{
		l_Icon   = new ImageIcon (BillingModule.getIconResource(c_Icons[l_Index]).getImage().getScaledInstance(16,16,Image.SCALE_SMOOTH));
		l_Button = new JToggleButton (l_Icon);
		l_Button.setBorder (l_Border);
		l_Button.setToolTipText (Translatrix.getTranslationString(c_ToolTips[l_Index]));	
		l_Button.addKeyListener(this);
		this.add (l_Button);
		if (m_StateGroup != null) m_StateGroup.add(l_Button);
		m_StateLookup.put (Integer.valueOf (c_States[l_Index]),l_Button);		
		}
	}

//---------------------------------------------------------------------------

public void actionPerformed (ActionEvent p_Event)
	{
	JButton			l_Source;
	JToggleButton	l_Target;
	
	if (p_Event.getSource() instanceof JButton)
		{
		l_Source = (JButton) p_Event.getSource();		
		m_Popup.show (l_Source,0,l_Source.getHeight());
		if (m_Popup.isShowing())
			{
			l_Target = m_StateLookup.get (InvoiceWorkflow.c_OpenState);
			if (l_Target != null) 
				l_Target.requestFocus();
			}
		}
	}

//---------------------------------------------------------------------------

private void processKeyChar (char p_Character)
	{	
	switch (p_Character)    
		{
		case KeyEvent.VK_ENTER  : m_Popup.setVisible(false);
								  this.firePropertyChange(c_ChosenStatesChanged, null, null);
								  break;
								
		case KeyEvent.VK_ESCAPE : m_Popup.setVisible(false);
								  this.firePropertyChange(c_ChosenStatesChanged, null, null);
								  break;
		
		case KeyEvent.VK_DELETE : this.resetButtons ();
		  						  m_Popup.setVisible(false);
		  						  this.firePropertyChange(c_ChosenStatesChanged, null, null);
								  break;
		}							
	}

//---------------------------------------------------------------------------

//***************************************************************************
//* Class Body		                                                       *
//***************************************************************************
//---------------------------------------------------------------------------

public void reset ()
	{
	this.resetButtons();
	}

//---------------------------------------------------------------------------

public void setEnabled (boolean p_EnabledIt)
	{
	Enumeration <Integer>	l_Keys;
	Integer					l_State;
	JToggleButton			l_Button;
		
	if (m_StateLookup == null) return;
	
	l_Keys = m_StateLookup.keys();	
	
	while (l_Keys.hasMoreElements())
		{
		l_State = l_Keys.nextElement();
		l_Button = m_StateLookup.get(l_State);
		l_Button.setEnabled (p_EnabledIt);
		}
	}

//---------------------------------------------------------------------------

public void setChosenStates (Collection <Integer> p_States)
	{
	Iterator <Integer>  	l_StateIterator;
	Integer					l_State;
	JToggleButton			l_Button;
		
	this.resetButtons ();
	
	if (p_States == null) return;
	
	l_StateIterator = p_States.iterator();
	while (l_StateIterator.hasNext())
		{
		l_State = l_StateIterator.next();
		l_Button = m_StateLookup.get(l_State);
		if (l_Button != null) l_Button.setSelected(true);
		}
	}

//---------------------------------------------------------------------------

public void setChosenState (Integer p_State)
	{
	JToggleButton			l_Button;

	this.resetButtons ();
	
	if (p_State == null) return;
	
	l_Button = m_StateLookup.get(p_State);
	if (l_Button != null) l_Button.setSelected(true);
	}

//---------------------------------------------------------------------------

public Collection <Integer> getChosenStates ()
	{
	Collection <Integer> 	l_ChosenStates;
	Enumeration <Integer>	l_Keys;
	Integer					l_State;
	JToggleButton			l_Button;
		
	l_ChosenStates = new HashSet <Integer> ();
	l_Keys = m_StateLookup.keys();	
	
	while (l_Keys.hasMoreElements())
		{
		l_State = l_Keys.nextElement();
		l_Button = m_StateLookup.get(l_State);
		if (l_Button.isSelected()) l_ChosenStates.add(l_State);
		}
	
	return l_ChosenStates;
	}

//---------------------------------------------------------------------------

public Integer getChosenState ()
	{
	Enumeration <Integer>		l_Keys;
	
	Integer						l_State;
	JToggleButton				l_Button;
	Integer						l_ChosenState = null;

	l_Keys = m_StateLookup.keys();	
	
	while (l_Keys.hasMoreElements())
		{
		l_State = l_Keys.nextElement();
		l_Button = m_StateLookup.get(l_State);
		
		if (l_Button.isSelected()) 
			{
			l_ChosenState = l_State;
			break;
			}
		}
		
	return l_ChosenState;
	}

//---------------------------------------------------------------------------

public void popup (Component p_Source, int p_HorizontalOffset, int p_VerticalOffset)
	{
	m_Popup.show (p_Source,p_HorizontalOffset,p_VerticalOffset);
	}

//---------------------------------------------------------------------------

public void hide ()
	{
	m_Popup.setVisible(false);
	}

//---------------------------------------------------------------------------

public void keyPressed(KeyEvent p_Event) 
	{
	if(!p_Event.isConsumed())
		{			
		this.processKeyChar (p_Event.getKeyChar());    
		}		
	}

//---------------------------------------------------------------------------

public void keyReleased(KeyEvent p_Event) 
	{
	// TODO Auto-generated method stub
	}

//---------------------------------------------------------------------------

public void keyTyped(KeyEvent p_Event) 
	{
	// TODO Auto-generated method stub	
	}

//---------------------------------------------------------------------------

//***************************************************************************
//* End of Class                                                            *
//***************************************************************************

}

