/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.rate;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collection;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.border.TitledBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Rate;
import lu.tudor.santec.gecamed.billing.ejb.session.beans.NomenclatureBean;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.NomenclatureInterface;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.SwingWorker;
import lu.tudor.santec.gecamed.core.gui.widgets.GECAMedBaseDialogImpl;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class RateSearchDialog extends GECAMedBaseDialogImpl implements ActionListener,
													  DocumentListener,
													  Relocalizable
{
	private static final long serialVersionUID = 1L;
	
	private JLabel					m_CodeLabel;
	private JLabel					m_AllKeywordsLabel;
	private JLabel					m_ExactPhraseLabel;
	private JLabel					m_AnyKeywordsLabel;
	private JLabel					m_NoneKeywordsLabel;
	private JLabel					m_StatusLabel;
	private JLabel					m_RatesLabel;
	
	private TitledBorder			m_CodePanelBorder;
	private TitledBorder			m_KeywordPanelBorder;
	private TitledBorder			m_RatesTableBorder;
	
	private JButton					m_SearchButton;
//	private JButton					m_OkayButton;
//	private JButton					m_CancelButton;
	
	private JProgressBar 			m_ProgressBar;
	private SwingWorker  			m_SearchWorker;
	private Collection <Rate>		m_SearchResult;
	
//	private JCheckBox	m_CAT;
//	private JCheckBox	m_CAC;
//	private JCheckBox	m_APCM;
//	private JCheckBox	m_ACM;
	
	private RateListBox				m_Rates;
	private NomenclatureInterface 	m_NomenclatureInterface;
	
	private JTextField[]			m_SearchFields;
	private int			 			m_SearchMode;
	private int						m_FoundRates;
	
	private Rate[]					m_ChosenRates;
	
	private static Logger m_Logger = Logger.getLogger ("gecamed.billing.gui.rate.RateSearchDialog");

//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************
	
	private final static int c_NumberOfFields = 5;
	
	private final static int 	c_Code			= 0;
	private final static int 	c_AllKeywords	= 1;
	private final static int 	c_AnyKeywords	= 2;
	private final static int 	c_ExactPhrase	= 3;
	private final static int 	c_NoneKeywords	= 4;
	
	
	private final static String c_Columns			= 		"3dlu,fill:max(200dlu;pref):grow," +
														"3dlu,fill:max(30dlu;pref)," +
														"3dlu,fill:max(30dlu;pref),3dlu";
														
	private final static String c_Rows			=    	"3dlu,fill:pref," +
										 				"3dlu,fill:pref," +
										 				"3dlu,fill:pref," +
										 				"3dlu,fill:pref:grow," + 
										 				"3dlu";

	private final static String c_CodePanelColumns = 		"3dlu,fill:max(30dlu;pref)," +
	 													"3dlu,fill:max(90dlu;pref),3dlu";

	private final static String c_CodePanelRows = 			"3dlu,fill:pref,3dlu";

	private final static String c_KeywordPanelColumns = 	"3dlu,fill:max(30dlu;pref)," +
													 	"3dlu,fill:max(90dlu;pref):grow,3dlu";
	
	private final static String c_KeywordPanelRows = 		"3dlu,fill:pref," +
														"3dlu,fill:pref," + 
														"3dlu,fill:pref," + 
														"3dlu,fill:pref";
	
	private final static String c_ProgressPanelColumns = 	"10dlu,fill:max(70dlu;pref)," +
														"3dlu,center:max(90dlu;pref),10dlu";

	private final static String c_ProgressPanelRows = 		"3dlu,fill:max(10dlu;pref),3dlu";

	private final static String c_RatePanelColumns = 		"3dlu,fill:pref:grow,3dlu";

	private final static String c_RatePanelRows = 			"3dlu,fill:pref:grow,3dlu";
	
//***************************************************************************
//* Constructor                                                             *
//***************************************************************************

public RateSearchDialog (int resultSelectionMode)	
	{	
	super (MainFrame.getInstance(),
			Translatrix.getTranslationString("RateSearchDialog.Title"),
			OK_CANCEL_BUTTON_MODE);
//	this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
	
    CellConstraints	l_Constraints;
    FormLayout		l_Layout;
    JPanel			l_CodePanel;
    JPanel			l_KeywordPanel;
    JPanel			l_ProgressPanel;
    JPanel			l_RatesPanel;
    
    MouseAdapter	l_MouseAdapter;
    KeyAdapter		l_KeyAdapter;
    
    m_SearchMode = c_Code;
    m_FoundRates = 0;
    
    m_CodePanelBorder 		= new TitledBorder ("");
    m_KeywordPanelBorder 	= new TitledBorder ("");
    m_RatesTableBorder  	= new TitledBorder ("");
    
    m_CodeLabel 		= new JLabel ();
    m_CodeLabel.setForeground(GECAMedColors.c_LabelColor);
	m_AllKeywordsLabel	= new JLabel ();
	m_AllKeywordsLabel.setForeground(GECAMedColors.c_LabelColor);
	m_ExactPhraseLabel	= new JLabel ();
	m_ExactPhraseLabel.setForeground(GECAMedColors.c_LabelColor);
	m_AnyKeywordsLabel	= new JLabel ();
	m_AnyKeywordsLabel.setForeground(GECAMedColors.c_LabelColor);
	m_NoneKeywordsLabel	= new JLabel ();
	m_NoneKeywordsLabel.setForeground(GECAMedColors.c_LabelColor);
	m_StatusLabel		= new JLabel ();
	m_StatusLabel.setForeground(GECAMedColors.c_LabelColor);
	m_RatesLabel		= new JLabel ();
	m_RatesLabel.setForeground(GECAMedColors.c_LabelColor);
  	
	m_SearchButton = new JButton ();
	m_SearchButton.addActionListener(this);
//	m_OkayButton   = new JButton ();
//	m_OkayButton.addActionListener(this);
//	m_CancelButton = new JButton ();
//	m_CancelButton.addActionListener(this);
	
//	this.setBackground(GECAMedColors.c_GECAMedBackground);
	this.getContentPane().setBackground(GECAMedColors.c_GECAMedBackground);
	this.relocalize ();

	//------------------------------------------------------------------------
	// Build Code Panel
	//------------------------------------------------------------------------
	
	l_Constraints  = new CellConstraints();
    
	l_CodePanel       = new JPanel ();
	l_CodePanel.setBorder(m_CodePanelBorder);
	l_CodePanel.setOpaque (false);
	    
	l_Layout = new FormLayout(c_CodePanelColumns, c_CodePanelRows);
	l_CodePanel.setLayout (l_Layout);
	
	m_SearchFields = new JTextField [c_NumberOfFields];
	
	m_SearchFields[c_Code] = new JTextField ();
	m_SearchFields[c_Code].addActionListener(this);
	m_SearchFields[c_Code].getDocument().addDocumentListener(this);
	   
    l_CodePanel.add (m_CodeLabel, 			l_Constraints.xywh(2, 2, 1, 1));
    l_CodePanel.add (m_SearchFields[c_Code],   l_Constraints.xywh(4, 2, 1, 1));
	
	//------------------------------------------------------------------------
	// Build Keyword Panel
	//------------------------------------------------------------------------

    l_KeywordPanel       = new JPanel ();
	l_KeywordPanel.setBorder(m_KeywordPanelBorder);
    l_KeywordPanel.setOpaque (false);
	
    l_Layout = new FormLayout(c_KeywordPanelColumns, c_KeywordPanelRows);
    l_KeywordPanel.setLayout (l_Layout);
 
    m_SearchFields[c_AllKeywords]		= new JTextField ();
    m_SearchFields[c_AllKeywords].addActionListener(this);
    m_SearchFields[c_AllKeywords].getDocument().addDocumentListener(this);
    
    m_SearchFields[c_AnyKeywords] 	= new JTextField ();
    m_SearchFields[c_AnyKeywords].addActionListener(this);
    m_SearchFields[c_AnyKeywords].getDocument().addDocumentListener(this);
    
    m_SearchFields[c_ExactPhrase]		= new JTextField ();
    m_SearchFields[c_ExactPhrase].addActionListener(this);
    m_SearchFields[c_ExactPhrase].getDocument().addDocumentListener(this);
    
    m_SearchFields[c_NoneKeywords]	= new JTextField ();
    m_SearchFields[c_NoneKeywords].addActionListener(this);
    m_SearchFields[c_NoneKeywords].getDocument().addDocumentListener(this);
   
	l_KeywordPanel.add (m_AllKeywordsLabel,	l_Constraints.xywh(2, 2, 1, 1));	
	l_KeywordPanel.add (m_AnyKeywordsLabel,	l_Constraints.xywh(2, 4, 1, 1));	
	l_KeywordPanel.add (m_ExactPhraseLabel,	l_Constraints.xywh(2, 6, 1, 1));	
	l_KeywordPanel.add (m_NoneKeywordsLabel,	l_Constraints.xywh(2, 8, 1, 1));	
	
	l_KeywordPanel.add (m_SearchFields[c_AllKeywords], l_Constraints.xywh(4, 2, 1, 1));
	l_KeywordPanel.add (m_SearchFields[c_AnyKeywords], l_Constraints.xywh(4, 4, 1, 1));
	l_KeywordPanel.add (m_SearchFields[c_ExactPhrase],	l_Constraints.xywh(4, 6, 1, 1));
	l_KeywordPanel.add (m_SearchFields[c_NoneKeywords],l_Constraints.xywh(4, 8, 1, 1));

	//------------------------------------------------------------------------
	// Build Progress Panel
	//------------------------------------------------------------------------

	l_ProgressPanel	= new JPanel ();
	l_Layout = new FormLayout(c_ProgressPanelColumns, c_ProgressPanelRows);
	l_ProgressPanel.setLayout (l_Layout);
	l_ProgressPanel.setOpaque (false);
	
	m_ProgressBar = new JProgressBar ();
	m_ProgressBar.setStringPainted(false);
	
	l_ProgressPanel.add (m_ProgressBar,  l_Constraints.xywh(2, 2, 1, 1));
	l_ProgressPanel.add (m_StatusLabel,l_Constraints.xywh(4, 2, 1, 1));

	//------------------------------------------------------------------------
	// Build Rates Panel
	//------------------------------------------------------------------------

	l_RatesPanel		= new JPanel (); 
	l_RatesPanel.setBorder (m_RatesTableBorder);
	l_RatesPanel.setOpaque (false);
	
    l_Layout = new FormLayout(c_RatePanelColumns, c_RatePanelRows);
    l_RatesPanel.setLayout (l_Layout);

    m_Rates = new RateListBox (resultSelectionMode);
    m_Rates.getViewport().setOpaque(false);
    m_Rates.setBackground(GECAMedColors.c_ScrollPaneBackground);

    //-----------------------------------------------------------------------
   	// Setup a MouseAdapter to intercept double-clicks on rates. Double-click
   	// on rate ought to select current rate and dismiss dialog.
    //-----------------------------------------------------------------------
  	
   	l_MouseAdapter = new MouseAdapter()
    		{
    		public void mouseClicked(MouseEvent p_Event)
    			{
    			if (p_Event.getClickCount() == 2)
    				{
    				chooseSelectedRate (); 
    				setVisible (false);
    				}
    			}
    		};

    	//-----------------------------------------------------------------------
    	// Setup a KayAdapter to intercept strikes on ENTER key. ENTER key
    	// should to select current rate and dismiss dialog.
    	//-----------------------------------------------------------------------

    	l_KeyAdapter = new KeyAdapter ()
    		{
    		public void keyPressed(KeyEvent p_Event) 
    			{
    			if(!p_Event.isConsumed())
    				{
    			    if (p_Event.getKeyChar() == KeyEvent.VK_ENTER)    
    					{
    			    		chooseSelectedRate ();
    			    		setVisible (false);
    					}
     			}
    			}   
    		};    		
    		 		
    	m_Rates.addMouseListener (l_MouseAdapter);   
    	m_Rates.addKeyListener   (l_KeyAdapter);
    m_Rates.packColumns();
    	
    	
    l_RatesPanel.add (m_Rates,	l_Constraints.xywh(2, 2, 1, 1));
 	
	l_Layout = new FormLayout(c_Columns, c_Rows);
    mainPanel.setLayout (l_Layout);
		    
    mainPanel.add (l_CodePanel,    l_Constraints.xywh(2, 2, 5, 1));
    mainPanel.add (l_KeywordPanel, l_Constraints.xywh(2, 4, 5, 1));
	
    mainPanel.add (l_ProgressPanel,l_Constraints.xywh(2, 6, 3, 1));
    mainPanel.add (m_SearchButton, l_Constraints.xywh(6, 6, 1, 1));
	
    mainPanel.add (l_RatesPanel,   l_Constraints.xywh(2, 8, 5, 1));
//	mainPanel.add (m_OkayButton,   l_Constraints.xywh(4, 10, 1, 1));
//	mainPanel.add (m_CancelButton, l_Constraints.xywh(6, 10, 1, 1));
	
	
	
	}

//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************

//---------------------------------------------------------------------------
/** overwrites JDialogs setVisible method. Method centers the dialog box relative to
 * screen before setting visibilty to <CODE>true</CODE>.
 * @param p_Visibility specifies whether to show or hide the dialog box. Setting 
 * p_Visibility to <CODE>true</CODE> will show the dialog box, <CODE>false</CODE> 
 * will hide it.
 */    
//---------------------------------------------------------------------------

public void setVisible (boolean p_Visibility)
    {
    Dimension l_ScreenSize;
    Dimension l_DialogSize;
        
    if (p_Visibility)
        {
        l_ScreenSize = Toolkit.getDefaultToolkit().getScreenSize();
//        l_DialogSize = this.getSize();
//
//        
//        this.setLocation((l_ScreenSize.width  - l_DialogSize.width)/2 ,
//                         (l_ScreenSize.height - l_DialogSize.height)/2 ); 
        this.setLocationRelativeTo(MainFrame.getInstance());
        }
    super.setVisible (p_Visibility);
    }    
        
//---------------------------------------------------------------------------

private NomenclatureInterface getNomenclatureInterface ()
	{
	if (m_NomenclatureInterface != null) return m_NomenclatureInterface;

	try {
		m_NomenclatureInterface = (NomenclatureInterface) ManagerFactory.getRemote(NomenclatureBean.class);
//		InitialContext l_Context = new InitialContext();
//		m_NomenclatureInterface = (NomenclatureInterface) l_Context.lookup("NomenclatureBean/remote");
//		l_Context.close();
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}

	return m_NomenclatureInterface;
	}

//---------------------------------------------------------------------------

private Collection <Rate> findRatesByCode (String p_Pattern)
	{
	Collection <Rate>	l_Rates = null;

	m_NomenclatureInterface = getNomenclatureInterface ();
	
	if (m_NomenclatureInterface != null)
		{
		try {
			l_Rates = m_NomenclatureInterface.getRatesStartingWith(p_Pattern);
			}
		catch (Exception p_Exception) 
			{
			m_Logger.warn(p_Exception.getLocalizedMessage());
			}
		}	
	return l_Rates;
	}

//---------------------------------------------------------------------------

private Collection <Rate> findRatesByExactPhrase (String p_Phrase)
	{
	Collection <Rate> l_Rates = null;

	m_NomenclatureInterface = getNomenclatureInterface ();
	
	if (m_NomenclatureInterface != null)
		{
		try {
			l_Rates = m_NomenclatureInterface.getRatesWithExactPhrase(p_Phrase);
			}
		catch (Exception p_Exception) 
			{
			m_Logger.warn(p_Exception.getLocalizedMessage());
			}
		}	
	return l_Rates;
	}

//---------------------------------------------------------------------------

private Collection<Rate> findRatesByKeywords (String[] p_Keywords, int p_SearchMode)
	{
	Collection <Rate> l_Rates = null;

	m_NomenclatureInterface = getNomenclatureInterface ();
	
	if (m_NomenclatureInterface != null)
		{
		try {
			switch (p_SearchMode)
				{
				case c_AllKeywords 	: l_Rates = m_NomenclatureInterface.getRatesWithAllKeywords(p_Keywords);
									  break;
				case c_AnyKeywords 	: l_Rates = m_NomenclatureInterface.getRatesWithAnyKeywords(p_Keywords);
				  					  break;
				case c_NoneKeywords	: l_Rates = m_NomenclatureInterface.getRatesWithNoneKeywords(p_Keywords);
				  					  break;
				}
			}
		catch (Exception p_Exception) 
			{
			m_Logger.warn(p_Exception.getLocalizedMessage());
			}
		}	
	return l_Rates;
	}

//---------------------------------------------------------------------------

Object SearchWorker ()
	{
	this.searchForRates();
	return "";
	}

//---------------------------------------------------------------------------

private synchronized void searchForRates ()
	{
	String 	 l_SearchString  = null;
	String[] l_Keywords		= null;

	m_SearchResult = null;
	
	l_SearchString = m_SearchFields[m_SearchMode].getText();

	switch (m_SearchMode)
		{
		case c_Code			: m_SearchResult = this.findRatesByCode (l_SearchString);
							  break;
		case c_ExactPhrase 	: m_SearchResult = this.findRatesByExactPhrase (l_SearchString);
							  break;
		case c_AllKeywords 	:
		case c_AnyKeywords 	:
		case c_NoneKeywords	: l_Keywords = l_SearchString.split("\\s");
							  m_SearchResult = this.findRatesByKeywords(l_Keywords,m_SearchMode);
							  break;
		}					
	}

//---------------------------------------------------------------------------

private void	updateStatusLabel ()
	{
	if (m_StatusLabel == null) return;
		
	if (m_FoundRates > 0)
		 m_StatusLabel.setText( m_FoundRates + " " + Translatrix.getTranslationString("RateSearchDialog.RatesFound"));
	else m_StatusLabel.setText(Translatrix.getTranslationString("RateSearchDialog.NoRatesFound"));
	}

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************
//---------------------------------------------------------------------------

public void chooseSelectedRate ()
	{
	Rate[]		l_SelectedRates;
	
	l_SelectedRates = m_Rates.getSelectedRates();
	
	if (l_SelectedRates != null)
		{
		m_ChosenRates = l_SelectedRates;
		}
	}

//---------------------------------------------------------------------------

public Rate getChosenRate ()
	{
	if (m_ChosenRates != null && m_ChosenRates.length > 0)
		 return m_ChosenRates[0];
	else return null;
	}


public Rate[] getChosenRates ()
	{
	return m_ChosenRates;
	}

//---------------------------------------------------------------------------
/**
 * Method is part of the Relocalizable interface. The method does everything
 * required to reflect changes of active Locale
 */
//---------------------------------------------------------------------------

public void relocalize() 
	{
	this.setTitle(Translatrix.getTranslationString("RateSearchDialog.Title"));
	
	if (m_CodePanelBorder != null)
		m_CodePanelBorder.setTitle(Translatrix.getTranslationString("RateSearchDialog.CodePanelBorder"));
	
	if (m_KeywordPanelBorder != null)
		m_KeywordPanelBorder.setTitle(Translatrix.getTranslationString("RateSearchDialog.KeywordPanelBorder"));
	
	if (m_RatesTableBorder != null)
		m_RatesTableBorder.setTitle(Translatrix.getTranslationString("RateSearchDialog.RatesTableBorder"));
	
	if (m_CodeLabel != null) 
		m_CodeLabel.setText(Translatrix.getTranslationString("RateSearchDialog.CodeLabel"));
	
	if (m_AllKeywordsLabel != null) 
		m_AllKeywordsLabel.setText(Translatrix.getTranslationString("RateSearchDialog.AllKeywordsLabel"));
	
	if (m_ExactPhraseLabel != null) 
		m_ExactPhraseLabel.setText(Translatrix.getTranslationString("RateSearchDialog.ExactPhraseLabel"));
	
	if (m_AnyKeywordsLabel != null) 
		m_AnyKeywordsLabel.setText(Translatrix.getTranslationString("RateSearchDialog.AnyKeywordsLabel"));
	
	if (m_NoneKeywordsLabel != null) 
		m_NoneKeywordsLabel.setText(Translatrix.getTranslationString("RateSearchDialog.NoneKeywordsLabel"));
	
	if (m_StatusLabel != null) updateStatusLabel ();
	
	if (m_RatesLabel != null) 
		m_RatesLabel.setText(Translatrix.getTranslationString("RateSearchDialog.RatesLabel"));
	
	if (m_SearchButton != null)
		m_SearchButton.setText(Translatrix.getTranslationString("RateSearchDialog.SearchButton"));
	
//	if (m_OkayButton != null)
//		m_OkayButton.setText(Translatrix.getTranslationString("core.ok"));
//	
//	if (m_CancelButton != null)
//		m_CancelButton.setText(Translatrix.getTranslationString("core.cancel"));
	
	}

//---------------------------------------------------------------------------

public void actionPerformed(ActionEvent p_ActionEvent) 
	{
	if ( (p_ActionEvent.getSource() == m_SearchFields[c_Code]) ||
		 (p_ActionEvent.getSource() == m_SearchFields[c_AllKeywords]) ||
		 (p_ActionEvent.getSource() == m_SearchFields[c_AnyKeywords]) ||
		 (p_ActionEvent.getSource() == m_SearchFields[c_ExactPhrase]) ||
		 (p_ActionEvent.getSource() == m_SearchFields[c_NoneKeywords]) ||
	     (p_ActionEvent.getSource() == m_SearchButton) )
		{
		
		
	    //-------------------------------------------------------------------
	    // Search operation as such will be performed in dedicated SwingWorker
	    // thread.
	    //-------------------------------------------------------------------
	    
	    m_SearchWorker = new SwingWorker() 
        		{
	    		public Object construct() 
	    			{
	    			return SearchWorker ();
	    			}
	    		public void start ()
	    			{
	    			m_ProgressBar.setValue (m_ProgressBar.getMinimum());
	    		    m_ProgressBar.setIndeterminate (true);
	    		    m_Rates.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
	    		    super.start();
	    			}
	    		public void finished ()
	    			{
	    			if (m_SearchResult != null) 
	    				{
	    				m_Rates.setRates (m_SearchResult);
	    				m_FoundRates = m_Rates.getRateCount();
	    				updateStatusLabel ();
	    				m_SearchResult = null;
	    				m_Rates.selectFirstRate();
	    				m_Rates.requestFocus();
	    				m_Rates.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	    				}
	    			m_ProgressBar.setIndeterminate (false);
	    			}
	    		public void interrupt ()
	    			{
	    			m_SearchResult = null;
	    			m_ProgressBar.setIndeterminate (false);
	    			super.interrupt();
	    			}
        		};

        m_SearchWorker.start ();  
        
	    //-------------------------------------------------------------------

  
		}
//	else if (p_ActionEvent.getSource() == m_OkayButton)
//		{
//		chooseSelectedRate ();
//		this.setVisible (false);
//		}
//	
//	else if (p_ActionEvent.getSource() == m_CancelButton)
//		{
//		m_Rates.removeAllRates();
//		m_ChosenRate = null;
//		this.setVisible (false);
//		}
	}

@Override
public void okActionCalled()
{
	chooseSelectedRate();
	m_Rates.clearSelection();
	super.okActionCalled();
}


@Override
public void cancelActionCalled()
{
//	m_Rates.removeAllRates();
	m_ChosenRates = null;
	m_Rates.clearSelection();
	super.cancelActionCalled();
}


@Override
public void closeActionCalled()
{
//	m_Rates.removeAllRates();
	m_ChosenRates = null;
	m_Rates.clearSelection();
	super.closeActionCalled();
}

//---------------------------------------------------------------------------
/**
 * Method is part of the DocumentListener interface. T
 */
//---------------------------------------------------------------------------

public void insertUpdate(DocumentEvent p_DocumentEvent) 
	{
	Document	l_Document;
	int		l_SearchField;
	int		l_Field;
	
	l_Document   = p_DocumentEvent.getDocument();
	
	l_Field      	= 0;
	l_SearchField 	= -1;
	
	while ( (l_SearchField < 0) && (l_Field < c_NumberOfFields))
		{
		if (l_Document == m_SearchFields[l_Field].getDocument()) l_SearchField = l_Field;
		l_Field++;
		}
		
	if (l_SearchField != m_SearchMode)
		{
		m_SearchMode = l_SearchField;
		for (l_Field = 0; l_Field < c_NumberOfFields; l_Field++)
			{
			if (l_Field != l_SearchField) m_SearchFields[l_Field].setText("");
			}
		}				
	}

//---------------------------------------------------------------------------

public void removeUpdate(DocumentEvent arg0) {
	// TODO Auto-generated method stub
	
}

public void changedUpdate(DocumentEvent arg0) {
	// TODO Auto-generated method stub
	
}
	
//---------------------------------------------------------------------------	
//***************************************************************************
//* End of Class				                            					*
//***************************************************************************
}
