/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.search;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;

import javax.swing.AbstractButton;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Invoice;
import lu.tudor.santec.gecamed.billing.ejb.entity.beans.InvoiceStub;
import lu.tudor.santec.gecamed.billing.ejb.session.beans.SearchInvoiceStubBean;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.InvoiceInterface;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.SearchInvoiceStubInterface;
import lu.tudor.santec.gecamed.billing.gui.BillingModule;
import lu.tudor.santec.gecamed.billing.gui.event.invoice.InvoiceChangeEvent;
import lu.tudor.santec.gecamed.billing.gui.event.invoice.InvoiceEventDispatcher;
import lu.tudor.santec.gecamed.billing.gui.event.invoice.InvoiceEventSource;
import lu.tudor.santec.gecamed.billing.gui.event.invoice.InvoiceListener;
import lu.tudor.santec.gecamed.billing.gui.invoice.stub.GenericInvoiceStubListBox;
import lu.tudor.santec.gecamed.billing.gui.invoice.stub.InvoiceStubComparator;
import lu.tudor.santec.gecamed.billing.gui.invoice.stub.InvoiceStubUtils;
import lu.tudor.santec.gecamed.billing.gui.print.BatchPrintDialog;
import lu.tudor.santec.gecamed.billing.gui.print.InvoicePrinter;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedTab;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.SwingWorker;
import lu.tudor.santec.gecamed.core.gui.widgets.CheckBoxEditorField;
import lu.tudor.santec.gecamed.core.gui.widgets.print.PrintParameterFetcher;
import lu.tudor.santec.gecamed.core.gui.widgets.searchtableheader.SearchResetAction;
import lu.tudor.santec.gecamed.core.gui.widgets.searchtableheader.SearchableTable;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.core.utils.querybuilder.WhereClause;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class SearchPanel extends GECAMedTab implements 	InvoiceEventSource,
														ActionListener,
														SearchableTable,
														TableModelListener,
														ListSelectionListener,
														Relocalizable
	{
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private GenericInvoiceStubListBox		m_InvoiceListBox;
	private InvoiceStubListModel			m_Invoices;
	private InvoiceStubRenderer				m_InvoiceRenderer;	
//	private InvoiceStubSearchTableHeader	m_SearchHeader;
	
	private JButton							m_SearchButton;
	private JButton							m_PrintListButton;
	private JButton							m_BatchPrintButton;

	private InvoiceSearchDialog				m_SearchDialog;
	private BatchPrintDialog				m_PrintDialog;
	
	private InvoiceStubUtils				m_InvoiceStubUtils;
	
	private SearchInvoiceStubInterface		m_SearchInterface;
	private Invoice							m_Invoice;
	
	private SwingWorker  					m_SearchWorker;
	private InvoiceEventDispatcher			m_InvoiceListeners;
	
	private static final Collection <Integer> m_Dependencies = new ArrayList <Integer> ();
    
	static 	{
    		m_Dependencies.add(InvoiceInterface.c_PatientDependency);
 //   		m_Dependencies.add(InvoiceInterface.c_CloserDependency);
    		m_Dependencies.add(InvoiceInterface.c_ModifierDependency);
	    	}

	//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************
	
	private final static String c_Columns= 	"3dlu,fill:pref:grow,3dlu,fill:max(50dlu;pref)";
	
	private final static String c_Rows=    	"3dlu,fill:max(10dlu;pref)," + 
											"3dlu,fill:max(10dlu;pref)," +
											"3dlu,fill:max(10dlu;pref),fill:pref:grow";

	private static final String c_SearchButton    	 = "SearchPanel.SearchButton";
	private static final String c_SearchTip    	 	 = "SearchPanel.SearchTip";
	private static final String c_PrintListButton    = "SearchPanel.PrintListButton";
	private static final String c_PrintListTip    	 = "SearchPanel.PrintListTip";
	private static final String c_BatchPrintButton   = "SearchPanel.BatchPrintButton";
	private static final String c_BatchPrintTip    	 = "SearchPanel.BatchPrintTip";

	private static final Hashtable <Integer,Integer> m_SortingProperties = new Hashtable <Integer,Integer> ();
	
	static {
			m_SortingProperties.put(InvoiceStubListModel.c_InvoiceNumberColumn, InvoiceStubComparator.c_Id);
			m_SortingProperties.put(InvoiceStubListModel.c_InvoiceDateColumn, 	InvoiceStubComparator.c_InvoiceDate);
			m_SortingProperties.put(InvoiceStubListModel.c_PatientSSNColumn, 	InvoiceStubComparator.c_PatientSSN);
			m_SortingProperties.put(InvoiceStubListModel.c_PatientNameColumn, 	InvoiceStubComparator.c_PatientFullName);
			m_SortingProperties.put(InvoiceStubListModel.c_DueDateColumn, 		InvoiceStubComparator.c_DueDate);
			m_SortingProperties.put(InvoiceStubListModel.c_ReminderDateColumn, 	InvoiceStubComparator.c_ReminderDate);
			m_SortingProperties.put(InvoiceStubListModel.c_AmountColumn, 		InvoiceStubComparator.c_Amount);
			m_SortingProperties.put(InvoiceStubListModel.c_PaymentColumn, 		InvoiceStubComparator.c_Payment);
			m_SortingProperties.put(InvoiceStubListModel.c_DeductionColumn, 	InvoiceStubComparator.c_Deduction);
			m_SortingProperties.put(InvoiceStubListModel.c_BalanceColumn, 		InvoiceStubComparator.c_Balance);
	};

	private static Logger m_Logger = Logger.getLogger (SearchPanel.class.getName());

//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************
//---------------------------------------------------------------------------

public SearchPanel ()
	{
	CellConstraints		l_Constraints;
	FormLayout			l_Layout;
	TableColumn			l_Column;
	TableColumnModel	l_ColumnModel;
	ImageIcon			l_Icon;
	MouseAdapter		l_MouseAdapter;
	
	l_Constraints  	= new CellConstraints();
	l_Layout		= new FormLayout(c_Columns, c_Rows);

	this.setLayout(l_Layout);
	
	this.setOpaque(false);

	m_Invoices = new InvoiceStubListModel ();
	m_Invoices.addTableModelListener(this);
	
	m_Invoices.setInsurances(BillingModule.getInstance().getHealthInsurances());
	m_Invoices.setSettlements(BillingModule.getInstance().getSettlementMethods());
	
	m_InvoiceRenderer = new InvoiceStubRenderer ();
	m_InvoiceListBox = new GenericInvoiceStubListBox (m_Invoices);
	m_InvoiceListBox.setInvoiceStubRenderer(m_InvoiceRenderer);
	
	l_MouseAdapter = new MouseAdapter()
    	{
    	public void mouseClicked(MouseEvent p_Event)
    		{
    		if (p_Event.getClickCount() == 2)
    			{
    			editSelectedInvoice (); 
    			}
    		}
    	};

	m_InvoiceListBox.addMouseListener(l_MouseAdapter);
	
//	m_SearchHeader = new InvoiceStubSearchTableHeader (m_InvoiceListBox.getTable());
//	m_SearchHeader.addPropertyChangeListener(this);  
//	
//	m_InvoiceRenderer.setSearchHeader(m_SearchHeader);
	
    	
	l_ColumnModel = m_InvoiceListBox.getTable().getColumnModel(); 
	
	l_Column = l_ColumnModel.getColumn(InvoiceStubListModel.c_TickedColumn);
	l_Column.setCellEditor(new CheckBoxEditorField());
	
	m_InvoiceListBox.getViewport().setOpaque(false);
	m_InvoiceListBox.setBackground(GECAMedColors.c_ScrollPaneBackground);
 
	m_InvoiceListBox.setSorting(InvoiceStubListModel.c_PatientNameColumn, GenericInvoiceStubListBox.c_Ascending);
	m_InvoiceListBox.addListSelectionListener(this);
	
	l_Icon = BillingModule.getButtonIcon ("search_invoices.png");
	m_SearchButton = new JButton ();
	m_SearchButton.setIcon(l_Icon);
	m_SearchButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_SearchButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_SearchButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
//	m_SearchButton.setEnabled (BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_printReminderLists));
	m_SearchButton.addActionListener(this);
	
	l_Icon = BillingModule.getButtonIcon ("search_print_list.png");
	m_PrintListButton = new JButton ();
	m_PrintListButton.setIcon(l_Icon);
	m_PrintListButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_PrintListButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_PrintListButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
//	m_PrintListButton.setEnabled (BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_printReminderLists));
	m_PrintListButton.addActionListener(this);
	
	l_Icon = BillingModule.getButtonIcon ("search_batch_print.png");
	m_BatchPrintButton = new JButton ();
	m_BatchPrintButton.setIcon(l_Icon);
	m_BatchPrintButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_BatchPrintButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_BatchPrintButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
//	m_BatchPrintButton.setEnabled (BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_printReminderLists));
	m_BatchPrintButton.addActionListener(this);
	this.relocalize();
	
	this.add (m_InvoiceListBox, 	l_Constraints.xywh(2, 2, 1, 6));
	this.add (m_SearchButton, 		l_Constraints.xywh(4, 2, 1, 1));
	this.add (m_PrintListButton, 	l_Constraints.xywh(4, 4, 1, 1));
	this.add (m_BatchPrintButton, 	l_Constraints.xywh(4, 6, 1, 1));

	KeyStroke l_Stroke = KeyStroke.getKeyStroke("ESCAPE");
	
	this.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(l_Stroke,"cancelSearch");
	this.getActionMap().put("cancelSearch", new SearchResetAction (this,"cancelSearch"));		
	
	m_InvoiceListeners = new InvoiceEventDispatcher ();
	}
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives                                       						*
//***************************************************************************
//---------------------------------------------------------------------------

private SearchInvoiceStubInterface getSearchInterface ()
	{
	if (m_SearchInterface != null) return m_SearchInterface;

	try {
		m_SearchInterface = (SearchInvoiceStubInterface) ManagerFactory.getRemote(SearchInvoiceStubBean.class);
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}

	return m_SearchInterface;
	}

//---------------------------------------------------------------------------

private Invoice fetchInvoiceForStub (InvoiceStub p_InvoiceStub)
	{
	if (m_InvoiceStubUtils == null) m_InvoiceStubUtils = new InvoiceStubUtils ();
	
	return m_InvoiceStubUtils.fetchInvoiceForStub(p_InvoiceStub, m_Dependencies);	
	}

//---------------------------------------------------------------------------

private Collection <InvoiceStub> getInvoiceStubs ()
	{
	Collection <InvoiceStub>	l_InvoiceStubs;
	
	l_InvoiceStubs = m_Invoices.getTickedInvoiceStubs();
	if ((l_InvoiceStubs == null) || (l_InvoiceStubs.size() == 0))
		l_InvoiceStubs = m_Invoices.getInvoiceStubs();
	
	return l_InvoiceStubs;
	}

//---------------------------------------------------------------------------

private void reflectSelectionChange () 
	{
	InvoiceStub			l_InvoiceStub;
	InvoiceChangeEvent	l_Event;
	
	l_InvoiceStub = m_InvoiceListBox.getSelectedInvoiceStub();
	if (l_InvoiceStub == null) return;
	
	m_Invoice = this.fetchInvoiceForStub(l_InvoiceStub);
	if (m_Invoice == null) return;
	
    l_Event = new InvoiceChangeEvent (this, InvoiceChangeEvent.c_InvoiceSelectionChanged,m_Invoice);
    m_InvoiceListeners.notifyInvoiceListeners(l_Event);
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Search Worker                                    						*
//***************************************************************************
//---------------------------------------------------------------------------

Object SearchWorker (WhereClause p_Clause)
	{
	this.searchInvoices (p_Clause);
	return "";
	}

//---------------------------------------------------------------------------

private void startSearchWorker (final WhereClause p_Clause)
	{
    m_SearchWorker = new SwingWorker() 
		{
		public Object construct() 
			{
			return SearchWorker (p_Clause);
			}
		public void start ()
			{
			MainFrame.getInstance().setWaitCursor(true);
			//m_MessageDelay.start ();
			super.start();
			}
		public void finished ()
			{
			String []	l_Filler;
		
			//m_MessageDelay.stop ();
			m_InvoiceListBox.packColumns();
			m_InvoiceListBox.revalidate();
			MainFrame.getInstance().setWaitCursor(false);		
			
			l_Filler = new String [1];
			
			if (m_InvoiceListBox.getNumberOfInvoiceStubs() > 1)
				{
				l_Filler[0] = Integer.valueOf(m_InvoiceListBox.getNumberOfInvoiceStubs()).toString();
				MainFrame.getInstance().showMessage(Translatrix.getTranslationString("InvoiceListingPanel.InvoicesFound", l_Filler));
				}
			else if (m_InvoiceListBox.getNumberOfInvoiceStubs() == 1)
				{
				MainFrame.getInstance().showMessage(Translatrix.getTranslationString("InvoiceListingPanel.InvoiceFound"));
				}
			else 
				{
				MainFrame.getInstance().showMessage(Translatrix.getTranslationString("InvoiceListingPanel.NoInvoicesFound"));
				}
			}
		public void interrupt ()
			{
			//m_MessageDelay.stop ();
			MainFrame.getInstance().setWaitCursor(false);
			super.interrupt();
			}
		};

	m_SearchWorker.start ();  
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body                                      						*
//***************************************************************************
//---------------------------------------------------------------------------

@Override
public void preparetoShowup()
{
	reflectSelectionChange();
}

//public void preparetoHide ()
//	{
//	Patient l_Patient;
//
//	if (this.isShowing() && (m_Invoice != null))
//		{
//		l_Patient = m_Invoice.getPatient();
//		if (l_Patient != null) BillingModule.setCurrentPatient (l_Patient);
//		}
//	}

//---------------------------------------------------------------------------

public void addInvoiceListener(InvoiceListener p_Listener) 
	{
	m_InvoiceListeners.addInvoiceListener(p_Listener);
	}

//---------------------------------------------------------------------------

public void removeInvoiceListener(InvoiceListener p_Listener) 
	{
	m_InvoiceListeners.removeInvoiceListener(p_Listener);
	}

//---------------------------------------------------------------------------

public void searchInvoices (WhereClause p_Clause)
	{
	SearchInvoiceStubInterface		l_SearchInterface = null;
	Collection <InvoiceStub>		l_InvoiceStubs;
	long							l_Time;
	long							l_Delta;
		
	l_SearchInterface = getSearchInterface ();
	
	if (l_SearchInterface == null) return;
	
	try {
		l_Time = System.currentTimeMillis();
		
		l_InvoiceStubs = l_SearchInterface.getInvoiceStubsByWhereClause(p_Clause);
		
		l_Delta = System.currentTimeMillis() - l_Time;
		
		m_Logger.info("Search took " + l_Delta + " ms");
	
		m_InvoiceListBox.setInvoiceStubs(l_InvoiceStubs);
		m_InvoiceListBox.selectFirstInvoiceStub();
		m_InvoiceListBox.getTable().requestFocusInWindow();
		
		l_Delta = System.currentTimeMillis() - l_Time;
		
		m_Logger.info("Total time required is " + l_Delta + " ms");
		}
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}
	}

//---------------------------------------------------------------------------

public void searchInvoices ()
	{
	if (m_SearchDialog == null) 
		{
		m_SearchDialog = new InvoiceSearchDialog ();
		m_SearchDialog.pack ();
		}
		
	MainFrame.showDialogCentered (m_SearchDialog);	

	if (m_SearchDialog.wasValidated()) 
		{
		this.startSearchWorker(m_SearchDialog.getWhereClause());
		}
	}

//---------------------------------------------------------------------------

public void reloadInvoices ()
{
	if (m_SearchDialog == null || m_SearchDialog.getWhereClause() == null)
		return;
	
	this.startSearchWorker(m_SearchDialog.getWhereClause());
}

//---------------------------------------------------------------------------

private void editSelectedInvoice ()
	{
	if (m_Invoice != null) BillingModule.getInstance().editInvoice(m_Invoice);
	}


public Invoice getSelectedInvoice ()
	{
	return m_Invoice;
	}

//---------------------------------------------------------------------------

private void handlePrintListButton ()
	{
	InvoicePrinter 			l_Printer;
	int						l_SortingColumn = -1;
	int						l_SortDirection	= 0;
	Collection<InvoiceStub> l_InvoiceStubs;
	HashMap<String, Object>	l_Parameter;
	
	l_SortingColumn = m_InvoiceListBox.getSortColumn();
	l_SortDirection = m_InvoiceListBox.getSortDirection(l_SortingColumn);
		
	if (m_SortingProperties.containsKey(l_SortingColumn))
		 l_SortingColumn = m_SortingProperties.get(l_SortingColumn);
	else l_SortingColumn = -1;
	
	l_InvoiceStubs = m_Invoices.getInvoiceStubs();
	l_Parameter = PrintParameterFetcher.createDefaultPrintParameter(l_InvoiceStubs);
	
	l_Printer = new InvoicePrinter ();
	l_Printer.printInvoiceList(l_InvoiceStubs, l_SortingColumn, l_SortDirection, l_Parameter);
	l_Printer.close();
	}

//---------------------------------------------------------------------------

private void handleBatchPrintButton ()
	{
	int	l_SortingColumn = -1;
	int	l_SortDirection	= 0;
	
	l_SortingColumn = m_InvoiceListBox.getSortColumn();
	l_SortDirection = m_InvoiceListBox.getSortDirection(l_SortingColumn);

	if (m_PrintDialog == null)
		{
		m_PrintDialog = new BatchPrintDialog ();
		m_PrintDialog.pack ();
		}
	
	m_PrintDialog.setInvoiceStubs(this.getInvoiceStubs());
	m_PrintDialog.setSortingCriteria(l_SortingColumn, l_SortDirection);
	
	MainFrame.showDialogCentered (m_PrintDialog);	
	}

//---------------------------------------------------------------------------

public void relocalize() 
	{
	if (m_SearchButton != null)
		{
		m_SearchButton.setText (Translatrix.getTranslationString(c_SearchButton));
		m_SearchButton.setToolTipText (Translatrix.getTranslationString(c_SearchTip));
		}
	
	if (m_PrintListButton != null)
		{
		m_PrintListButton.setText (Translatrix.getTranslationString(c_PrintListButton));
		m_PrintListButton.setToolTipText (Translatrix.getTranslationString(c_PrintListTip));
		}
	
	if (m_BatchPrintButton != null)
		{
		m_BatchPrintButton.setText (Translatrix.getTranslationString(c_BatchPrintButton));
		m_BatchPrintButton.setToolTipText (Translatrix.getTranslationString(c_BatchPrintTip));
		}
	}

//---------------------------------------------------------------------------

public void actionPerformed (ActionEvent p_Event) 
	{
	if (p_Event.getSource().equals(m_SearchButton))
		{
		this.searchInvoices();
		}
	else if (p_Event.getSource().equals(m_PrintListButton))
		{
		this.handlePrintListButton ();
		}
	else if (p_Event.getSource().equals (m_BatchPrintButton))
		{
		this.handleBatchPrintButton();
		}
	}

//---------------------------------------------------------------------------

public void resetSearch() 
	{
	// TODO Auto-generated method stub
	}

//---------------------------------------------------------------------------

public void propertyChange (PropertyChangeEvent p_Event) 
	{
	// TODO Auto-generated method stub
	}

//---------------------------------------------------------------------------

public void tableChanged(TableModelEvent p_Event) 
	{
	int			l_Row;
	int			l_RowCount;
	int			l_SelectedRows [];
	InvoiceStub	l_InvoiceStub;
	boolean		l_IsTicked;
	
	if (p_Event.getType() == TableModelEvent.UPDATE)
		{
		l_RowCount = m_InvoiceListBox.getTable().getSelectedRowCount();
		if (l_RowCount > 1)
			{
			m_Invoices.removeTableModelListener(this);
			
			l_Row = p_Event.getFirstRow();
			l_InvoiceStub = m_Invoices.getInvoiceStubAt(l_Row);
			l_IsTicked  = l_InvoiceStub.getTicked();
			l_SelectedRows = m_InvoiceListBox.getTable().getSelectedRows();	
			
			for (l_Row = 0; l_Row < l_SelectedRows.length; l_Row++)
				{
				l_InvoiceStub = m_InvoiceListBox.getInvoiceStubAt (l_SelectedRows[l_Row]);
				l_InvoiceStub.setTicked(l_IsTicked);
				m_InvoiceListBox.setInvoiceStubAt(l_InvoiceStub, l_SelectedRows[l_Row]);
				}
			
			m_Invoices.addTableModelListener(this);
			}
		}
	}

//---------------------------------------------------------------------------

public void valueChanged (ListSelectionEvent p_Event) 
	{
	if (!p_Event.getValueIsAdjusting())
		{
		this.reflectSelectionChange();
		}
	}

///---------------------------------------------------------------------------
//***************************************************************************
//* End Of Class                                                            *
//***************************************************************************
}
