package lu.tudor.santec.gecamed.billing.gui.statistics;

import java.awt.Component;
import java.awt.Font;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.DefaultTableModel;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Act;
import lu.tudor.santec.gecamed.billing.ejb.session.beans.StatisticsBean;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.StatisticsInterface;
import lu.tudor.santec.gecamed.core.gui.GECAMedLists;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.GECAMedGuiUtils;
import lu.tudor.santec.gecamed.core.gui.utils.LineColorCellRenderer;
import lu.tudor.santec.gecamed.core.gui.widgets.GECAMedBaseDialog;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;

/**
 * @author hermenj
 *
 * @version
 * <br>$Log: PatientActsDialog.java,v $
 */
public class PatientActsDialog extends GECAMedBaseDialog {

	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger.getLogger(PatientActsDialog.class.getName());
	private static final long serialVersionUID = 1L;
	private JScrollPane scroller;
	private JTable table;
	private ActTableModel model;
	private StatisticsInterface stats;
	

	public PatientActsDialog() {
		super(MainFrame.getInstance(), 
				Translatrix.getTranslationString("BillingModule.ShowPatientActs"), 
				GECAMedBaseDialog.CLOSE_BUTTON_MODE);
		
		this.model = new ActTableModel();
		this.table = new JTable(this.model);
		
		this.table.getColumnModel().getColumn(0).setMinWidth(100);
		this.table.getColumnModel().getColumn(0).setMaxWidth(100);
		
		this.table.getColumnModel().getColumn(1).setMinWidth(60);
		this.table.getColumnModel().getColumn(1).setMaxWidth(60);
		
		this.table.getColumnModel().getColumn(2).setMinWidth(60);
		this.table.getColumnModel().getColumn(2).setMaxWidth(60);
		
		this.table.getColumnModel().getColumn(3).setMaxWidth(60);
		
		this.table.getColumnModel().getColumn(4).setMaxWidth(60);
		
		this.table.getColumnModel().getColumn(6).setMaxWidth(100);
		this.table.getColumnModel().getColumn(7).setMaxWidth(60);
		
		LineColorCellRenderer renderer = new LineColorCellRenderer();
		this.table.setDefaultRenderer(String.class, renderer);
		
		LineColorCellRenderer numberRenderer = new LineColorCellRenderer(JLabel.RIGHT);
		this.table.getColumnModel().getColumn(7).setCellRenderer(numberRenderer);
		
		LineColorCellRenderer boldRenderer = new LineColorCellRenderer(0, renderer.getFont().deriveFont(Font.BOLD), false);
		this.table.getColumnModel().getColumn(3).setCellRenderer(boldRenderer);
		
		this.scroller = new JScrollPane(this.table);
		this.addMainPanel(scroller);
		
		this.setSize(800, 500);
		this.setLocationRelativeTo(MainFrame.getInstance());
	}
	
	public void showPatientActs(Patient p) {
		if (p == null) {
			logger.info("Not fetching acts for Patient, no Patient given.");
			return;
		}
		
		setTitle(Translatrix.getTranslationString("BillingModule.ShowPatientActs") + ": " + p.toString());
		
		try {
			stats = (StatisticsInterface) ManagerFactory.getRemote(StatisticsBean.class);
			Collection<Act> acts = stats.getActsForPatient(p);
			this.model.setData(acts);
			
			showCenteredDialog();
		} catch (Exception e) {
			logger.error("Error fetching acts for Patient: " +p, e);
		}
	}
	
	
	public class ActTableModel extends DefaultTableModel {
		private static final long serialVersionUID = 1L;
		private ArrayList<Act> acts = new ArrayList<Act>();
		
		private DateFormat df = new SimpleDateFormat("dd/MM/yyyy");
		private DateFormat tf = new SimpleDateFormat("HH:mm");
		private NumberFormat m_AmountFormat;

		
		private final String[] COLS = {
			Translatrix.getTranslationString("ActListModel.DateHeader"),
			Translatrix.getTranslationString("ActListModel.TimeHeader"),
			Translatrix.getTranslationString("ActListModel.PhysicianHeader"),
			Translatrix.getTranslationString("ActListModel.CodeHeader"),
			Translatrix.getTranslationString("ActListModel.SuffixHeader"),
			Translatrix.getTranslationString("ActListModel.LabelHeader"),
			Translatrix.getTranslationString("ActListModel.AdditionalInfoHeader"),
			Translatrix.getTranslationString("ActListModel.AmountHeader")
		};
		
		private final Class[] COL_CLASSES = {
				String.class,
				String.class,
				String.class,
				String.class,
				String.class,
				String.class,
				String.class,
				Double.class,
			};

		public ActTableModel() {
		    m_AmountFormat = NumberFormat.getInstance(Locale.FRENCH);
		    m_AmountFormat.setMinimumFractionDigits(2);
		    m_AmountFormat.setMaximumFractionDigits(2);
		}
		
		public ActTableModel(Collection<Act> acts) {
			setData(acts);
			fireTableDataChanged();
		}
		
		public void setData(Collection<Act> acts) {
			this.acts = new ArrayList<Act>(acts);
		}

		@Override
		public int getRowCount() {
			try {
				return this.acts.size();				
			} catch (Exception e) {
				return 0;
			}
		}

		@Override
		public int getColumnCount() {
			return COLS.length;
		}
		
		

		@Override
		public String getColumnName(int column) {
			return COLS[column];
		}

		@Override
		public boolean isCellEditable(int row, int column) {
			return false;
		}

		@Override
		public Object getValueAt(int row, int column) {
			
			Act a = acts.get(row);
			if (a == null)
				return null;
			
			switch (column) {
			case 0:
				try {
					return df.format(a.getPerformedDate());
				} catch (Exception e) {
					return "";
				}
			case 1:
				try {
					return tf.format(a.getPerformedDate());
				} catch (Exception e) {
					return "";
				}
			case 2:
				try {
					return (GECAMedLists.getIdMapReference(Physician.class).get(a.getPhysicianId())).getMnemonic();
				} catch (Exception e) {
					return "";
				}
			case 3:
				return a.getCode();
			case 4:
				return a.getSuffixes();
			case 5:
				return a.getLabel();
			case 6:
				return a.getAdditionalActInfo();
			case 7:
				try {
					return m_AmountFormat.format(a.getAmount());
				} catch (Exception e) {
					return "";
				}
			default:
				break;
			}
			return null;
		}

		@Override
		public Class<?> getColumnClass(int columnIndex) {
			return COL_CLASSES[columnIndex];
		}
		
	}
	
}
