/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.ejb.session.interfaces;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import javax.annotation.security.RolesAllowed;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.Template;
import lu.tudor.santec.gecamed.core.ejb.entity.beans.XSLTemplate;
import lu.tudor.santec.gecamed.formeditor.ejb.entity.beans.FormTemplate;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;

/**
 * This SessionBean interface manages the access to the printing templates
 * stores in the database. Templates can be stored per physician or for all users.
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: TemplateManagerInterface.java,v $
 * <br>Revision 1.15  2014-02-11 12:49:59  ferring
 * <br>Template loading modified
 * <br>
 * <br>Revision 1.14  2013-01-22 14:21:32  ferring
 * <br>iReport templates will now be compiled only once until the server is stopped
 * <br>
 * <br>Revision 1.13  2012-07-03 14:17:46  ferring
 * <br>iReport printing improved
 * <br>
 * <br>Revision 1.12  2011-10-27 10:33:18  ferring
 * <br>XSL template management for forms changed
 * <br>
 * <br>Revision 1.11  2010-12-29 09:04:59  ferring
 * <br>Option added to print several invoices of same patient and same physician, using the same iReport on one reminder, instead of using one for each.
 * <br>
 * <br>Revision 1.10  2010-04-08 11:01:20  hermen
 * <br>added option to compile report
 * <br>
 * <br>Revision 1.9  2008-12-22 09:40:38  hermen
 * <br>added downloading of template to template settings
 * <br>
 * <br>Revision 1.8  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.7  2008-01-15 09:29:39  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.6  2007-12-12 17:09:20  mack
 * <br>Added new methods
 * <br>
 * <br>Revision 1.5  2007-12-07 15:51:19  mack
 * <br>Added getTemplateById method
 * <br>
 * <br>Revision 1.4  2007-12-03 09:06:57  hermen
 * <br>updated Javadoc
 * <br>
 *
 */
public interface TemplateManagerInterface {
	
//    /**
//     * retrieves all TemplateType Beans
//     * @return all TemplateType Beans
//     * @throws Exception
//     */
//	@RolesAllowed("gecam")
//    public String[] getTemplateTypes() throws Exception;
	
	/**
	 * @author ferring
	 * returns all templates with the specified type
	 * 
	 * @param p_TemplateType the type of the templates
	 * @return all templates with that type
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public Collection<Template> getTemplatesByType (String p_TemplateType);

     /**
     * gets a Template with a specified ID
      * @param p_TemplateID: ID of the template to fetch
     * @return matching Template Bean
     * @throws Exception
     */
	@RolesAllowed("gecam")
    public Template getTemplateByID (Integer p_TemplateID) throws Exception;

     /**
     * retrieves all Template Beans for the specified type and physician
     * @param templateType wanted type of templates
     * @param physicianID wanted physician id
     * @return matching Template Beans
     * @throws Exception
     */
	@RolesAllowed("gecam")
	public Collection<Template> getTemplatesByTypeAndPhysicianID (String templateType, Integer physicianID) throws Exception;

//    /**
//     * retrieves one single Template Bean for the specified type and physician. Is Obsolete
//     * @param templateType wanted type of template
//     * @param physicianID wanted physician id
//     * @return matching Template Bean
//     * @throws Exception
//     */
//	@RolesAllowed("gecam")
//	public Template findTemplateByTypeAndPhysicianID(String templateType, Integer physicianID) throws Exception;

    /**
     * saves the Template Bean to the db
     * @param template the Bean to save
     * @return
     * @throws Exception
     */
	@RolesAllowed("gecam")
	public Template saveTemplate(Template template) throws Exception;
	
	@RolesAllowed("gecam")
	public FormTemplate saveTemplate(Template template, Integer formTemplateId) throws Exception;

    /**
     * retrieves all Template Beans for the specified physician
     * @param physicianID wanted physician id
     * @return matching Template Beans
     * @throws Exception
     */
	@RolesAllowed("gecam")
    public Collection<Template> getTemplatesByPhysicianID(Integer physicianID) throws Exception;

    /**
     * deletes the Template Bean with given type and physicianID
     * @param physicianID wanted physician id
     * @param type wanted type of templates
     * @throws Exception
     */
	@RolesAllowed("gecam")
	public void deleteTemplate(Integer physicianID, String type) throws Exception;
    
   /**
     * deletes the specified Template Bean
     * @param template specifies the Template Bean to delete
     * @throws Exception
     */
	@RolesAllowed("gecam")
	public void deleteTemplate(Template template) throws Exception;

//    /**
//     * retrieves the Template Bean for the specified type and physician
//     * @param templateType wanted type of templates
//     * @param physicianID wanted physician id
//     * @return matching Template Bean
//     * @throws Exception
//     */
//	@RolesAllowed("gecam")
//    public Template getTemplate(String type, Integer physicianID) throws Exception;

//    /**
//     * retrieves the Template Bean for the specified type, physician and date
//     * @param templateType wanted type of templates
//     * @param physicianID wanted physician id
//     * @param  p_DateOfInterest specifies the date of the data to be printed.
//     * @return matching Template Bean
//     * @throws Exception
//     */
//	@RolesAllowed("gecam")
//    public Template getTemplate(String type, Integer physicianID, Date p_DateOfInterest) throws Exception;
   /**
     * creates and returns a JasperPrint object for the specified JasperReport
     * using the provided HashMap of parameters
     * @param report the report template
     * @param parameters the Hashmap of parameters
     * @return  the filled JasperPrint object
     * @throws Exception
     */
	@RolesAllowed("gecam")
    public JasperPrint createJasperPrint(JasperReport report, HashMap<String, Object> parameters)  throws Exception;

//    /**
//     * retrieves the path to toe default template from the database (table: template_type)
//     * @param templateName
//     * @return
//     */
//	@RolesAllowed("gecam")
//    public String getDefaultTemplatePath(String templateName);

//    /**
//     * compiles the given jasperDesign to a jasperreport object
//     * @param jasperDesign
//     * @return
//     */
//    @RolesAllowed("gecam")
//    public JasperReport compileReport(JasperDesign jasperDesign)  throws Exception;
	
	/**
	 * Loads the report from the database or, if it is a default report, from the pre-compiled map or compiles it,
	 * if it isn't in the map, yet. 
	 * 
     * @param referenceType The type to return the template for
	 * @return The compiled JasperReport 
	 */
	@RolesAllowed("gecam")
	public JasperReport findTemplate (String templateBeanName);
	
	/**
	 * Loads the report from the database or, if it is a default report, from the pre-compiled map or if compiles it,
	 * if it isn't in the map, yet. 
	 * 
     * @param referenceType The type to return the template for
     * @param physicianId The physician to call the available templates for or <code>null</code> if the templates
     * @param dateOfInterest specifies the date of the data to be printed.
	 * @return The compiled JasperReport 
	 */
	@RolesAllowed("gecam")
	public JasperReport findTemplate (String templateBeanName, Integer physicianId, Date dateOfInterest);
    
    /**
     * Returns all XSL templates for a special type. 
     * @param referenceType The type to return the template for
     * @param referenceId The id to define the type more exact (probably will only used for the FormTemplateId)
     * @param physicianId The physician to call the available templates for or <code>null</code> if the templates
     * for the whole office shell be shown.
     * @return
     * @throws Exception
     */
    @RolesAllowed("gecam")
    public List<XSLTemplate> getXSLTemplates (String referenceType, Integer referenceId, Integer physicianId) throws Exception;
    
    /**
     * @param template
     * @return
     * @throws Exception
     */
    @RolesAllowed("gecam")
    public XSLTemplate saveTemplate (XSLTemplate template) throws Exception;

    /**
      * deletes the specified XSLTemplate Bean
      * @param template specifies the XSLTemplate Bean to delete
      * @throws Exception
      */
 	@RolesAllowed("gecam")
 	public void deleteTemplate (XSLTemplate template) throws Exception;
 	
 	/**
     * deletes the specified XSLTemplate Bean
     * @param templateId specifies the XSLTemplate Bean to delete
     * @throws Exception
     */
 	@RolesAllowed("gecam")
 	public void deleteTemplate (Integer templateId) throws Exception;
}
