/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/*
 * CustomFileFilter.java
 *
 * Created on March 8, 2004, 1:46 PM
 */

package lu.tudor.santec.gecamed.core.gui.utils;

import java.io.File;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.filechooser.FileFilter;

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

//---------------------------------------------------------------------------
/** The CustomFileFilter class is a customizable file filter. You may add acceptable
 * extension using the addExtension method. Furtermore, you may specify an
 * internationalized description using the setDesription method.
 * @author nico.mack@tudor.lu
 * @version 1.0.0
 */
//---------------------------------------------------------------------------

public class CustomFileFilter extends FileFilter 
    {
    private Vector <String> m_Extensions;
    private String  		m_Description; 
    
    protected static Pattern 
    
   c_ExtensionPattern = Pattern.compile ("^.*?\\.(\\w+)$",Pattern.CASE_INSENSITIVE);
   
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

//---------------------------------------------------------------------------
/** Creates a new (empty) instance of CustomFileFilter class */    
//---------------------------------------------------------------------------

public CustomFileFilter() 
    {
    super();
    
    m_Extensions = new Vector <String> ();
    }

//---------------------------------------------------------------------------
/** Creates a new instance of CustomFileFilter class for the given extension
 *  with the given description
 * @param p_Extension Specifies this file filter's extension.
 * @param p_Description Specifies this file filter's description. Description 
 * will show up in file
 * chooser dialogs.
 */    
//---------------------------------------------------------------------------

public CustomFileFilter (String p_Extension, String p_Description) 
    {
    super();
    
    m_Extensions  = new Vector <String> ();
    this.addExtension(p_Extension);
    m_Description = p_Description;
    }

//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************

//---------------------------------------------------------------------------
/** Method returns the file extension of the specified file.
 * @param p_File specifies the file to get the file extension of
 */
//---------------------------------------------------------------------------

protected String getExtension (File p_File)
    {
    String  l_Filename;    
    String  l_Extension = null;
    Matcher l_Matcher;
    
    if (p_File != null)
        {
        l_Filename = p_File.getName();    
        
       l_Matcher  = c_ExtensionPattern.matcher (l_Filename);
        
        if (l_Matcher.matches())
            {
            l_Extension = l_Matcher.group (1);
            }    
         }
 
    if (l_Extension == null) l_Extension = new String ("");
    
    return l_Extension;
    }

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************

//---------------------------------------------------------------------------
/** adds a new extension to the list of accepted file extensions
 * @param p_Extension specifies the file extension to be added to the accepted list
 */
//---------------------------------------------------------------------------

public void addExtension (String p_Extension)
    {
    if (p_Extension != null)
        m_Extensions.add (p_Extension);
    }


//---------------------------------------------------------------------------
/** Method is part of the FileFilter interface. Method returns <CODE>true</CODE> if
 * the file extension matches one of the extensions previously added via the
 * addExtension method.
 * @param p_File specifies the file whose extension is to be checked
 * @return <CODE>true</CODE> if file extension matches previously specified ones, <CODE>false</CODE>
 * otherwise
 */
//---------------------------------------------------------------------------

public boolean accept (File p_File)
    {
    String  l_Extension;
    int     l_Index;
    boolean l_Accepted = false; 
    
    if (p_File.isDirectory()) return true;

    l_Extension = getExtension (p_File);
    l_Index     = 0;
    
    while ((l_Index < m_Extensions.size()) && (!l_Accepted))
        {
        l_Accepted = l_Extension.equals (m_Extensions.elementAt(l_Index++));
        }    
            
    return l_Accepted;
    }
 
//---------------------------------------------------------------------------
/** sets the description associated with this file filter
 * @param p_Description The description to be associated with this file filter
 */
//---------------------------------------------------------------------------

public void setDescription (String p_Description)	
    {
    if (p_Description != null) m_Description = p_Description;
    }

//---------------------------------------------------------------------------
/** returns the description associated with this file filter
 * @return the description associated with this file filter
 */
//---------------------------------------------------------------------------

public String getDescription() 
    {
    return m_Description;
    }

//***************************************************************************
//* End of Class                                                            *
//***************************************************************************

}
