/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.gui.widgets.hotkeys;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.PrintJob;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.border.EmptyBorder;

import lu.tudor.santec.gecamed.core.gui.GECAMedAction;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedFonts;
import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.RegistrationDesk;
import lu.tudor.santec.i18n.Translatrix;

import com.jgoodies.forms.builder.ButtonBarBuilder;

/**
 * Dialog to display and print all hotkeys used in the client 
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 *
 * @Version
 * <br>$Log: HotKeyListDialog.java,v $
 * <br>Revision 1.7  2011-03-04 09:53:11  ferring
 * <br>Hotkeys for physicians added.
 * <br>It is now possible to make a hotkey consisting of key + modifiers.
 * <br>
 * <br>Revision 1.6  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.5  2008-01-15 09:29:38  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.4  2007-12-04 14:59:04  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class HotKeyListDialog extends JDialog implements ActionListener {

	private static final long serialVersionUID = 1L;
	private JLabel label;
	private JButton closeButton;
	private JButton printButton;
	private JLabel topLabel;
	private JPanel panel;

	/**
	 * creates the dialog
	 */
	public HotKeyListDialog() {
		super(MainFrame.getInstance(), Translatrix.getTranslationString("core.hotkeys"), true);
		this.setLayout(new BorderLayout(5,5));
		this.setDefaultCloseOperation(DISPOSE_ON_CLOSE);
		
		this.setBackground(GECAMedColors.c_GECAMedBackground);
		
		this.getRootPane().setBorder(new EmptyBorder(10,10,10,10));
		this.getRootPane().setBackground(GECAMedColors.c_GECAMedBackground);
		
		this.panel = new JPanel(new BorderLayout(5,5));
		this.panel.setBackground(GECAMedColors.c_GECAMedBackground);
		this.panel.setBorder(new EmptyBorder(10,10,10,10));
		
		this.topLabel = new JLabel(Translatrix.getTranslationString("core.hotkeys"));
		this.topLabel.setIcon(GECAMedModule.getIcon(GECAMedIconNames.HOTKEYS));
		this.panel.add(this.topLabel, BorderLayout.NORTH);
		
		
		this.label = new JLabel();
		this.label.setOpaque(true);
		this.label.setBackground(Color.WHITE);
		this.label.setFont(GECAMedFonts.TEXTFIELD_FONT);
		this.label.setVerticalAlignment(JLabel.TOP);
		this.label.setText(listHotKeys());
		
		this.closeButton = new JButton(Translatrix.getTranslationString("core.close"));
		this.closeButton.addActionListener(this);
		
		this.printButton = new JButton(Translatrix.getTranslationString("core.print"));
		this.printButton.addActionListener(this);
		
		ButtonBarBuilder bb = new ButtonBarBuilder();
		bb.addGlue();
		bb.addGridded(this.printButton);
		bb.addRelatedGap();
		bb.addGridded(this.closeButton);
		JPanel buttonPanel = bb.getPanel();
		buttonPanel.setBackground(Color.WHITE);
		
		this.panel.add(new JScrollPane(this.label), BorderLayout.CENTER);
		this.add(this.panel, BorderLayout.CENTER);
		this.add(buttonPanel, BorderLayout.SOUTH);
		
		this.setSize(600,700);
		MainFrame.showDialogCentered(this);
		
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		if (e.getSource().equals(this.closeButton))
			this.dispose();
		else if (e.getSource().equals(this.printButton)) {
			Toolkit tk = Toolkit.getDefaultToolkit();
			PrintJob pj = tk.getPrintJob(MainFrame.getInstance(), "", null );
			if ( pj != null ) {
			  Graphics g = pj.getGraphics();
			  this.label.printAll( g );
			  g.dispose();
			  pj.end();
			}
		}
	}
	
	
	/**
	 * @return the list of all hotkeys as formated html table
	 */
	public String listHotKeys() {
		StringBuffer sb = new StringBuffer("<html><body><table>" );
		sb.append("<tr>" +
				"<td><h3>" + Translatrix.getTranslationString( "Hotkeys.Name") +"</h3></td>" +
				"<td><h3>" + Translatrix.getTranslationString("Hotkeys.Shortcut") + "</h3></td>" +
				"<td><h3>" + Translatrix.getTranslationString("Hotkeys.HotKey") + "</h3></td></tr>");
		sb.append("<hr>");
		sb.append("<tr><td><b>" + Translatrix.getTranslationString("core.globalhotkeys") + ":</b></td></tr>" );
		for (Iterator<String> iter = RegistrationDesk.getHotKeyKeys().iterator(); iter.hasNext();) {
			String element = iter.next();
			sb.append("<tr><td>&nbsp;&nbsp;&nbsp;" + ((GECAMedAction) RegistrationDesk.getHotKey(element)).getTranslatedName() +"</td>" +
					"<td>" + "" + "</td>" +
					"<td  align=\"center\">" + element + "</td></tr>");
			sb.append("<tr><td>&nbsp;&nbsp;&nbsp;" + element +"</td>" +
					"<td>" + "" + "</td>" +
					"<td  align=\"center\">" + element + "</td></tr>");
		}
		
		if (MainFrame.getInstance().getModules() != null)
			for (GECAMedModule modul : MainFrame.getInstance().getModules()) {
				sb.append(modul.listHotKeys());
			}
		sb.append("</table>");
		return sb.toString();
	}

}
