/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.dicom.gui;

import java.util.Date;
import java.util.Vector;

import javax.swing.ImageIcon;

import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.dicom.ejb.entity.beans.DCMEntry;
import lu.tudor.santec.gecamed.dicom.ejb.entity.beans.DCMSeries;
import lu.tudor.santec.gecamed.dicom.ejb.entity.beans.DCMStudy;
import lu.tudor.santec.gecamed.dicom.ejb.session.beans.DCMManagerBean;
import lu.tudor.santec.gecamed.dicom.ejb.session.interfaces.DCMManagerInterface;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;

/**
 * Treetablemodel for the patients DICOM history view
 *
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @version
 * <br>$Log: DCMTreeTableModel.java,v $
 * <br>Revision 1.10  2013-12-27 18:09:23  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.9  2013-07-15 06:18:35  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.8  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.7  2008-09-15 10:07:23  hermen
 * <br>fixed import status dialog
 * <br>
 * <br>Revision 1.6  2008-04-08 13:15:32  hermen
 * <br>added dicom entry to medical history
 * <br>
 * <br>Revision 1.5  2008-03-12 10:30:06  hermen
 * <br>changed PatientViewTab to abstract and added getName() method
 * <br>
 * <br>Revision 1.4  2008-03-10 13:50:09  hermen
 * <br>updated preview image handling
 * <br>
 * <br>Revision 1.3  2008-03-05 07:27:39  hermen
 * <br>a lot of improvements and bugfixes
 * <br>
 * <br>Revision 1.2  2008-02-27 09:15:11  hermen
 * <br>removed debugging
 * <br>
 * <br>Revision 1.1  2008-02-27 08:33:43  hermen
 * <br>new with treetablemodel
 * <br>
 * <br>Revision 1.1  2008-02-26 10:45:51  hermen
 * <br>initial checkin of the dicom module
 * <br>
 *
 */
public class DCMTreeTableModel extends AbstractTreeTableModel {

	private static final long serialVersionUID = 1L;
	
	/**
	 * names of the different columns
	 */
	private final static String[] columns = {
		Translatrix.getTranslationString("dcm.study") + "/" + 	Translatrix.getTranslationString("dcm.series"),
		Translatrix.getTranslationString("dcm.description"),
		Translatrix.getTranslationString("dcm.preview")
	};
	
	/**
	 * classes of the different columns
	 */
	@SuppressWarnings("unchecked")
	private final static Class[] columnClasses = {
		String.class,
		String.class,
		ImageIcon.class
	};
	
	
	/**
 	 * static logger for this class
 	 */
 	private static Logger logger = Logger.getLogger(DCMTreeTableModel.class
 			.getName());

 	private DCMManagerInterface dcmManager;
 	
	private Patient patient;

	/**
	 * creates a new model
	 */
	public DCMTreeTableModel() {
		try {
 			dcmManager = (DCMManagerInterface) ManagerFactory.getRemote(DCMManagerBean.class);
		}	catch (Exception p_Exception) 
 		{
 			logger.warn(p_Exception.getLocalizedMessage());
 		}
 	}

	/* (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.TreeTableModel#getColumnCount()
	 */
	public int getColumnCount() {
		return columns.length;
	}
	
	/* (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.AbstractTreeTableModel#getColumnName(int)
	 */
	public String getColumnName(int c) {
		return columns[c];
	}

	/* (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.AbstractTreeTableModel#getColumnClass(int)
	 */
	@Override
	public Class<?> getColumnClass(int column) {
		return columnClasses[column];
	}

	/* (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.AbstractTreeTableModel#isLeaf(java.lang.Object)
	 */
	@Override
	public boolean isLeaf(Object node) {
		return super.isLeaf(node);
	}

	/* (non-Javadoc)
	 * @see javax.swing.tree.TreeModel#getChild(java.lang.Object, int)
	 */
	public Object getChild(Object parent, int index) {
		if (parent instanceof DCMEntry && ((DCMEntry)parent).getChilds() != null)
			return ((DCMEntry)parent).getChilds().get(index);
		return null;
	}

	/* (non-Javadoc)
	 * @see javax.swing.tree.TreeModel#getChildCount(java.lang.Object)
	 */
	public int getChildCount(Object parent) {
		if (parent instanceof DCMEntry) {
			return ((DCMEntry)parent).getChildCount();
		}	
		return 0;
	}

	/* (non-Javadoc)
	 * @see javax.swing.tree.TreeModel#getIndexOfChild(java.lang.Object, java.lang.Object)
	 */
	public int getIndexOfChild(Object parent, Object child) {
		if (parent instanceof DCMEntry && ((DCMEntry)parent).getChilds() != null) {
			Vector<DCMEntry> childs = ((DCMEntry)parent).getChilds();
 			for (int i = 0; i < childs.size(); i++) {
 				if (childs.get(i).equals(child))
 					return i;
			}
		}
		return -1;
	}
	
	/* (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.TreeTableModel#getValueAt(java.lang.Object, int)
	 */
	public Object getValueAt(Object node, int column) {
		if (node instanceof DCMEntry) {
			DCMEntry entry = (DCMEntry) node;
			switch (column) {
			
				case 1:
					// if element is a study
					if (entry.getEntryType() == DCMEntry.STUDY) {
						DCMStudy study = (DCMStudy) entry;
						return "<html><b>" +
							study.getDescription() + "</b><br>";
					// if element is a series
					}else if (entry.getEntryType() == DCMEntry.SERIES) {
						DCMSeries series = (DCMSeries) entry;
						return "<html><b>" +
							((series.getDescription() != null)?series.getDescription():"") + "  " +
							((series.getProtocol() != null)?series.getProtocol():"") + 
							"</b><br>" + 
							((series.getImageInfos() != null)?series.getImageInfos():"") + " " +
							((series.getPhysician() != null)?series.getPhysician().replaceAll("\\^", " "):"");
					}
					return "";
				case 2:
					// if element is a series --> show preview
					if (entry.getEntryType() == DCMEntry.SERIES) {
						DCMSeries series = (DCMSeries) entry;
						return series.getPreviewIcon();
					} else {
						return null;
					}
				default:
					break;
			}
			return node.toString();
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.AbstractTreeTableModel#getRoot()
	 */
	public Object getRoot() {
		return root;
	}

	/**
	 * sets a new Patient to the model
	 * @param p
	 */
	public void setPatient(Patient p) {
		this.patient = p;
		this.reload();
	}
	
	/**
	 * returns the current set patient
	 * @return
	 */
	public Patient getPatient() {
		return this.patient;
	}
	
	/**
	 * reloads the model
	 */
	public void reload() {
		try {
			this.root = new RootNode(dcmManager.getDCMHistoryTree(this.patient.getId()));
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * internal class that represents the root-node of the treetable
	 * 
	 *
	 * @author Johannes Hermen johannes.hermen(at)tudor.lu
	 *
	 * @version
	 * <br>$Log: DCMTreeTableModel.java,v $
	 * <br>Revision 1.10  2013-12-27 18:09:23  donak
	 * <br>Cleanup of imports
	 * <br>
	 * <br>Revision 1.9  2013-07-15 06:18:35  ferring
	 * <br>logging changed
	 * <br>
	 * <br>Revision 1.8  2008-09-25 09:43:07  heinemann
	 * <br>fixed copyrights
	 * <br>
	 * <br>Revision 1.7  2008-09-15 10:07:23  hermen
	 * <br>fixed import status dialog
	 * <br>
	 * <br>Revision 1.6  2008-04-08 13:15:32  hermen
	 * <br>added dicom entry to medical history
	 * <br>
	 * <br>Revision 1.5  2008-03-12 10:30:06  hermen
	 * <br>changed PatientViewTab to abstract and added getName() method
	 * <br>
	 * <br>Revision 1.4  2008-03-10 13:50:09  hermen
	 * <br>updated preview image handling
	 * <br>
	 * <br>Revision 1.3  2008-03-05 07:27:39  hermen
	 * <br>a lot of improvements and bugfixes
	 * <br>
	 * <br>Revision 1.2  2008-02-27 09:15:11  hermen
	 * <br>removed debugging
	 * <br>
	 * <br>Revision 1.1  2008-02-27 08:33:43  hermen
	 * <br>new with treetablemodel
	 * <br>
	 *
	 */
	class RootNode implements DCMEntry {
		
		private Vector<DCMEntry> childs;

		/**
		 * @param childs
		 */
		public RootNode(Vector<DCMEntry> childs) {
			this.setChilds(childs);
		}

		/* (non-Javadoc)
		 * @see lu.tudor.santec.gecamed.dicom.ejb.entity.beans.DCMEntry#getChildCount()
		 */
		public int getChildCount() {
			if (this.childs != null) {
				return this.childs.size();
			}
			System.out.println(0);
			return 0;
		}

		/* (non-Javadoc)
		 * @see lu.tudor.santec.gecamed.dicom.ejb.entity.beans.DCMEntry#getChilds()
		 */
		public Vector<DCMEntry> getChilds() {
			return this.childs;
		}

		public void setChilds(Vector<DCMEntry> childs) {
			this.childs = childs;
		}

		/* (non-Javadoc)
		 * @see lu.tudor.santec.gecamed.dicom.ejb.entity.beans.DCMEntry#getDate()
		 */
		public Date getDate() {
			return null;
		}

		/* (non-Javadoc)
		 * @see lu.tudor.santec.gecamed.dicom.ejb.entity.beans.DCMEntry#getEntryType()
		 */
		public int getEntryType() {
			return -1;
		}

		public Integer getIncidentEntryID() {
		    // TODO Auto-generated method stub
		    return null;
		}

	}
	
}
