package lu.tudor.santec.gecamed.importexport.gui.export;

import java.util.Map;
import java.util.Set;

/**
 * This exception allows to have simple status information about why a file
 * could not be exported form the gecamed server
 * 
 * @author Matthias Kutscheid
 * 
 */
public class PatientFileNotFoundException extends IllegalArgumentException {

	private static final long serialVersionUID = 99005864185366069L;
	/** A Map to store the relation of file names to their error state */
	private final Map<String, ErrorStates> values;

	/**
	 * Instantiate a new {@link PatientFileNotFoundException} with the given
	 * link between the erroneous file names and their error status.
	 * 
	 * @param values
	 *            A Map containing all erroneous files and their error status
	 *            for this exception.
	 */
	public PatientFileNotFoundException(Map<String, ErrorStates> values) {
		super("Some files could not be found");
		this.values = values;
	}

	/**
	 * Get the set of failed file names for this exception. The set is not
	 * guaranteed to be sorted in any way.
	 * 
	 * @return An unsorted set of all file names covered by this exception.
	 */
	public Set<String> getAllFileNames() {
		return values.keySet();
	}

	/**
	 * Get the error status for a given file name returned from
	 * {@link #getAllFileNames()}.
	 * 
	 * @param fileName
	 *            One string from {@link #getAllFileNames()}
	 * @return The error status for the given file name or null if no error
	 *         status for the given name has been found
	 */
	public ErrorStates getErrorForFile(String fileName) {
		return values.get(fileName);
	}

	/**
	 * This enumeration holds the different types of errors that can happen when
	 * trying to read a file from the server.
	 * 
	 * @author Matthias Kutscheid
	 * 
	 */
	public enum ErrorStates {
		/**The original file name was null*/
		ORIGINAL_FILENAME_NULL,
		/**The file could not be found on the server or was empty*/
		FILE_NOT_FOUND, 
		/**An IO exception occurred when trying to write the file to the local disk*/
		IO_EXCEPTION;
	}
}
