/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.labo.ejb.entity.beans;

import java.io.Serializable;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.NamedQuery;
import javax.persistence.OneToMany;
import javax.persistence.Table;

import lu.tudor.santec.gecamed.addressbook.ejb.entity.beans.Contact;
import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The Laboratory class represents a medical laboratory that transmits laboratory
 * results via the Healthnet Labo platform.
 * @author nico.mack@tudor.lu
 * @since 08/03/10 
 */

@Entity
@Table(name = "laboratory", schema = "labo")

@javax.persistence.NamedQueries 
	({
	@NamedQuery(name = Laboratory.c_AllLaboratories, 	   query = "SELECT OBJECT(o) FROM Laboratory o"),
	@NamedQuery(name = Laboratory.c_LaboratoryByUCMCode,   query = "SELECT OBJECT(o) FROM Laboratory o WHERE o.ucmCode = :" + Laboratory.c_UCMCodeParameter)
	})

public class Laboratory extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;
	
	private String		m_Name;
	private String		m_UCMCode;
	private String		m_Phone;
	private String		m_Mobile;
	private String		m_Fax;
	private String		m_Email;
	
	private Set <Contact>				m_ContactPersons;
	private Set <LaboratoryAddress>		m_Addresses;
	private Set <LaboratoryCertificate>	m_Certificates;
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	

//***************************************************************************
//* Constants                                                               *
//***************************************************************************	
	
	public static final transient String c_AllLaboratories     = "getAllLaboratories";
	public static final transient String c_LaboratoryByUCMCode = "getLaboratoryByUCMCode";

	public static final transient String c_UCMCodeParameter	   = "ucmCode";
	
	
//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	

//---------------------------------------------------------------------------
/**
 * Returns the Name of this laboratory
 * @return Laboratory Name
 */
//---------------------------------------------------------------------------

@Column (name ="name")

public String getName() 
	{
	return m_Name;
	}

//---------------------------------------------------------------------------
/**
 * Sets the Name for this laboratory
 * @param p_Name specifies the new name for this laboratory
 */
//---------------------------------------------------------------------------

public void setName (String p_Name) 
	{
	m_Name = p_Name;
	}

//---------------------------------------------------------------------------
/**
 * Returns the UCM (Union des Caisses de Maladie) Code of this laboratory
 * @return the UCM Code
 */
//---------------------------------------------------------------------------

@Column (name ="ucm_code")

public String getUcmCode() 
	{
	return m_UCMCode;
	}

//---------------------------------------------------------------------------
/**
 * Sets the UCM (Union des Caisses de Maladie) Code for this laboratory
 * @param p_UCMCode specifies the new UCM code for this laboratory
 */
//---------------------------------------------------------------------------

public void setUcmCode (String p_UCMCode) 
	{
	m_UCMCode = p_UCMCode;
	}

//---------------------------------------------------------------------------
/**
 * Returns the landline phone number of this laboratory
 * @return landline phone number of this laboratory
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="phone")

public String getPhone() 
	{
	return m_Phone;
	}

//---------------------------------------------------------------------------
/**
 * Sets this laboratory's landline phone number
 * @param p_Phone specifies the new landline phone number
 */
//---------------------------------------------------------------------------

public void setPhone(String p_Phone) 
	{
	m_Phone = p_Phone;
	}

//---------------------------------------------------------------------------
/**
 * Returns the mobile phone number of this laboratory
 * @return mobile phone number of this laboratory
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="mobile")

public String getMobile() 
	{
	return m_Mobile;
	}

//---------------------------------------------------------------------------
/**
 * Sets this laboratory's mobile phone number
 * @param p_Mobile specifies the new mobile phone number
 */
//---------------------------------------------------------------------------

public void setMobile (String p_Mobile) 
	{
	m_Mobile = p_Mobile;
	}

//---------------------------------------------------------------------------
/**
 * Returns the fax number of this laboratory
 * @return fax number of this laboratory
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="fax")

public String getFax() 
	{
	return m_Fax;
	}

//---------------------------------------------------------------------------
/**
 * Sets this laboratory's fax number
 * @param p_Mobile specifies the new mobile phone number
 */
//---------------------------------------------------------------------------

public void setFax (String p_Fax) 
	{
	m_Fax = p_Fax;
	}

//---------------------------------------------------------------------------
/**
 * Returns the email address of this laboratory
 * @return email address of this laboratory
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="email")

public String getEmail() 
	{
	return m_Email;
	}

//---------------------------------------------------------------------------
/**
 * Sets this laboratory's email address
 * @param p_Email specifies the email address for this laboratory
 */
//---------------------------------------------------------------------------

public void setEmail (String p_Email) 
	{
	m_Email = p_Email;
	}

//---------------------------------------------------------------------------
/**
 * Returns the contact persons defined for this laboratory. Contact persons
 * are fetched lazily, i.e. they have to be explicitely fetched.
 * @return All the contact persons defined for this laboratory.
 * @
 */
//---------------------------------------------------------------------------

@ManyToMany(cascade = CascadeType.ALL, fetch = FetchType.EAGER)
@JoinTable(name = "rel_laboratory_contact", schema = "LABO",
						joinColumns = {@JoinColumn(name = "laboratory_id")},
						inverseJoinColumns = {@JoinColumn(name = "contact_id")})

public Set<Contact> getContactPersons ()
	{
	return m_ContactPersons;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified set of contact persons as the new contact persons for
 * this laboratory.
 * @param p_ContactPersons specifies the new set of contact persons.
 */
//---------------------------------------------------------------------------

public void setContactPersons (Set<Contact> p_ContactPersons )
	{
	m_ContactPersons = p_ContactPersons;
	}

//---------------------------------------------------------------------------
/**
 * Returns the street addresses defined for this laboratory. Addresses
 * are fetched lazily, i.e. they have to be explicitely fetched.
 * @return All the street addresses defined for this laboratory.
 */
//---------------------------------------------------------------------------

@OneToMany(cascade = CascadeType.ALL, fetch = FetchType.EAGER)
@JoinColumn(name = "laboratory_id")
public Set<LaboratoryAddress> getAddresses ()
	{
	return m_Addresses;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified set of addresses as the new street addresses for
 * this laboratory.
 * @param p_Addresses specifies the new set of street addresses.
 */
//---------------------------------------------------------------------------

public void setAddresses (Set<LaboratoryAddress> p_Addresses )
	{
	m_Addresses = p_Addresses;
	}

//---------------------------------------------------------------------------
/**
 * Returns all the certificates (public keys) stored associated with this
 * laboratory. Certificates are fetched lazily, i.e. they have to be explicitely 
 * fetched.
 * @return All the certificates defined for this laboratory.
  */
//---------------------------------------------------------------------------

@OneToMany(cascade = CascadeType.ALL, fetch = FetchType.EAGER)
@JoinColumn(name = "laboratory_id")
public Set<LaboratoryCertificate> getCertificates ()
	{
	return m_Certificates;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified set of certificates as the new certificates for
 * this laboratory.
 * @param p_Certificates specifies the new set of certificates.
 */
//---------------------------------------------------------------------------

public void setCertificates (Set<LaboratoryCertificate> p_Certificates )
	{
	m_Certificates = p_Certificates;
	}

//---------------------------------------------------------------------------
/**
 * returns a readable representation of the laboratory
 */
//---------------------------------------------------------------------------

public String toString() 
	{
	String	l_Rendered;
	
	l_Rendered = (getName() != null)?getName():"";

	return l_Rendered;
	}

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
