/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/*
 * Author: Johannes Hermen Tudor/Santec
 * Mail: johannes.hermen@tudor.lu
 * Created: Jan 17, 2005
 *
 */
package lu.tudor.santec.gecamed.office.gui.sitemanagement;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Date;

import javax.swing.JPanel;

import lu.tudor.santec.gecamed.address.ejb.entity.beans.AddressType;
import lu.tudor.santec.gecamed.address.gui.widgets.AddressPanel;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Site;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.SiteAddress;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;


/**
 * represents a JPanel to show/edit the sites addresses
 *
 * @author Johannes Hermen johannes.hermen
 * 
 * @Version
 * <br>$Log: SiteAddressPanel.java,v $
 * <br>Revision 1.3  2013-12-27 18:09:26  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.2  2013-07-15 06:18:37  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.1  2009-02-11 14:05:53  hermen
 * <br>added management of sites to the office admin tab
 * <br>
 */
public class SiteAddressPanel extends JPanel implements KeyListener, ItemListener, PropertyChangeListener 
{
    //~ Static fields/initializers =============================================

	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger.getLogger(SiteAddressPanel.class
			.getName());
	
    private static final long serialVersionUID = 1L;
    private static final String COLUMNSPACE = "2dlu"; 
    
	private boolean listenersActivated = false;
	
	private SitePanel siteManagementPanel;

	private SiteAddress currentAddress;
	private AddressPanel addressPanel; 


    //~ Constructors ===========================================================
    /**
     */
    public SiteAddressPanel(SitePanel siteManagementPanel)
    {
    	this.setOpaque(false);
        this.siteManagementPanel = siteManagementPanel;
        this.buildPanel();
    }

    //~ Methods ================================================================
    /**
     * saves the Values of all fields to the passed siteAddress object
     *
     * @param address the siteAddress object to store the values in
     */
    private void getAddress(SiteAddress address)
    {
    	if (address == null) {
    		address = new SiteAddress();
    	}
    	try {
    		address.setType(AddressType.PRIVATE);
    	} catch (Exception e) {
    		logger.log(Level.WARN, "setting address type failed" ,e);
		}
    	
    	address.setZip(this.addressPanel.getZip());
    	
    	address.setLocalityId(this.addressPanel.getLocalityID());
        address.setLocality(this.addressPanel.getTfLocality());
    	
        address.setStreetName(this.addressPanel.getStreetName());
        address.setStreetNumber(this.addressPanel.getStreetNumber());
        
        address.setCountry(this.addressPanel.getCountry());
    	
        if (address.getDate() == null) {
        	address.setDate(new Date());
        }
    }

    /**
     * fills the fields of the panel with data from the passed siteAddress
     * object.
     *
     * @param address the siteAddress object to load the values from
     */
    private void setAddress(SiteAddress address)
    {
    	listenersActivated = false;
    		 
		this.addressPanel.setZip(address.getZip());
		
		this.addressPanel.setLocalityID(address.getLocalityId());	
		if (address.getLocalityId() == null) {
			this.addressPanel.setLocality(address.getLocality());
		}
		
		this.addressPanel.setStreet(address.getStreetName(), address.getStreetNumber());
    	
		this.addressPanel.setCountry(address.getCountry());

        listenersActivated = true;
    }

    /**
     * adds the components to the panel
     */
    private void buildPanel()
    {
        initComponents();
        // build Layout
        String line = ""; 
        for (int i = 1; i <= 12; i++) {
            line = line + "25dlu:grow, " + COLUMNSPACE + ", "; 
        }
        FormLayout layout = new FormLayout(line,"pref"); 
        this.setLayout(layout);
//        this.setBorder(new GECAMedTitledBorder(Translatrix.getTranslationString(
//                    "pm.address"))); 
        CellConstraints cc = new CellConstraints();
        this.add(this.addressPanel, cc.xyw(1, 1, 24));
    }

    /**
     * initializes the Components
     */
    private void initComponents()
    {    	
    	this.addressPanel = new AddressPanel(false);
    	this.addressPanel.addPropertyChangeListener(this);
    	
    }

	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyTyped(java.awt.event.KeyEvent)
	 */
	public void keyTyped(KeyEvent e) {
		this.siteManagementPanel.setUnsavedChanges(true);
	}
	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) {
	}
	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyReleased(java.awt.event.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) {
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
	 */
	public void itemStateChanged(ItemEvent e) {
		if (listenersActivated ) {
			this.siteManagementPanel.setUnsavedChanges(true);
			getAddress(currentAddress);
		}
	}
	
	/**
	 * sets the site to the panel
	 * @param site
	 */
	public void setSite(Site site) {
		this.currentAddress = site.getSiteAddress();
		if (this.currentAddress == null)
		    this.currentAddress = new SiteAddress();
		this.setAddress(currentAddress);
	}
	
	/**
	 * gets the site from the panel
	 * @param site
	 */
	public void getData(Site site) {
		getAddress(currentAddress);
		site.setSiteAddress(currentAddress);
	}
	
	/* (non-Javadoc)
	 * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
	 */
	public void propertyChange(PropertyChangeEvent evt) {
		if (listenersActivated && evt.getPropertyName().equals(AddressPanel.UNSAVED_CHANGES)) {
			this.siteManagementPanel.setUnsavedChanges(true);
			getAddress(currentAddress);
		}
	}
	
	/* (non-Javadoc)
	 * @see javax.swing.JComponent#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		listenersActivated = enabled;
		this.addressPanel.setEditable(enabled);
		super.setEnabled(enabled);
	}
	
}
