/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.ejb.session.interfaces;

import java.io.Serializable;
import java.math.BigInteger;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import javax.annotation.security.RolesAllowed;

import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.PatientIncidentStatisticStub;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.PatientStub;
import lu.tudor.santec.gecamed.patient.utils.PatientNameFormat;
import lu.tudor.santec.gecamed.patient.utils.PatientUpdateObject;

/**
 * Sesstion Bean Interface that manages the access to the Patient EJB. 
 * it provides methods to add and delete or modify Patients.
 *
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @version
 * <br>$Log: PatientAdminInterface.java,v $
 * <br>Revision 1.33  2013-12-27 18:09:27  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.32  2013-04-18 14:58:08  troth
 * <br>Add patient to the export query which have a act in the table billing in the database.
 * <br>
 * <br>Revision 1.31  2013-03-26 10:25:49  ferring
 * <br>Option added in admin settings, to format the patient names
 * <br>
 * <br>Revision 1.30  2013-03-21 15:35:44  troth
 * <br>1. Add a dialog inform the user if patienten search limit is reach.
 * <br>2. Make some GUI changes.
 * <br>
 * <br>Revision 1.29  2013-02-14 16:28:48  troth
 * <br>Change the export search data for patient to patientStub
 * <br>
 * <br>Revision 1.28  2013-02-12 17:22:13  troth
 * <br>add new function for the MM Export tab to search for patient.
 * <br>
 * <br>Revision 1.27  2012-11-28 16:30:12  troth
 * <br>New PatientAdminBean function to get patient by over his RIS-Id.
 * <br>
 * <br>Revision 1.26  2012-03-28 12:48:19  ferring
 * <br>Prescription printing now always takes the correct patient, independent of the SSN.
 * <br>
 * <br>Revision 1.25  2010-07-08 08:25:37  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.24  2010-06-24 11:55:09  hermen
 * <br>version 1.1
 * <br>
 * <br>Revision 1.23  2009-03-10 13:39:38  hermen
 * <br>added view and stub bean for patient incident statistics
 * <br>
 * <br>Revision 1.22  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.21  2008-04-28 08:05:48  hermen
 * <br>changed patient search to use patient_stub view and bean
 * <br>
 * <br>Revision 1.20  2008-04-03 15:03:01  hermen
 * <br>fixed matricule query
 * <br>
 * <br>Revision 1.19  2008-02-25 15:47:36  heinemann
 * <br>added batch fetching of patient names
 * <br>
 * <br>Revision 1.18  2008-02-11 16:48:45  heinemann
 * <br>added getter for SCN
 * <br>
 * <br>Revision 1.17  2008-01-17 09:52:33  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public interface PatientAdminInterface {

	/**
	 * saves the given Patient
	 *
	 * @param patient Patient Object
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public abstract Patient savePatient(Patient patient) throws Exception;

	/**
	 * Delete the given patient.
	 *
	 * @param patientId
	 * @return
	 * @throws Exception 
	 */
	@RolesAllowed("gecam")
	public Boolean deletePatient(Integer patientId) throws Exception;


	/**
	 * Returns the spezified patient
	 *
	 * @return a Patient Object representing the spezified patient
	 * @throws Exception
	 */
	public abstract Patient getPatient(Integer patientID) throws Exception;
	
	/**
	 * This method is deprecated, as it returns only one patient, although there might be more with the same SSN.
	 * 
	 * returns the patient whose social security number matches the specified one.
	 *
	 * @return Patient object whose social security number matches the specified one, null if none could be found.
	 * @throws Exception
	 * @ejb.interface-method view-type = "remote"
	 */
	@RolesAllowed("gecam")
	@Deprecated
	public Patient getPatientBySocialSecurityNumber (String socialSecurityNumber) throws Exception;
	
	public Patient getPatientOfIncident (Integer incidentId) throws Exception;

	/**
	 * Returns the name of the patient als string
	 * @param patientId
	 * @return
	 */
	public String getPatientName(Integer patientId);

	@RolesAllowed("gecam")
	public HashMap<Integer, String> getPatientNames(List<Integer> patientIds);
	
	/**
	 * sends a ObjectMessage to the JMS-Channel the clients are listening on.
	 *
	 * @param object the Serializable Object to send
	 */
	@RolesAllowed("gecam")
	public abstract void sendMessage(PatientUpdateObject object);

	/**
	 * Starts a lookup for duplicate ssn. Will return null if no
	 * copy has been found, or a list of patient that claims the
	 * same ssn.
	 * @param ssn
	 * @return
	 */
	@RolesAllowed("gecam")
	public List<Patient> checkDuplicateSSN(String ssn);
	
	
	/**
	 * returns the number of open invoices for the given patient
	 * @param patientId the patients ID
	 * @return number of open invoices
	 */
	@RolesAllowed("gecam")
	public Long getOpenInvoices(Integer patientId);

	@RolesAllowed("gecam")
	public String getSocialSecurityNumber(Integer patientId);

	@RolesAllowed("gecam")
	public Collection<PatientStub> getPatientStubListBySSec(String ssecString, Integer limit) throws Exception;
	
	@RolesAllowed("gecam")
	public Collection<PatientStub> getPatientStubListBySearchString(String searchNameString, String searchAddressString, String filter, Integer limit) throws Exception;
	
	@RolesAllowed("gecam")
	public Collection<PatientIncidentStatisticStub> getPatientIncidentStatistics(Integer siteID, Date dateFrom, Date dateTo) throws Exception;
	
	@RolesAllowed("gecam")
	public Collection<Patient> getPatientByRisID(String risID, Integer limit) throws Exception;
	
	@RolesAllowed("gecam")
	public BigInteger countPatientToExport(String queryString) throws Exception;
	
	@RolesAllowed("gecam")
	public Collection<Integer> findPatientIdsToExport(String queryString, Integer limit) throws Exception;
	
	@RolesAllowed("gecam")
	public Collection<PatientStub> findPatientToExport(String queryString, Integer limit) throws Exception;
	
	public void log (Integer level, String message);
	
	public void formatPatientNames (List<PatientNameFormat> options);
}
