/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.gui.administrative;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.utils.ComponentTitledPanel;
import lu.tudor.santec.gecamed.core.gui.widgets.GECAMedBaseDialogImpl;
import lu.tudor.santec.gecamed.core.utils.SettingPluginNames;
import lu.tudor.santec.gecamed.patient.gui.PatientManagerModule;
import lu.tudor.santec.gecamed.patient.utils.PatientNameFormat;
import lu.tudor.santec.gecamed.patient.utils.SettingConstants;
import lu.tudor.santec.i18n.Translatrix;
import lu.tudor.santec.settings.SettingsPlugin;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.jgoodies.forms.builder.PanelBuilder;
import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;


/**
 * Settings Plugin to enable/disable and change the order of the different parts of the
 * administrative patient data view.
 * 
 *
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @version
 * <br>$Log: AdministrativeElementsPlugin.java,v $
 * <br>Revision 1.22  2013-12-27 18:07:53  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.21  2013-07-15 06:18:34  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.20  2013-03-27 09:13:41  ferring
 * <br>Example for patient name formatting added as tool tipp
 * <br>
 * <br>Revision 1.19  2013-03-26 10:25:49  ferring
 * <br>Option added in admin settings, to format the patient names
 * <br>
 * <br>Revision 1.18  2012-05-14 11:56:40  ferring
 * <br>default value set for SSNField behaviour
 * <br>
 * <br>Revision 1.17  2012-05-08 07:25:05  ferring
 * <br>SSN field edit mode can now be switch to fix format, free format or no format
 * <br>
 * <br>Revision 1.16  2010-11-18 15:15:24  troth
 * <br>Complete - # 739: Behandelnder Arzt �ndert sich nicht mehr automatisch
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/739
 * <br>
 * <br>Revision 1.15  2010-09-30 11:06:55  troth
 * <br>Complete - # 602: Patient/Arzt �ndert automatisch
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/602
 * <br>
 * <br>Revision 1.14  2009-02-27 15:25:56  hermen
 * <br>added button to create an empty incident for a patient visit
 * <br>
 * <br>Revision 1.13  2009-01-06 14:48:08  hermen
 * <br>small bugfixes
 * <br>
 * <br>Revision 1.12  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.11  2008-08-20 12:15:20  hermen
 * <br>changed defaults
 * <br>
 * <br>Revision 1.10  2008-06-12 08:43:42  hermen
 * <br>changed some setting defaults
 * <br>
 * <br>Revision 1.9  2008-03-13 10:11:46  hermen
 * <br>fixed translations
 * <br>
 * <br>Revision 1.8  2008-01-21 14:03:27  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class AdministrativeElementsPlugin extends SettingsPlugin implements ActionListener, ListSelectionListener {

	/**
	 * config file containing the defaults of the plugin 
	 */
	private static final URL CONFIG_FILE = AdministrativeElementsPlugin.class.getResource("AdministrativeElementsPlugin.properties");
	
	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger
			.getLogger(AdministrativeElementsPlugin.class.getName());
	
    
    
	private static final long serialVersionUID = 1L;

	private JPanel          m_ModulePanel;
    private JLabel          m_CommentLabel;
    private JLabel          m_ModulesLabel;
    private JList           m_ModulesList;
    private JButton         m_EnableButton;
    private JButton         m_DisableButton;
    private JButton         m_UpButton;
    private JButton         m_DownButton;

    private Vector<ModuleElement> m_modules = new Vector<ModuleElement>();
	private String enabledModules = "";
	private String disabledModules = "";

	private JPanel m_FacturationPanel;
	private JRadioButton facturationAddressButton;
	private JRadioButton assurePrincipalButton;
	
	private JPanel maidenNameSelectionPanel;
	private JCheckBox maidenSwitchBox;
	private JLabel maidenInfoLabel;
	
	private JRadioButton	patientLastNameCapitalized;
	private JRadioButton	patientLastNameUpperCase;
    private JRadioButton	patientLastNameAsEntered;
	
	private JRadioButton	patientFirstNameCapitalized;
	private JRadioButton	patientFirstNameUpperCase;
    private JRadioButton	patientFirstNameAsEntered;
    
    private JPanel			physicianFormatPanel;
    private JCheckBox		physicianFormatWithTitle;
    private JCheckBox		physicianFormatSwitchNames;
    private JCheckBox		physicianFormatWithComma;
    
    private JButton			m_ChangePatientNames;
    
	private JPanel			ssnEditModePanel;
	private JRadioButton	ssnFixFormatModeButton;
	private JRadioButton	ssnFreeFormatModeButton;
	private JRadioButton	ssnNoFormatModeButton;

	private JCheckBox emptyIncidentBox;
	private JPanel incidentPanel;
	private JLabel emptyIncidentLabel;

	private JCheckBox doctorAssociationManualSwitchBox;
	private JCheckBox doctorAssociationAutomaticSwitchBox;
	
	private JCheckBox alwaysApplyAdtPatientUpdatesBox;

    private static Logger m_Logger = Logger.getLogger ("lu.tudor.santec.gecamed.core.gui.AdministrativeElementsPlugin");

    /**
     * This Methode creates a new object of the ModuleManagerPlugin.
     */
    public AdministrativeElementsPlugin() {
        super(SettingPluginNames.PATIENT_ADMIN_ELEM_PLUGIN_NAME);
        FormLayout              l_Layout;
        PanelBuilder            l_Builder;
        JPanel					l_PatientNamePanel;
        JLabel					l_Description;
        CellConstraints         l_Constraints;
        ButtonGroup				l_Group;

        // add the ResourceBundle for: lu.tudor.santec.gecam.client.gui.pm
        Translatrix.addBundle("lu.tudor.santec.gecamed.patient.gui.resources.Translatrix");

        // Create Module Managers SubPanel

        l_Layout      = new FormLayout("fill:120dlu:grow, 1dlu, fill:50dlu:grow, 1dlu",
                "pref, 5dlu, pref, pref, pref, pref, pref, 30dlu, 2dlu");

        l_Constraints = new CellConstraints();
        
        /* **************************************** */
        // 		AVAILABLE PANELS
        /* **************************************** */

        m_ModulePanel = createSubPanel(Translatrix.getTranslationString("AdministrativeElementsPlugin.ElementManager"));
        m_ModulePanel.setLayout(l_Layout);

        l_Builder     = new PanelBuilder(l_Layout,m_ModulePanel);

        // init the components:
        
        m_CommentLabel = new JLabel(Translatrix.getTranslationString("AdministrativeElementsPlugin.Comment"));
        m_ModulesLabel = new JLabel(Translatrix.getTranslationString("AdministrativeElementsPlugin.availableElements"));

        m_ModulesList = new JList();
        m_ModulesList.setCellRenderer(new ModuleFormatterJListRenderer());
        m_ModulesList.addListSelectionListener(this);
        m_ModulesList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        m_ModulesList.setSelectedIndex(0);

        m_EnableButton = new JButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.enable"));
        m_EnableButton.addActionListener(this);
        m_DisableButton = new JButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.disable"));
        m_DisableButton.addActionListener(this);
        m_UpButton = new JButton(GECAMedModule.getSmallIcon(GECAMedModule.UP));
        m_UpButton.addActionListener(this);
        m_DownButton = new JButton(GECAMedModule.getSmallIcon(GECAMedModule.DOWN));
        m_DownButton.addActionListener(this);

        l_Builder.add(m_CommentLabel,l_Constraints.xyw(1,1,3));
        l_Builder.add(m_ModulesLabel,l_Constraints.xyw(1,3,1));
        l_Builder.add(new JScrollPane(m_ModulesList),l_Constraints.xywh(1,4,1,5));
        l_Builder.add(m_EnableButton,l_Constraints.xyw(3,4,1));
        l_Builder.add(m_DisableButton,l_Constraints.xyw(3,5,1));
        l_Builder.add(m_UpButton,l_Constraints.xyw(3,6,1));
        l_Builder.add(m_DownButton,l_Constraints.xyw(3,7,1));

        addSubPanel(m_ModulePanel);

        setIcon(PatientManagerModule.getIcon(PatientManagerModule.PATIENT_ADMINSTRATIVE_MODULES));
        setLabel(Translatrix.getTranslationString("AdministrativeElementsPlugin.ElementManager"));

        loadAvailableElements(CONFIG_FILE);
        setStationary(SettingConstants.ENABLED_ELEMENTS, enabledModules);
        setStationary(SettingConstants.DISABLED_ELEMENTS, disabledModules);
        
        /* **************************************** */
        // 		INVOICE RECIPIENT
        /* **************************************** */

        m_FacturationPanel = createSubPanel(Translatrix.getTranslationString("pm.address_facturation"));
        m_FacturationPanel.setLayout(new GridLayout(0,1));

        l_Group = new ButtonGroup();
        facturationAddressButton = new JRadioButton(Translatrix.getTranslationString("pm.address_facturation"));
        l_Group.add(facturationAddressButton);
        facturationAddressButton.setSelected(true);
        m_FacturationPanel.add(facturationAddressButton);
        assurePrincipalButton = new JRadioButton(Translatrix.getTranslationString("pm.main_insurant"));
        l_Group.add(assurePrincipalButton);
        m_FacturationPanel.add(assurePrincipalButton);

        addSubPanel(m_FacturationPanel);
        
        /* **************************************** */
		// 		MAIDEN NAME
		/* **************************************** */

        maidenNameSelectionPanel = createSubPanel(Translatrix.getTranslationString("AdministrativeElementsPlugin.maidenPanel"));
        maidenNameSelectionPanel.setLayout(new BorderLayout());

        maidenInfoLabel = new JLabel();
        maidenInfoLabel.setIcon(GECAMedModule.getIcon(GECAMedIconNames.INFO));
        maidenInfoLabel.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.maidenComment"));

        maidenInfoLabel.setVerticalAlignment(SwingConstants.TOP);
        maidenInfoLabel.setVerticalTextPosition(SwingConstants.TOP);

        maidenSwitchBox = new JCheckBox(Translatrix.getTranslationString("AdministrativeElementsPlugin.maidenSwitch"));
        maidenSwitchBox.setSelected(false);

        maidenNameSelectionPanel.add(maidenInfoLabel, BorderLayout.CENTER);
        maidenNameSelectionPanel.add(maidenSwitchBox, BorderLayout.SOUTH);

        addSubPanel(maidenNameSelectionPanel);
        
        
        /* **************************************** */
		// 		PATIENT NAME MODE
		/* **************************************** */
        
        l_PatientNamePanel			= createSubPanel(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNamePanelTitle"));
        l_PatientNamePanel.setLayout(new GridLayout(0, 1));
        l_PatientNamePanel.setOpaque(false);
        addSubPanel(l_PatientNamePanel);
        
        l_Description				= new JLabel(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientFirstNameDescription"));
        l_Description.setOpaque(false);
        l_PatientNamePanel.add(l_Description);
        
        l_Group	= new ButtonGroup();
        
        patientFirstNameCapitalized	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameCapitalized"));
        patientFirstNameCapitalized.setOpaque(false);
        patientFirstNameCapitalized.setToolTipText("AdministrativeElementsPlugin.patientNameCapitalizedExample");
        l_PatientNamePanel.add(patientFirstNameCapitalized);
        l_Group.add(patientFirstNameCapitalized);
        
        patientFirstNameUpperCase	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameUpperCase"));
        patientFirstNameUpperCase.setOpaque(false);
        patientFirstNameUpperCase.setToolTipText("AdministrativeElementsPlugin.patientNameUpperCaseExample");
        l_PatientNamePanel.add(patientFirstNameUpperCase);
        l_Group.add(patientFirstNameUpperCase);
        
        patientFirstNameAsEntered	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameAsEntered"));
        patientFirstNameAsEntered.setOpaque(false);
        patientFirstNameAsEntered.setToolTipText("AdministrativeElementsPlugin.patientNameAsEnteredExamle");
        l_PatientNamePanel.add(patientFirstNameAsEntered);
        l_Group.add(patientFirstNameAsEntered);
        
        l_Description				= new JLabel(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientLastNameDescription"));
        l_Description.setOpaque(false);
        l_PatientNamePanel.add(l_Description);
        
        l_Group	= new ButtonGroup();
        
        patientLastNameCapitalized	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameCapitalized"));
        patientLastNameCapitalized.setOpaque(false);
        patientLastNameCapitalized.setToolTipText("AdministrativeElementsPlugin.patientNameCapitalizedExample");
        l_PatientNamePanel.add(patientLastNameCapitalized);
        l_Group.add(patientLastNameCapitalized);
        
        patientLastNameUpperCase	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameUpperCase"));
        patientLastNameUpperCase.setOpaque(false);
        patientLastNameUpperCase.setToolTipText("AdministrativeElementsPlugin.patientNameUpperCaseExample");
        l_PatientNamePanel.add(patientLastNameUpperCase);
        l_Group.add(patientLastNameUpperCase);
        
        patientLastNameAsEntered	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameAsEntered"));
        patientLastNameAsEntered.setOpaque(false);
        patientLastNameAsEntered.setToolTipText("AdministrativeElementsPlugin.patientNameAsEnteredExamle");
        l_PatientNamePanel.add(patientLastNameAsEntered);
        l_Group.add(patientLastNameAsEntered);
        
        
        m_ChangePatientNames	= new JButton(
        		Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameChange"));
        m_ChangePatientNames.addActionListener(this);
//        l_ChangePatientNames.setToolTipText(Translatrix.getTranslationString("AdministrativeElementsPlugin.patientNameChangeDescription"));
        l_PatientNamePanel.add(m_ChangePatientNames);
        
        
        /* **************************************** */
		// 		PHYSICAN NAME FORMAT
		/* **************************************** */
        
        physicianFormatPanel	= createSubPanel(Translatrix.getTranslationString("AdministrativeElementsPlugin.physicianPrintFormat"));
        physicianFormatPanel.setLayout(new GridLayout(0, 1));
        addSubPanel(physicianFormatPanel);
        
        physicianFormatWithTitle	= new JCheckBox(Translatrix.getTranslationString("AdministrativeElementsPlugin.physicianFormatWithTitle"));
        physicianFormatWithTitle.setSelected(true);
        physicianFormatPanel.add(physicianFormatWithTitle);
        
        physicianFormatSwitchNames	= new JCheckBox(Translatrix.getTranslationString("AdministrativeElementsPlugin.physicianFormatSwitchNames"));
        physicianFormatSwitchNames.setSelected(false);
        physicianFormatPanel.add(physicianFormatSwitchNames);
        
        physicianFormatWithComma	= new JCheckBox(Translatrix.getTranslationString("AdministrativeElementsPlugin.physicianFormatWithComma"));
        physicianFormatWithComma.setSelected(false);
        physicianFormatPanel.add(physicianFormatWithComma);
        
        
        /* **************************************** */
		// 		SSN EDIT MODE
		/* **************************************** */
        
        // add the SSN edit mode selection
        l_Group					= new ButtonGroup();
        
        ssnEditModePanel		= createSubPanel(Translatrix.getTranslationString("AdministrativeElementsPlugin.ssnEditMode"));
        ssnEditModePanel.setLayout(new GridLayout(0, 1));
        addSubPanel(ssnEditModePanel);
        
        ssnFixFormatModeButton	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.ssnFixFormatMode"));
        ssnFixFormatModeButton.setToolTipText(Translatrix.getTranslationString("AdministrativeElementsPlugin.ssnFixFormatModeTip"));
        ssnEditModePanel.add(ssnFixFormatModeButton);
        l_Group.add(ssnFixFormatModeButton);
        
        ssnFreeFormatModeButton	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.ssnFreeFormatMode"));
        ssnFreeFormatModeButton.setToolTipText(Translatrix.getTranslationString("AdministrativeElementsPlugin.ssnFreeFormatModeTip"));
        ssnEditModePanel.add(ssnFreeFormatModeButton);
        l_Group.add(ssnFreeFormatModeButton);
        
        ssnNoFormatModeButton	= new JRadioButton(Translatrix.getTranslationString("AdministrativeElementsPlugin.ssnNoFormatMode"));
        ssnNoFormatModeButton.setToolTipText(Translatrix.getTranslationString("AdministrativeElementsPlugin.ssnNoFormatModeTip"));
        ssnEditModePanel.add(ssnNoFormatModeButton);
        l_Group.add(ssnNoFormatModeButton);
        
        /* **************************************** */
		// 		EMPTY INCIDENT BUTTON
		/* **************************************** */
        
        incidentPanel = createSubPanel(Translatrix.getTranslationString("AdministrativeElementsPlugin.emptyIncident"));
        incidentPanel.setLayout(new BorderLayout());

        emptyIncidentLabel = new JLabel();
        
        emptyIncidentBox = new JCheckBox("<html>"+Translatrix.getTranslationString("AdministrativeElementsPlugin.emptyIncidentBox"));
        emptyIncidentBox.setSelected(false);

        incidentPanel.add(emptyIncidentLabel, BorderLayout.CENTER);
        incidentPanel.add(emptyIncidentBox, BorderLayout.SOUTH);

        addSubPanel(incidentPanel);
        
        /* **************************************** */
		// 		REASSIGNMENT OF PHYSICIANS
		/* **************************************** */
        
        // automatic / manual doctor Association
        doctorAssociationAutomaticSwitchBox = new JCheckBox(Translatrix._("AdministrativeElementsPlugin.doctorAssociationAutomaticSwitchBoxText"));
        doctorAssociationAutomaticSwitchBox.setSelected(true);
        
        JPanel subPanel = new JPanel();
        subPanel.setLayout(new BorderLayout());
        
        doctorAssociationManualSwitchBox = new JCheckBox(Translatrix.getTranslationString("AdministrativeElementsPlugin.doctorAssociationManualSwitchBoxText"));
        doctorAssociationManualSwitchBox.setSelected(false);
        
        if (doctorAssociationAutomaticSwitchBox.isSelected()) doctorAssociationManualSwitchBox.setEnabled(false);
		else doctorAssociationManualSwitchBox.setEnabled(true);
        
        doctorAssociationAutomaticSwitchBox.addChangeListener(new ChangeListener() {
			
			public void stateChanged(ChangeEvent e) {
				if (doctorAssociationAutomaticSwitchBox.isSelected()) doctorAssociationManualSwitchBox.setEnabled(true);
				else doctorAssociationManualSwitchBox.setEnabled(false);
			}
		});
        
        subPanel.add(doctorAssociationManualSwitchBox, BorderLayout.CENTER);
        
        ComponentTitledPanel doctorAssociationPanel = new ComponentTitledPanel(doctorAssociationAutomaticSwitchBox, subPanel);

        addSubPanel(doctorAssociationPanel);
        
        /* **************************************** */
		// ALWAYS APPLY HL7 ADT UPDATES ON PATIENTS
		/* **************************************** */
        alwaysApplyAdtPatientUpdatesBox = new JCheckBox(Translatrix._("AdministrativeElementsPlugin.alwaysApplyAdtUpdates"));
        alwaysApplyAdtPatientUpdatesBox.setToolTipText(Translatrix._("AdministrativeElementsPlugin.alwaysApplyAdtUpdatesToolTip"));
        
        JPanel hl7SubPanel = createSubPanel("HL7");
        hl7SubPanel.setLayout(new BorderLayout());
        hl7SubPanel.add(alwaysApplyAdtPatientUpdatesBox, BorderLayout.CENTER);
        addSubPanel(hl7SubPanel);
    }


    /* (non-Javadoc)
     * @see lu.tudor.santec.settings.SettingsPlugin#revertToDefaults()
     */
    public void revertToDefaults() {
        setValue(SettingConstants.ENABLED_ELEMENTS,  getDefault(SettingConstants.ENABLED_ELEMENTS));
        setValue(SettingConstants.DISABLED_ELEMENTS,  getDefault(SettingConstants.DISABLED_ELEMENTS));

		if ((Boolean) getDefault(SettingConstants.ADDRESS_FACTURATION)) {
			facturationAddressButton.setSelected(true);
		} else {
			assurePrincipalButton.setSelected(true);
		}
		
		/* ------------------------------------------------------- */
		if ((Boolean) getDefault(SettingConstants.MAIDEN_CONFIG)) {
			maidenSwitchBox.setSelected(true);
		} else {
			maidenSwitchBox.setSelected(false);
		}
		
		/* ------------------------------------------------------- */
		Integer	patientNameMode	= (Integer)getDefault(SettingConstants.PATIENT_FIRST_NAME_MODE);
		if (patientNameMode == null)
			patientNameMode = Integer.valueOf(SettingConstants.PATIENT_NAME_DEFAULT);
		
		switch (patientNameMode.intValue())
		{
			case SettingConstants.PATIENT_NAME_AS_ENTERED:
				patientFirstNameAsEntered.setSelected(true);
				break;

			case SettingConstants.PATIENT_NAME_CAPITALIZED:
				patientFirstNameCapitalized.setSelected(true);
				break;

			case SettingConstants.PATIENT_NAME_UPPER_CASE:
				patientFirstNameUpperCase.setSelected(true);
				break;
		}
		
		/* ------------------------------------------------------- */
		patientNameMode	= (Integer)getDefault(SettingConstants.PATIENT_LAST_NAME_MODE);
		if (patientNameMode == null)
			patientNameMode = Integer.valueOf(SettingConstants.PATIENT_NAME_DEFAULT);
		
		switch (patientNameMode.intValue())
		{
			case SettingConstants.PATIENT_NAME_AS_ENTERED:
				patientLastNameAsEntered.setSelected(true);
				break;

			case SettingConstants.PATIENT_NAME_CAPITALIZED:
				patientLastNameCapitalized.setSelected(true);
				break;

			case SettingConstants.PATIENT_NAME_UPPER_CASE:
				patientLastNameUpperCase.setSelected(true);
				break;
		}
		PatientNameDocument.updateMode();
		
		/* ------------------------------------------------------- */
		Boolean physicianFormat	= (Boolean) getDefault(SettingConstants.PHYSICIAN_WITH_TITLE);
		physicianFormatWithTitle.setSelected(physicianFormat.booleanValue());
		
		physicianFormat	= (Boolean) getDefault(SettingConstants.PHYSICIAN_SWITCH_NAMES);
		physicianFormatSwitchNames.setSelected(physicianFormat.booleanValue());
		
		physicianFormat	= (Boolean) getDefault(SettingConstants.PHYSICIAN_WITH_COMMA);
		physicianFormatWithComma.setSelected(physicianFormat.booleanValue());
		
		/* ------------------------------------------------------- */
		Integer ssnEditMode	= (Integer)getDefault(SettingConstants.SSN_EDIT_MODE);
		if (ssnEditMode == null)
			ssnEditMode	= Integer.valueOf(SettingConstants.SSN_DEFAULT_FORMAT_MODE);
		
		switch (ssnEditMode.intValue())
		{
			case SettingConstants.SSN_FIX_FORMAT_MODE:
				ssnFixFormatModeButton.setSelected(true);
				break;
			
			case SettingConstants.SSN_FREE_FORMAT_MODE:
				ssnFreeFormatModeButton.setSelected(true);
				break;
				
			case SettingConstants.SSN_NO_FORMAT_MODE:
				ssnNoFormatModeButton.setSelected(true);
				break;
		}

		/* ------------------------------------------------------- */
		if ((Boolean) getDefault(SettingConstants.SHOW_EMPTY_INCIDENT_BUTTON)) {
		    emptyIncidentBox.setSelected(true);
		} else {
		    emptyIncidentBox.setSelected(false);
		}

		/* ------------------------------------------------------- */
		if ((Boolean) getDefault(SettingConstants.ALLOW_MANUAL_DOCTOR_ASSOCIATION)) {
			doctorAssociationManualSwitchBox.setSelected(true);
		} else {
			doctorAssociationManualSwitchBox.setSelected(false);
		}

		/* ------------------------------------------------------- */
		if ((Boolean) getDefault(SettingConstants.ALLOW_AUTOMATIC_DOCTOR_ASSOCIATION)) {
			doctorAssociationAutomaticSwitchBox.setSelected(true);
		} else {
			doctorAssociationAutomaticSwitchBox.setSelected(false);
		}
		
		if ((Boolean) getDefault(SettingConstants.ALWAYS_APPLY_ADT_UPDATES))
			alwaysApplyAdtPatientUpdatesBox.setSelected(true);
		else
			alwaysApplyAdtPatientUpdatesBox.setSelected(false);
		

        reflectSettings();

        super.revertToDefaults();
    }

    /* (non-Javadoc)
     * @see lu.tudor.santec.settings.SettingsPlugin#updateSettings()
     */
    public void updateSettings() {

        resetValueChanged();

        setValue(SettingConstants.ADDRESS_FACTURATION, (Boolean) this.facturationAddressButton.isSelected());

        setValue(SettingConstants.ENABLED_ELEMENTS,  getEnabledElements());
        
        setValue(SettingConstants.DISABLED_ELEMENTS, getDisabledElements());

        setValue(SettingConstants.MAIDEN_CONFIG, (Boolean) this.maidenSwitchBox.isSelected());
        
        setValue(SettingConstants.PATIENT_FIRST_NAME_MODE, Integer.valueOf(
	    			patientFirstNameAsEntered.isSelected()	? SettingConstants.PATIENT_NAME_AS_ENTERED
	    		:	patientFirstNameCapitalized.isSelected()? SettingConstants.PATIENT_NAME_CAPITALIZED
	    		:	patientFirstNameUpperCase.isSelected()	? SettingConstants.PATIENT_NAME_UPPER_CASE
	    		:	SettingConstants.PATIENT_NAME_DEFAULT));
        
        setValue(SettingConstants.PATIENT_LAST_NAME_MODE, Integer.valueOf(
	    			patientLastNameAsEntered.isSelected()	? SettingConstants.PATIENT_NAME_AS_ENTERED
	    		:	patientLastNameCapitalized.isSelected()	? SettingConstants.PATIENT_NAME_CAPITALIZED
	    		:	patientLastNameUpperCase.isSelected()	? SettingConstants.PATIENT_NAME_UPPER_CASE
	    		:	SettingConstants.PATIENT_NAME_DEFAULT));
		PatientNameDocument.updateMode();
        
		setValue(SettingConstants.PHYSICIAN_WITH_TITLE,		Boolean.valueOf(physicianFormatWithTitle.isSelected()));
		setValue(SettingConstants.PHYSICIAN_SWITCH_NAMES,	Boolean.valueOf(physicianFormatSwitchNames.isSelected()));
		setValue(SettingConstants.PHYSICIAN_WITH_COMMA,		Boolean.valueOf(physicianFormatWithComma.isSelected()));
		
        setValue(SettingConstants.SSN_EDIT_MODE, Integer.valueOf(
        			ssnFixFormatModeButton.isSelected()	? SettingConstants.SSN_FIX_FORMAT_MODE
        		:	ssnFreeFormatModeButton.isSelected()? SettingConstants.SSN_FREE_FORMAT_MODE
        		:	ssnNoFormatModeButton.isSelected()	? SettingConstants.SSN_NO_FORMAT_MODE
        		:	SettingConstants.SSN_DEFAULT_FORMAT_MODE));
        
        setValue(SettingConstants.SHOW_EMPTY_INCIDENT_BUTTON, (Boolean) this.emptyIncidentBox.isSelected());
        
        if((Boolean) this.doctorAssociationAutomaticSwitchBox.isSelected())
        	setValue(SettingConstants.ALLOW_MANUAL_DOCTOR_ASSOCIATION, (Boolean) this.doctorAssociationManualSwitchBox.isSelected());
        else 
        	setValue(SettingConstants.ALLOW_MANUAL_DOCTOR_ASSOCIATION, false);
        
        setValue(SettingConstants.ALLOW_AUTOMATIC_DOCTOR_ASSOCIATION, (Boolean) this.doctorAssociationAutomaticSwitchBox.isSelected());
        
        setValue(SettingConstants.ALWAYS_APPLY_ADT_UPDATES, Boolean.valueOf(alwaysApplyAdtPatientUpdatesBox.isSelected()));
        
        super.updateSettings();
    }


    /* (non-Javadoc)
     * @see lu.tudor.santec.settings.SettingsPlugin#reflectSettings()
     */
    public void reflectSettings() {

    	Integer option;
    	Boolean	flag;
        // create a vector with the enabled / disabled elements: m_modules
        m_modules = new Vector<ModuleElement>();

        Collection<String> l_enabledModules = Arrays.asList(((String)getValue(SettingConstants.ENABLED_ELEMENTS)).split(" "));
        Collection<String> defaultEnabledModules = Arrays.asList(((String)getStationary(SettingConstants.ENABLED_ELEMENTS)).split(" "));
        
        Collection<String> l_disabledModules = Arrays.asList(((String)getValue(SettingConstants.DISABLED_ELEMENTS)).split(" "));
        Collection<String> defaultDisabledModules = Arrays.asList(((String)getStationary(SettingConstants.DISABLED_ELEMENTS)).split(" "));

        Collection<String> newModules = new ArrayList<String>();
        newModules.addAll(defaultEnabledModules);
        newModules.addAll(defaultDisabledModules);
        
        newModules.removeAll(l_enabledModules);
        
        newModules.removeAll(l_disabledModules);
        
        newModules.addAll(l_disabledModules);

        
        
        // a) enabled:
        for (Iterator<String> iter = l_enabledModules.iterator(); iter.hasNext();) {
			String l_module = (String) iter.next();
			if (l_module.length() > 3) {
				ModuleElement l_element = new ModuleElement(l_module, true);
	            m_modules.add(l_element);
			}
		}
        
        // b) disabled:
        for (Iterator<String> iter = newModules.iterator(); iter.hasNext();) {
        	String l_module = (String) iter.next();
        	if (l_module.length() > 3) {
	            ModuleElement l_element = new ModuleElement(l_module, false);
	            m_modules.add(l_element);
        	}
        }

        m_ModulesList.setListData(m_modules);
        m_ModulesList.setSelectedIndex(0);

		try {
			if ((Boolean) getValue(SettingConstants.ADDRESS_FACTURATION)) {
				facturationAddressButton.setSelected(true);
			} else {
				assurePrincipalButton.setSelected(true);
			}
		} catch (Exception e) {
			facturationAddressButton.setSelected(true);
		}
		
		/* ------------------------------------------------------- */
		// maiden switcher
		try {
			if ((Boolean) getValue(SettingConstants.MAIDEN_CONFIG)) {
				maidenSwitchBox.setSelected(true);
			} else {
				maidenSwitchBox.setSelected(false);
			}
		} catch (Exception e) {
			maidenSwitchBox.setSelected(true);
		}
		
		// patient name mode
		option	= ((Integer)getValue(SettingConstants.PATIENT_FIRST_NAME_MODE));
		if (option == null)
			option	= SettingConstants.PATIENT_NAME_DEFAULT;
		
		switch (option.intValue())
		{
			case SettingConstants.PATIENT_NAME_AS_ENTERED:
				patientFirstNameAsEntered.setSelected(true);
				break;
				
			case SettingConstants.PATIENT_NAME_CAPITALIZED:
				patientFirstNameCapitalized.setSelected(true);
				break;
				
			case SettingConstants.PATIENT_NAME_UPPER_CASE:
				patientFirstNameUpperCase.setSelected(true);
				break;
		}

		option	= ((Integer)getValue(SettingConstants.PATIENT_LAST_NAME_MODE));
		if (option == null)
			option	= SettingConstants.PATIENT_NAME_DEFAULT;
		
		switch (option.intValue())
		{
			case SettingConstants.PATIENT_NAME_AS_ENTERED:
				patientLastNameAsEntered.setSelected(true);
				break;
				
			case SettingConstants.PATIENT_NAME_CAPITALIZED:
				patientLastNameCapitalized.setSelected(true);
				break;
				
			case SettingConstants.PATIENT_NAME_UPPER_CASE:
				patientLastNameUpperCase.setSelected(true);
				break;
		}
		
		// format physician
		flag	= (Boolean) getValue(SettingConstants.PHYSICIAN_WITH_TITLE);
		physicianFormatWithTitle.setSelected(flag != null ? flag.booleanValue() : true);

		flag	= (Boolean) getValue(SettingConstants.PHYSICIAN_SWITCH_NAMES);
		physicianFormatSwitchNames.setSelected(flag != null ? flag.booleanValue() : false);

		flag	= (Boolean) getValue(SettingConstants.PHYSICIAN_WITH_COMMA);
		physicianFormatWithComma.setSelected(flag != null ? flag.booleanValue() : false);
		
		// SSN edit mode
		option	= ((Integer)getValue(SettingConstants.SSN_EDIT_MODE));
		if (option == null)
			option	= SettingConstants.SSN_DEFAULT_FORMAT_MODE;
		
		switch (option.intValue())
		{
			case SettingConstants.SSN_FIX_FORMAT_MODE:
				ssnFixFormatModeButton.setSelected(true);
				break;
				
			case SettingConstants.SSN_FREE_FORMAT_MODE:
				ssnFreeFormatModeButton.setSelected(true);
				break;
				
			case SettingConstants.SSN_NO_FORMAT_MODE:
				ssnNoFormatModeButton.setSelected(true);
				break;
		}
		
		try {
			if ((Boolean) getValue(SettingConstants.SHOW_EMPTY_INCIDENT_BUTTON)) {
				emptyIncidentBox.setSelected(true);
			} else {
			    emptyIncidentBox.setSelected(false);
			}
		} catch (Exception e) {
		    emptyIncidentBox.setSelected(false);
		}
		
		// ALLOW_MANUAL_DOCTOR_ASSOCIATION switcher
		try {
			if ((Boolean) getValue(SettingConstants.ALLOW_MANUAL_DOCTOR_ASSOCIATION)) {
				doctorAssociationManualSwitchBox.setSelected(true);
			} else {
				doctorAssociationManualSwitchBox.setSelected(false);
			}
		} catch (Exception e) {
			doctorAssociationManualSwitchBox.setSelected(true);
		}
		
		// ALLOW_AUTOMATIC_DOCTOR_ASSOCIATION switcher
		try {
			if ((Boolean) getValue(SettingConstants.ALLOW_AUTOMATIC_DOCTOR_ASSOCIATION)) {
				doctorAssociationAutomaticSwitchBox.setSelected(true);
			} else {
				doctorAssociationAutomaticSwitchBox.setSelected(false);
			}
		} catch (Exception e) {
			doctorAssociationAutomaticSwitchBox.setSelected(true);
		}
		
		// ALWAYS APPLY ADT UPDATES
		try
		{
			alwaysApplyAdtPatientUpdatesBox.setSelected((Boolean) getValue(SettingConstants.ALWAYS_APPLY_ADT_UPDATES));
		}
		catch (Exception e)
		{
			alwaysApplyAdtPatientUpdatesBox.setSelected(SettingConstants.DEFAULT_ALWAYS_APPLY_ADT_UPDATES);
		}
		
        super.reflectSettings();
    }

    /**
     * loads the available elements from the config file
     * @param url
     */
    private void loadAvailableElements(URL url){
    	try {
			BufferedReader br = new BufferedReader(
					new InputStreamReader(url.openStream()));
			String line;
			while ((line = br.readLine()) != null) {
				if (!line.startsWith("#") && line.length() > 3) {
					try {
						String[] fields = line.split("=");

						if (fields[1].toLowerCase().equals("on")) {
							this.enabledModules += fields[0] + " ";
						} else {
							this.disabledModules += fields[0] + " ";
						}
					} catch (Exception e) {
						m_Logger.log(Level.WARN, "line is no valid settingline: " + line);
					}
				}
			}
		} catch (Exception e) {
			m_Logger.log(Level.WARN, "error parsing file ", e);
		}
    }

    //---------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent p_ActionEvent) {

        if (p_ActionEvent.getSource() == m_EnableButton) {
            // get selected Element:
            ModuleElement e = (ModuleElement) m_ModulesList.getSelectedValue();

            // change the value to "true":
            e.setEnabled(true);

            int l_currentLocation = m_ModulesList.getSelectedIndex();
            m_ModulesList.setSelectedIndex((l_currentLocation+1));

            m_ModulesList.repaint();
        }

        if (p_ActionEvent.getSource() == m_DisableButton) {
            // get selected Element:
        	ModuleElement e = (ModuleElement) m_ModulesList.getSelectedValue();

            // change the value to "true":
        	e.setEnabled(false);

            int l_currentLocation = m_ModulesList.getSelectedIndex();
            m_ModulesList.setSelectedIndex((l_currentLocation+1));

            m_ModulesList.repaint();
        }

        if (p_ActionEvent.getSource() == m_UpButton) {

            int l_currentLocation = m_ModulesList.getSelectedIndex();
            ModuleElement o = (ModuleElement)  m_ModulesList.getModel().getElementAt(l_currentLocation);

            if(o != m_modules.firstElement()){

                // remove element at selected location:
                m_modules.remove(o);

                // add it to selected location - 1:
                m_modules.add((l_currentLocation-1), o);
                m_ModulesList.setSelectedIndex((l_currentLocation-1));


                m_ModulesList.repaint();
            }
        }

        if (p_ActionEvent.getSource() == m_DownButton) {
            int l_currentLocation = m_ModulesList.getSelectedIndex();
            ModuleElement o = (ModuleElement)  m_ModulesList.getModel().getElementAt(l_currentLocation);

            if(o != m_modules.lastElement()){

                // remove element at selected location:
                m_modules.remove(o);

                // add it to selected location - 1:
                m_modules.add((l_currentLocation+1), o);
                m_ModulesList.setSelectedIndex((l_currentLocation+1));

                m_ModulesList.repaint();
            }
        }
        
        if (p_ActionEvent.getSource() == m_ChangePatientNames)
        {
        	applyFormatToAllPatient();
        }
    }

    /* (non-Javadoc)
     * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
     */
    public void valueChanged(ListSelectionEvent p_SelectioEvent){
        try {
        } catch (Exception ioe) {
        }
    }

    /**
     * Returns the currently in the Selection list enabled Modules.
     * @return A space separated list of enabled modules.
     */
    public String getEnabledElements(){

        String l_result = "";

        // walk over all modules:
        int l_numberOfElements = m_ModulesList.getModel().getSize();

        for(int i = 0; i < l_numberOfElements; i++){
        	ModuleElement e = (ModuleElement) m_ModulesList.getModel().getElementAt(i);
            // if they are enabled, add the class names to the result
            if(e.isEnabled()){
                l_result += e.getName() + " ";
            }
        }
        // ok, done:
        return l_result;
    }

    /**
     * Returns the currently in the Selection list disabled Modules.
     * @return A space separated list of disabled modules.
     */
    public String getDisabledElements(){

        String l_result = "";

        // walk over all modules:
        int l_numberOfElements = m_ModulesList.getModel().getSize();

        for(int i = 0; i < l_numberOfElements; i++){
        	ModuleElement e = (ModuleElement) m_ModulesList.getModel().getElementAt(i);

            // if they are enabled, add the class names to the result
            if(! e.isEnabled()){
                l_result += e.getName() + " ";
            }
        }

        // ok, done:
        return l_result;
    }
    
    @Override
    protected Object getDefault(String p_Key)
    {
    	return super.getDefault(p_Key);
    }

    /* (non-Javadoc)
     * @see lu.tudor.santec.settings.SettingsPlugin#relocalize()
     */
    public void relocalize() {

       super.relocalize();

       m_CommentLabel.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.Comment"));
       m_EnableButton.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.enable"));
       m_DisableButton.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.disable"));

       m_ModulesLabel.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.availableElements"));
       setLabel(Translatrix.getTranslationString("AdministrativeElementsPlugin.ElementManager"));

       maidenInfoLabel.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.maidenComment"));
       
       emptyIncidentBox.setText("<html>"+Translatrix.getTranslationString("AdministrativeElementsPlugin.emptyIncidentBox"));
       
       doctorAssociationManualSwitchBox.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.doctorAssociationSwitchBoxText"));
       
       doctorAssociationAutomaticSwitchBox.setText(Translatrix.getTranslationString("AdministrativeElementsPlugin.doctorAssociationAutomaticSwitchBoxText"));
       
       alwaysApplyAdtPatientUpdatesBox.setText(Translatrix._("AdministrativeElementsPlugin.alwaysApplyAdtUpdates"));
       alwaysApplyAdtPatientUpdatesBox.setToolTipText(Translatrix._("AdministrativeElementsPlugin.alwaysApplyAdtUpdatesToolTip"));
    }


	/**
	 * 
	 */
	public void setStationaryValues() {
		setStationary(SettingConstants.ADDRESS_FACTURATION, Boolean.TRUE);
		setStationary(SettingConstants.MAIDEN_CONFIG, Boolean.TRUE);
		setStationary(SettingConstants.SSN_EDIT_MODE, SettingConstants.SSN_DEFAULT_FORMAT_MODE);
		setStationary(SettingConstants.PATIENT_FIRST_NAME_MODE, SettingConstants.PATIENT_NAME_DEFAULT);
		setStationary(SettingConstants.PATIENT_LAST_NAME_MODE, SettingConstants.PATIENT_NAME_DEFAULT);
		setStationary(SettingConstants.PHYSICIAN_WITH_TITLE, Boolean.TRUE);
		setStationary(SettingConstants.PHYSICIAN_SWITCH_NAMES, Boolean.FALSE);
		setStationary(SettingConstants.PHYSICIAN_WITH_COMMA, Boolean.FALSE);
		setStationary(SettingConstants.SHOW_EMPTY_INCIDENT_BUTTON, Boolean.FALSE);
		setStationary(SettingConstants.ALLOW_MANUAL_DOCTOR_ASSOCIATION, Boolean.TRUE);
		setStationary(SettingConstants.ALLOW_AUTOMATIC_DOCTOR_ASSOCIATION, Boolean.TRUE);
		setStationary(SettingConstants.ALWAYS_APPLY_ADT_UPDATES, SettingConstants.DEFAULT_ALWAYS_APPLY_ADT_UPDATES);
	}
	
	
	public void applyFormatToAllPatient ()
	{
		List<PatientNameFormat>	options	= new LinkedList<PatientNameFormat>();
		Integer	firstNameMode	= (Integer) getValue(SettingConstants.PATIENT_FIRST_NAME_MODE);
		Integer lastNameMode	= (Integer) getValue(SettingConstants.PATIENT_LAST_NAME_MODE);
		boolean	saved			= true;
		int		option;
		
		
		if (firstNameMode == null)
			firstNameMode	= SettingConstants.PATIENT_NAME_DEFAULT;
		if (lastNameMode == null)
			lastNameMode	= SettingConstants.PATIENT_NAME_DEFAULT;
		
		// create the options and check, if the settings are saved
		if (firstNameMode == SettingConstants.PATIENT_NAME_CAPITALIZED)
		{
			if (!patientFirstNameCapitalized.isSelected())
				saved	= false;
			options.add(PatientNameFormat.FIRST_NAME_CAPITALIZE);
		}
		else if (firstNameMode == SettingConstants.PATIENT_NAME_UPPER_CASE)
		{
			if (!patientFirstNameUpperCase.isSelected())
				saved	= false;
			options.add(PatientNameFormat.FIRST_NAME_TO_UPPER_CASE);
		}
		
		if (lastNameMode == SettingConstants.PATIENT_NAME_CAPITALIZED)
		{
			if (!patientLastNameCapitalized.isSelected())
				saved	= false;
			options.add(PatientNameFormat.LAST_NAME_CAPITALIZE);
		}
		else if (lastNameMode == SettingConstants.PATIENT_NAME_UPPER_CASE)
		{
			if (!patientLastNameUpperCase.isSelected())
				saved	= false;
			options.add(PatientNameFormat.LAST_NAME_TO_UPPER_CASE);
		}
		
		
		if (!saved)
		{
			// the settings haven't been saved, yet. Do not proceed
			GECAMedBaseDialogImpl.showMessageDialog(this, 
					Translatrix.getTranslationString("AdministrativeElementsPlugin.formatPatientNameErrorTitle"), 
					Translatrix.getTranslationString("AdministrativeElementsPlugin.formatPatientNameErrorMessage"), 
					GECAMedBaseDialogImpl.OK_BUTTON_MODE, 
					GECAMedModule.getIcon(GECAMedIconNames.ERROR));
			
			return;
		}
		
		// inform the user that it will take some time
		option	= GECAMedBaseDialogImpl.showMessageDialog(this, 
				Translatrix.getTranslationString("AdministrativeElementsPlugin.formatPatientNameWarningTitle"), 
				Translatrix.getTranslationString("AdministrativeElementsPlugin.formatPatientNameWarningMessage"), 
				GECAMedBaseDialogImpl.OK_CANCEL_BUTTON_MODE);
		
		if (option == GECAMedBaseDialogImpl.OK_OPTION)
		{
			new PatientNameFormatTask(getOwner(), options).start();
		}
	}
	
	
	private JDialog getOwner ()
	{
		Component	c	= this;
		
		while (c != null && !(c instanceof JDialog))
		{
			c = c.getParent();
		}
		
		return (JDialog) c;
	}

    /**
     * Renderer for the ModuleElement
     * 
     *
     * @author Johannes Hermen johannes.hermen(at)tudor.lu
     *
     * @version
     * <br>$Log: AdministrativeElementsPlugin.java,v $
     * <br>Revision 1.22  2013-12-27 18:07:53  donak
     * <br>Cleanup of imports
     * <br>
     * <br>Revision 1.21  2013-07-15 06:18:34  ferring
     * <br>logging changed
     * <br>
     * <br>Revision 1.20  2013-03-27 09:13:41  ferring
     * <br>Example for patient name formatting added as tool tipp
     * <br>
     * <br>Revision 1.19  2013-03-26 10:25:49  ferring
     * <br>Option added in admin settings, to format the patient names
     * <br>
     * <br>Revision 1.18  2012-05-14 11:56:40  ferring
     * <br>default value set for SSNField behaviour
     * <br>
     * <br>Revision 1.17  2012-05-08 07:25:05  ferring
     * <br>SSN field edit mode can now be switch to fix format, free format or no format
     * <br>
     * <br>Revision 1.16  2010-11-18 15:15:24  troth
     * <br>Complete - # 739: Behandelnder Arzt �ndert sich nicht mehr automatisch
     * <br>http://santec.tudor.lu/trac/gecamed/ticket/739
     * <br>
     * <br>Revision 1.15  2010-09-30 11:06:55  troth
     * <br>Complete - # 602: Patient/Arzt �ndert automatisch
     * <br>http://santec.tudor.lu/trac/gecamed/ticket/602
     * <br>
     * <br>Revision 1.14  2009-02-27 15:25:56  hermen
     * <br>added button to create an empty incident for a patient visit
     * <br>
     * <br>Revision 1.13  2009-01-06 14:48:08  hermen
     * <br>small bugfixes
     * <br>
     * <br>Revision 1.12  2008-09-25 09:43:06  heinemann
     * <br>fixed copyrights
     * <br>
     * <br>Revision 1.11  2008-08-20 12:15:20  hermen
     * <br>changed defaults
     * <br>
     * <br>Revision 1.10  2008-06-12 08:43:42  hermen
     * <br>changed some setting defaults
     * <br>
     * <br>Revision 1.9  2008-03-13 10:11:46  hermen
     * <br>fixed translations
     * <br>
     * <br>Revision 1.8  2008-01-21 14:03:27  hermen
     * <br>updated Javadoc and refactured code
     * <br>
     *
     */
    public class ModuleFormatterJListRenderer extends JLabel implements
			ListCellRenderer {

		private static final long serialVersionUID = 1L;

		/**
		 * builds a new DefaultComboboxRenderer
		 */
		public ModuleFormatterJListRenderer() {
			setOpaque(true);
			setHorizontalAlignment(LEFT);
			setVerticalAlignment(CENTER);
		}


		/* (non-Javadoc)
		 * @see javax.swing.ListCellRenderer#getListCellRendererComponent(javax.swing.JList, java.lang.Object, int, boolean, boolean)
		 */
		public Component getListCellRendererComponent(JList list, Object value,
		        int index, boolean isSelected, boolean cellHasFocus) {
		    if (isSelected) {
		        setBackground(list.getSelectionBackground());
		        setForeground(list.getSelectionForeground());
		    } else {
		        setBackground(list.getBackground());
		        setForeground(list.getForeground());
		    }
		    try {
		        String l_description = ((ModuleElement)value).getName();
		
		        this.setText(l_description.substring(l_description.lastIndexOf(".")+1));
		
		        if(((ModuleElement)value).isEnabled()){
		            this.setIcon(GECAMedModule.getSmallIcon(GECAMedModule.OK));
		        } else {
		            this.setIcon(GECAMedModule.getSmallIcon(GECAMedModule.CANCEL));
		        }
		    } catch (Exception e) {
		    	logger.log(Level.WARN, "rendering plugins failed", e);
		    }
		    return this;
		}
    }

    /**
     * class to represent an module element
     *
     * @author Johannes Hermen johannes.hermen(at)tudor.lu
     *
     * @version
     * <br>$Log: AdministrativeElementsPlugin.java,v $
     * <br>Revision 1.22  2013-12-27 18:07:53  donak
     * <br>Cleanup of imports
     * <br>
     * <br>Revision 1.21  2013-07-15 06:18:34  ferring
     * <br>logging changed
     * <br>
     * <br>Revision 1.20  2013-03-27 09:13:41  ferring
     * <br>Example for patient name formatting added as tool tipp
     * <br>
     * <br>Revision 1.19  2013-03-26 10:25:49  ferring
     * <br>Option added in admin settings, to format the patient names
     * <br>
     * <br>Revision 1.18  2012-05-14 11:56:40  ferring
     * <br>default value set for SSNField behaviour
     * <br>
     * <br>Revision 1.17  2012-05-08 07:25:05  ferring
     * <br>SSN field edit mode can now be switch to fix format, free format or no format
     * <br>
     * <br>Revision 1.16  2010-11-18 15:15:24  troth
     * <br>Complete - # 739: Behandelnder Arzt �ndert sich nicht mehr automatisch
     * <br>http://santec.tudor.lu/trac/gecamed/ticket/739
     * <br>
     * <br>Revision 1.15  2010-09-30 11:06:55  troth
     * <br>Complete - # 602: Patient/Arzt �ndert automatisch
     * <br>http://santec.tudor.lu/trac/gecamed/ticket/602
     * <br>
     * <br>Revision 1.14  2009-02-27 15:25:56  hermen
     * <br>added button to create an empty incident for a patient visit
     * <br>
     * <br>Revision 1.13  2009-01-06 14:48:08  hermen
     * <br>small bugfixes
     * <br>
     * <br>Revision 1.12  2008-09-25 09:43:06  heinemann
     * <br>fixed copyrights
     * <br>
     * <br>Revision 1.11  2008-08-20 12:15:20  hermen
     * <br>changed defaults
     * <br>
     * <br>Revision 1.10  2008-06-12 08:43:42  hermen
     * <br>changed some setting defaults
     * <br>
     * <br>Revision 1.9  2008-03-13 10:11:46  hermen
     * <br>fixed translations
     * <br>
     * <br>Revision 1.8  2008-01-21 14:03:27  hermen
     * <br>updated Javadoc and refactured code
     * <br>
     *
     */
    public class ModuleElement {

    	private String name;
    	private boolean enabled;

    	public ModuleElement(String name, boolean enabled) {
    		this.name = name;
    		this.enabled = enabled;
    	}

    	public String getName() {
    		return name;
    	}

    	public boolean isEnabled() {
    		return enabled;
    	}

    	public void setEnabled(boolean enabled) {
    		this.enabled = enabled;
    	}

    }
}

