/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.gui.hospitalisation;

import java.net.URL;
import java.util.HashMap;
import java.util.Vector;

import javax.swing.table.AbstractTableModel;

import lu.tudor.santec.gecamed.core.gui.GECAMedLists;
import lu.tudor.santec.gecamed.core.gui.IconFetcher;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.patient.ejb.entity.interfaces.HospitalisationElement;
import lu.tudor.santec.gecamed.patient.ejb.session.beans.HospitalisationBean;
import lu.tudor.santec.gecamed.patient.ejb.session.interfaces.HospitalisationBeanInterface;
import lu.tudor.santec.gecamed.patient.gui.PatientManagerModule;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

public class HospitalisationTableModel extends AbstractTableModel {
 	private static final long serialVersionUID = 1L;

 	private HospitalisationBeanInterface hospManager;
//	private OfficeManagerInterface officeManager;

	private static Logger logger = Logger.getLogger ("lu.tudor.santec.gecamed.patient.gui.hospitalisation.PatientHospitalisationPanel");

	private static final URL ICON_HOSP_A = PatientManagerModule.class.getResource("resources/icons/"+PatientManagerModule.HOSPITALISATION_PERIODE_AMBULATORY);
	private static final URL ICON_HOSP_1 = PatientManagerModule.class.getResource("resources/icons/"+PatientManagerModule.HOSPITALISATION_PERIODE_FIRSTCLASS);
	private static final URL ICON_HOSP_2 = PatientManagerModule.class.getResource("resources/icons/"+PatientManagerModule.HOSPITALISATION_PERIODE_SECONDCLASS);
	private static final URL ICON_PASSAGE = PatientManagerModule.class.getResource("resources/icons/"+PatientManagerModule.HOSP_PASSAGE);
	private static final URL ICON_PRESCRIPTION = PatientManagerModule.class.getResource("resources/icons/"+PatientManagerModule.HOSP_PRESCRIPTION);
	private static final URL ICON_REPORT = PatientManagerModule.class.getResource("resources/icons/"+PatientManagerModule.HOSP_REPORT);
	private static final URL ICON_BLANK = PatientManagerModule.class.getResource("resources/icons/"+PatientManagerModule.EMPTY_ICON16);
	
	private static final String ICON_HOSP_A_URL = IconFetcher.fixJarURL(ICON_HOSP_A).toString();
	private static final String ICON_HOSP_1_URL = IconFetcher.fixJarURL(ICON_HOSP_1).toString();
	private static final String ICON_HOSP_2_URL = IconFetcher.fixJarURL(ICON_HOSP_2).toString();
	private static final String ICON_PASSAGE_URL = IconFetcher.fixJarURL(ICON_PASSAGE).toString();
	private static final String ICON_PRESCRIPTION_URL = IconFetcher.fixJarURL(ICON_PRESCRIPTION).toString();
	private static final String ICON_REPORT_URL = IconFetcher.fixJarURL(ICON_REPORT).toString();
	private static final String ICON_BLANK_URL = IconFetcher.fixJarURL(ICON_BLANK).toString();
	
	private final static String[] columns = {
		Translatrix.getTranslationString("patient.history.date"),
		Translatrix.getTranslationString("patient.history.physician"),
		Translatrix.getTranslationString("patient.history.type"),
		Translatrix.getTranslationString("patient.history.rx"),
		Translatrix.getTranslationString("patient.history.description")
	};
 	
	private Vector<HospitalisationElement> elements = new Vector<HospitalisationElement>();

	private HashMap<Integer, Physician> physicians = new  HashMap<Integer, Physician>();


	public HospitalisationTableModel() {
 		
		try {
 			hospManager = (HospitalisationBeanInterface) ManagerFactory.getRemote(HospitalisationBean.class);
// 			officeManager = (OfficeManagerInterface) ManagerFactory.getRemote(OfficeManagerBean.class);
 			for (Physician p : GECAMedLists.getListReference(Physician.class)) {
				physicians .put(p.getId(), p);
			}
 		} 
 		catch (Exception p_Exception) 
 			{
 			logger.warn(p_Exception.getLocalizedMessage());
 			}
 	}

 	
	public int getRowCount() {
		return elements.size();
	}

	public int getColumnCount() {
		return columns.length;
	}
	
	@Override
	public String getColumnName(int column) {
		return columns[column];
	}

	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return false;
	}

	public Object getValueAt(int rowIndex, int columnIndex) {
		
		if (rowIndex == -1) 
			return null;
		
		HospitalisationElement he;
		try {
			he = elements.get(rowIndex);
		} catch (Exception e) {
			logger.log(Level.WARN, "retrieving HospitalisationElement failed", e);
			return "";
		}
		switch (columnIndex) {
		case 0:
			if (HospitalisationElement.HOSPITALISATION == he.getType() )
				return he.getDate();
			else 
				return null;
		case 1:
			if (he.getPhysicianID() != null) {
				try {
					return physicians.get(he.getPhysicianID()).getName();				
				} catch (Exception e) {
					return "";
				}
			} else {
				return "";
			}
		case 2:
			return he.getType();
		case 3:
			return he.isRx();
		case 4:
			String desc = he.getDescription();
			switch (he.getType()) {
			case HospitalisationElement.HOSPITALISATION_PERIOD_1:
				return desc.replaceFirst(":image", ICON_HOSP_1_URL);
			case HospitalisationElement.HOSPITALISATION_PERIOD_2:
				return desc.replaceFirst(":image",  ICON_HOSP_2_URL);
			case HospitalisationElement.HOSPITALISATION_PERIOD_A:
				return desc.replaceFirst(":image",  ICON_HOSP_A_URL);
			case HospitalisationElement.PASSAGE:
				return desc.replaceFirst(":image",  ICON_PASSAGE_URL).replaceAll(":blank",  ICON_BLANK_URL);
			case HospitalisationElement.PRESCRIPTION:
				return desc.replaceFirst(":image",  ICON_PRESCRIPTION_URL).replaceAll(":blank",  ICON_BLANK_URL);
			case HospitalisationElement.REPORT:
				return desc.replaceFirst(":image",  ICON_REPORT_URL).replaceAll(":blank",  ICON_BLANK_URL);
			default:
				return desc;
			}
		default:
			return "";
		}
	}
	
	public void setPatient(Patient p) {
		this.elements.removeAllElements();
		try {
			this.elements.addAll(hospManager.getHospitalisationHistory(p.getId(), Translatrix.getLocale()));
		} catch (Exception e) {
			logger.log(Level.WARN, "Loading of Hospitalisations for " + p + " failed", e);
		}
		fireTableDataChanged();
	}
	
	
	public HospitalisationElement getHospitalisationElement(int row) {
		return elements.get(row);
	}

}
