/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.gui.simpleview;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;

import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;

import lu.tudor.santec.gecamed.billing.gui.BillingModule;
import lu.tudor.santec.gecamed.core.ejb.entity.beans.PhoneType;
import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.IconFetcher;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.GECAMedGuiUtils;
import lu.tudor.santec.gecamed.core.gui.utils.GradientFactory;
import lu.tudor.santec.gecamed.core.gui.widgets.HeaderedPanel;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Incident;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.PatientAddress;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.PatientMemo;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.PatientPhone;
import lu.tudor.santec.gecamed.patient.ejb.session.beans.PatientAdminBean;
import lu.tudor.santec.gecamed.patient.ejb.session.interfaces.PatientAdminInterface;
import lu.tudor.santec.gecamed.patient.gui.PatientManagerModule;
import lu.tudor.santec.gecamed.patient.gui.PatientPanel;
import lu.tudor.santec.gecamed.patient.gui.PatientViewTab;
import lu.tudor.santec.gecamed.patient.gui.administrative.PatientAdminPanel;
import lu.tudor.santec.gecamed.patient.gui.history.viewtab.HistoryViewTab;
import lu.tudor.santec.gecamed.patient.gui.memo.SimpleMemoView;
import lu.tudor.santec.i18n.Translatrix;
import bizcal.util.DateUtil;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * The new view of a patient. It is made of a short overview of the important
 * adminstrative data like name, address, phone, job, day of birth, age, health
 * insurance, last consultation, unpaid invoices, third party payer and if there
 * are memos attached to this patient.
 * 
 * Further information like important data, antecedents and medical history are
 * placed in two JSplitPanes below. Each element of this can be adjusted in size
 * and switched on/off.
 * 
 * 
 * @author martin.heinemann@tudor.lu 22.11.2007 14:12:58
 * 
 * 
 * @version <br>
 *          $Log: OverviewPanel.java,v $
 *          Revision 1.3  2013-11-19 13:06:55  troth
 *          Set the important data panels in real time in the history view.
 *
 *          Revision 1.2  2013-02-05 13:03:12  ferring
 *          unnecessary import removed
 *
 *          Revision 1.1  2010-10-19 14:37:14  troth
 *          rename class SimpleViewPanel in OverviewPanel
 *
 *          Revision 1.20  2010-05-03 08:02:53  troth
 *          remove function createPostAddress because it is not needed
 *
 *          Revision 1.19  2010-04-26 16:36:18  troth
 *          Redesign of the prescription view
 *
 *          Revision 1.18  2010-03-23 15:17:28  troth
 *          Redesign of the patient view history
 * <br>
 *          Revision 1.17 2008-09-25 09:43:06 heinemann <br>
 *          fixed copyrights <br>
 * <br>
 *          Revision 1.16 2008-08-21 15:45:12 heinemann <br>
 *          removed the bogus behaviour when there is no office entry in the
 *          database. The fields on the print outs are now left blank. The user
 *          gets an information that the office address is missing. <br>
 * <br>
 *          Revision 1.15 2008-08-19 10:25:08 heinemann <br>
 *          cleanup <br>
 * <br>
 *          Revision 1.14 2008-07-04 13:38:17 heinemann <br>
 *          Complete - task 5: Create Icon for History List Views <br>
 * <br>
 *          Revision 1.13 2008-07-03 11:59:57 heinemann <br>
 *          *** empty log message *** <br>
 * <br>
 *          Revision 1.12 2008-05-26 12:44:40 heinemann <br>
 *          *** empty log message *** <br>
 * <br>
 *          Revision 1.11 2008-03-12 10:30:06 hermen <br>
 *          changed PatientViewTab to abstract and added getName() method <br>
 * <br>
 *          Revision 1.10 2008-03-12 10:14:14 heinemann <br>
 *          *** empty log message *** <br>
 * <br>
 *          Revision 1.9 2008-01-21 14:58:42 heinemann <br>
 *          code cleanup and java doc <br>
 * <br>
 *          Revision 1.8 2008-01-03 14:44:09 heinemann <br>
 *          now showing age next to the day of birth <br>
 * <br>
 *          Revision 1.7 2008-01-02 15:48:23 heinemann <br>
 *          moved mouselistener from JSplitpane to the top JLabel <br>
 * <br>
 *          Revision 1.6 2008-01-02 13:48:32 heinemann <br>
 *          added nice gradient panels <br>
 * <br>
 * 
 */
public class OverviewPanel extends PatientViewTab implements ActionListener {

	private static final long serialVersionUID = 1L;
	// ========================================================================
	// Class constants
	// ========================================================================
	public static final Color BLUE_BACK = new Color(226, 235, 255);
	public static final Color LIGHT_BACK = new Color(237, 242, 251);
	public static final Color LIGHT_BORDER = new Color(163, 184, 204);
	public static final Color DARK_BACK = new Color(184, 207, 229);

	public static final Font LABEL_FONT = new Font("Dialog", 0, 12);

	public static final String NAME = "SIMPLE_VIEW_PANEL";

	/* ------------------------------------------------------- */
	// action commands for the buttons
	/* ------------------------------------------------------- */
	public static final String CMD_ADDRESS = "p_address";
	public static final String CMD_PHONE = "p_phone";
	public static final String CMD_JOB = "p_job";
	public static final String CMD_BIRTH = "p_birth";
	public static final String CMD_UCM = "p_ucm";
	public static final String CMD_LASTVISIT = "p_lastvisit";
	public static final String CMD_MEMOWARNING = "p_memowarning";
	public static final String CMD_THIRDPARTY = "p_thirdparty";
	public static final String CMD_ANAMNESIS = "p_anamnesis";
	public static final String CMD_ANTECEDENTS = "p_antecedents";
	public static final String CMD_IMPORTANT = "p_important";
	public static final String CMD_OPEN_INVOICE = "p_open_invoice";

	private static final String FORMAT_LIGHT = "<html><span style=\"font-family:Arial;\">";

//	private final DateFormat birthFormat = SimpleDateFormat.getDateInstance(DateFormat.MEDIUM);
//	private final DateFormat lastVisitFormat = SimpleDateFormat.getDateInstance(DateFormat.MEDIUM);
	private final DateFormat birthFormat = GECAMedGuiUtils.getDateFormat(false);
	private final DateFormat lastVisitFormat = GECAMedGuiUtils.getDateFormat(false);

	// ========================================================================
	// Class members
	// ========================================================================

	private PatientPanel patientPanel;

	private JPanel topMainPanel;

	private JPanel centerPanel;

	private JPanel addressPanel;

	private JButton adrAddresButton;

	private JButton adrPhoneButton;

	private JButton adrBirthButton;

	private JLabel adrStreetLabel;

	private JLabel adrLocalityLabel;

	private JLabel adrPhoneHomeLabel;

	private JLabel adrPhoneMobileLabel;

	private JLabel adrBirthLabel;

	private CellConstraints cc;

	private JButton adrUcmButton;

	private JLabel adrUcmLabel;

	private JPanel secondColumnPanel;

	private JPanel infoPanel;

	private JButton infoLastVisitButton;

	private JLabel infoLastVisitLabel;

	private JPanel iconPanel;

	private JButton iconWarningMemoButton;

	private JButton iconThirdPartyButton;

	private JPanel topPanel;
	
	private HistoryViewTab historyViewTab;

	private JSplitPane splitPane;
	
	private Patient patient;
	
	private JButton iconOpenInvoiceButton;

	public OverviewPanel(PatientPanel patientPanel) {
		/* ================================================== */
		this.patientPanel = patientPanel;
		this.setIcon(PatientManagerModule
				.getMediumIcon(PatientManagerModule.ANAMNESIS));
		this.setTitle(Translatrix.getTranslationString("patient.history"));
		initComponents();

		buildPanel();
		/* ================================================== */
	}

	private void buildPanel() {
		/* ================================================== */
		this.setLayout(new BorderLayout(8, 1));
		this.setBackground(DARK_BACK);
		/* ------------------------------------------------------- */
		// a panel for the top
		this.topPanel = new JPanel(new BorderLayout());
		topPanel.setBackground(BLUE_BACK);
		topPanel.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0,
				Color.BLACK));
		/* ------------------------------------------------------- */
		// a panel for the bottom
		this.centerPanel = new JPanel();
		/* ------------------------------------------------------- */
		this.cc = new CellConstraints();

		// -------------------------------------------------------
		buildSecondColumn();
		// -------------------------------------------------------
		this.topMainPanel = new JPanel();
		// layout for top panel

		topMainPanel.setLayout(new FormLayout(
		// cols
				"3dlu," + "fill:pref," + "5dlu," + "fill:pref:grow," + "0dlu",
				// rows
				"3dlu," + "fill:pref," + "2dlu"));

		topMainPanel.setOpaque(false);

		
		// -------------------------------------------------------
		buildAddressPanel();
		topMainPanel.add(addressPanel, cc.xy(2, 2));
		topMainPanel.add(secondColumnPanel, cc.xy(4, 2));
		// -------------------------------------------------------
		
		topPanel.add(topMainPanel, BorderLayout.CENTER);
		// ============================================================
		// center panel
		// ============================================================
		buildCenterPanel();

		topPanel.setOpaque(false);

		this.add(GradientFactory.makeGradient(topPanel,
				new Color(207, 226, 255), LIGHT_BACK), BorderLayout.NORTH);
		this.add(centerPanel, BorderLayout.CENTER);
		/* ================================================== */
	}

	/**
	 * 
	 */
	private void buildCenterPanel() {
		// ==================================================
		this.centerPanel.setLayout(new BorderLayout());
		centerPanel.setOpaque(false);
		// -------------------------------------------------------
		// the view will consist of one Splitpanes.
		// -------------------------------------------------------
		this.splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
		splitPane.setOneTouchExpandable(false);
		splitPane.setOpaque(false);

		// ===============================================================
		// create the panel for the history view
		// ===============================================================
		HeaderedPanel historyPanel = new HeaderedPanel(historyViewTab);
		historyPanel.setOpaque(false);
		historyPanel.setHeader(Translatrix.getTranslationString("patient.history"));
		// -------------------------------------------------------
		//splitPane.setRightComponent(historyPanel);
		splitPane.setBottomComponent(historyPanel);

		// ==========================================================
		// important data panel
		// ==========================================================
		HeaderedPanel ip = patientPanel.getImportantDataModel().getImportantDataPanelView(false);
		
//		ip.setPreferredSize(new Dimension(200,200));
		
		splitPane.setTopComponent(ip);
		
		// ==========================================================
		splitPane.setDividerSize(8);
		splitPane.setDividerLocation(200);
		centerPanel.add(splitPane);
		// ==========================================================
	}

	/**
	 * 
	 */
	private void initComponents() {
		/* ================================================== */
		// elements of the historique medical
		/* ------------------------------------------------------- */
		this.historyViewTab = new HistoryViewTab(patientPanel);
		historyViewTab.setBorder(null);
	}


	/**
	 * 
	 */
	private void buildAddressPanel() {
		/* ================================================== */
		this.addressPanel = new JPanel(new FormLayout(

		// cols
				"fill:pref," + "2dlu," + "fill:pref," + "5dlu," + "fill:pref,"
						+ "2dlu," + "fill:pref," + "5dlu," + "fill:pref,"
						+ "2dlu," + "fill:pref," + "2dlu",

				// rows
				"fill:pref," + "fill:pref," + "fill:pref"));

		addressPanel.setBackground(LIGHT_BACK);
		addressPanel.setBorder(BorderFactory.createLineBorder(LIGHT_BORDER));
		/* ------------------------------------------------------- */
		// add the buttons and icons
		this.adrAddresButton = createButton(PatientManagerModule
				.getIcon(PatientManagerModule.BW_ADDRESS), CMD_ADDRESS, this);
		this.adrPhoneButton = createButton(PatientManagerModule
				.getIcon(PatientManagerModule.BW_PHONE), CMD_PHONE, this);
		this.adrBirthButton = createButton(PatientManagerModule
				.getIcon(PatientManagerModule.BW_BIRTH), CMD_BIRTH, this);
		// create the ucm button
		this.adrUcmButton = createButton(PatientManagerModule
				.getIcon(PatientManagerModule.UCM), CMD_UCM, this);

		/* ------------------------------------------------------- */

		/* ------------------------------------------------------- */
		// create the labels for the content
		this.adrStreetLabel = new JLabel();
		this.adrLocalityLabel = new JLabel();
		this.adrPhoneHomeLabel = new JLabel();
		this.adrPhoneMobileLabel = new JLabel();
		this.adrBirthLabel = new JLabel();
		this.adrUcmLabel = new JLabel("Bla");
		/* ------------------------------------------------------- */

		/* ------------------------------------------------------- */
		// assemble the panel
		addressPanel.add(adrAddresButton, cc.xy(1, 1));
		addressPanel.add(adrStreetLabel, cc.xy(3, 1));
		addressPanel.add(adrLocalityLabel, cc.xy(3, 2));

		addressPanel.add(adrPhoneButton, cc.xy(5, 1));
		addressPanel.add(adrPhoneHomeLabel, cc.xy(7, 1));
		addressPanel.add(adrPhoneMobileLabel, cc.xy(7, 2));

		addressPanel.add(adrBirthButton, cc.xy(9, 1));
		addressPanel.add(adrBirthLabel, cc.xy(11, 1));

		addressPanel.add(adrUcmButton, cc.xy(9, 2));
		addressPanel.add(adrUcmLabel, cc.xy(11, 2));
		/* ================================================== */
	}

	/**
	 * Build second row panel
	 */
	private void buildSecondColumn() {
		/* ================================================== */
		this.secondColumnPanel = new JPanel(new FormLayout(
		// cols
				"fill:pref," + "fill:pref",
				// rows
				"fill:pref:grow"));
		secondColumnPanel.setOpaque(false);
		/* ------------------------------------------------------- */

		// JSeparator js = new JSeparator();
		// secondColumnPanel.add(js, cc.xy(1, 2));
		/* ------------------------------------------------------- */
		// a panel for the information icons
		this.iconPanel = new JPanel(new FormLayout(
		// cols
				"0dlu," + "fill:pref," + "2dlu," + "fill:pref," + "2dlu,"
						+ "fill:pref," + "2dlu",
		// rows
				"fill:pref"));
		iconPanel.setOpaque(false);
		/* ------------------------------------------------------- */
		this.iconWarningMemoButton = createButton(PatientManagerModule
				.getIcon(PatientManagerModule.WARNING), CMD_MEMOWARNING, this);
		this.iconThirdPartyButton = createButton(PatientManagerModule
				.getIcon(PatientManagerModule.COPY_PATIENT), CMD_THIRDPARTY,
				this);
		/* ------------------------------------------------------- */
		// open invoice button
		this.iconOpenInvoiceButton = createButton(PatientManagerModule
				.getIcon(PatientManagerModule.OPEN_INVOICE), CMD_OPEN_INVOICE,
				this);
		/* ------------------------------------------------------- */
		iconPanel.add(this.iconWarningMemoButton, cc.xy(2, 1));
		iconPanel.add(this.iconThirdPartyButton, cc.xy(4, 1));
		iconPanel.add(this.iconOpenInvoiceButton, cc.xy(6, 1));
		/* ------------------------------------------------------- */
		secondColumnPanel.add(iconPanel, cc.xy(1, 1));
		/* ------------------------------------------------------- */
		// create a panel for comment and last visit
		this.infoPanel = new JPanel(new FormLayout(
		// cols
				"0dlu," + "fill:pref," + "2dlu," + "fill:60dlu," + "2dlu",
		// rows
				"fill:pref"));
		infoPanel.setOpaque(false);
		/* ------------------------------------------------------- */
		// last visit
		this.infoLastVisitButton = createButton(IconFetcher
				.getScaledIcon(PatientManagerModule.class,
						PatientManagerModule.LAST_VISIT, 32), CMD_LASTVISIT,
				this);
		this.infoLastVisitLabel = new JLabel();

		this.infoLastVisitLabel.setFont(LABEL_FONT);
		this.infoLastVisitLabel.setBorder(BorderFactory
				.createLineBorder(LIGHT_BORDER));

		/* ------------------------------------------------------- */
		this.infoLastVisitButton.setToolTipText(Translatrix
				.getTranslationString("pm.lastVisit"));
		this.infoLastVisitLabel.setToolTipText(Translatrix
				.getTranslationString("pm.lastVisit"));
		/* ------------------------------------------------------- */

		this.infoPanel.add(infoLastVisitButton, cc.xy(2, 1));
		this.infoPanel.add(infoLastVisitLabel, cc.xy(4, 1));
		/* ------------------------------------------------------- */

		secondColumnPanel.add(infoPanel, cc.xy(2, 1));
		/* ================================================== */
	}

	/**
	 * Creates a button that has an icon, small insets etc
	 * 
	 * @param icon
	 * @param cmd
	 * @param l
	 * @return
	 */
	public static JButton createButton(Icon icon, String cmd, ActionListener l) {
		return createButton(icon, cmd, l, true);
	}

	/**
	 * @param icon
	 * @param cmd
	 * @param l
	 * @param effects
	 * @return
	 */
	public static JButton createButton(Icon icon, String cmd, ActionListener l,
			boolean effects) {
		/* ================================================== */
		final JButton b = new JButton(icon);
		b.setActionCommand(cmd);
		b.addActionListener(l);
		b.setMargin(new java.awt.Insets(0, 0, 0, 0));

		if (effects) {
			b.setFocusable(false);

			b.setContentAreaFilled(false);
			b.setBorderPainted(false);

			b.addMouseListener(new MouseAdapter() {

				@Override
				public void mouseEntered(MouseEvent e) {
					/* ====================================================== */
					b.setBorderPainted(true);
					b.setContentAreaFilled(true);
					/* ====================================================== */
				}

				@Override
				public void mouseExited(MouseEvent e) {
					/* ====================================================== */
					b.setBorderPainted(false);
					b.setContentAreaFilled(false);
					/* ====================================================== */
				}
			});
		}

		return b;
		/* ================================================== */
	}

	/**
	 * @param icon
	 * @param cmd
	 * @param l
	 * @param effects
	 * @return
	 */
//	private JToggleButton createToggleButton(Icon icon, String cmd,
//			ActionListener l, boolean effects) {
//		/* ================================================== */
//		final JToggleButton b = new JToggleButton(icon);
//		b.setActionCommand(cmd);
//		b.addActionListener(l);
//		b.setMargin(new java.awt.Insets(0, 0, 0, 0));
//
//		if (effects) {
//			b.setFocusable(false);
//
//			b.setContentAreaFilled(false);
//			b.setBorderPainted(false);
//
//			b.addMouseListener(new MouseAdapter() {
//
//				@Override
//				public void mouseEntered(MouseEvent e) {
//					/* ====================================================== */
//					b.setBorderPainted(true);
//					b.setContentAreaFilled(true);
//					/* ====================================================== */
//				}
//
//				@Override
//				public void mouseExited(MouseEvent e) {
//					/* ====================================================== */
//					b.setBorderPainted(false);
//					b.setContentAreaFilled(false);
//					/* ====================================================== */
//				}
//			});
//		}
//		b.setSelected(true);
//		return b;
//		/* ================================================== */
//	}

	/**
	 * @param l
	 * @param text
	 */
	private void fillLabel(JLabel l, String text) {
		/* ================================================== */
		l.setText(FORMAT_LIGHT + text);
		/* ================================================== */
	}

	// ====================================================================
	// functions that are not used to build up the gui components
	//
	// ====================================================================

	@Override
	public void setPatient(Patient patient) {
		/* ================================================== */
		this.patient = patient;
		if (patient == null) {
			/* ------------------------------------------------------- */
		} else {
			/* ------------------------------------------------------- */
			updateForms();
			historyViewTab.setPatient(patient);
		}

		/* ================================================== */
	}

	/**
	 * Updates the state of the icons
	 */
	public void updateForms() {
		/* ================================================== */
		this.updateField(CMD_ADDRESS);
		this.updateField(CMD_PHONE);
		this.updateField(CMD_JOB);

		this.updateField(CMD_BIRTH);
		this.updateField(CMD_UCM);
		this.updateField(CMD_MEMOWARNING);

		this.updateField(CMD_THIRDPARTY);
		this.updateField(CMD_OPEN_INVOICE);
		/* ------------------------------------------------------- */
		// historyTablePanel.setPatient(patient);
		// antecedentsPanel.setPatient(patient);
		/* ------------------------------------------------------- */
		this.updateField(CMD_LASTVISIT);
		/* ================================================== */
	}
	
	/**
	 * @param cmd
	 */
	@SuppressWarnings("unchecked")
	public void updateField(String cmd) {
		/* ================================================== */
		if (CMD_ADDRESS.equals(cmd)) {

			/* ------------------------------------------------------- */
			PatientAddress pa = patient.getHomeAddress();
			if (pa != null) {
				/* ------------------------------------------------------- */
				StringBuffer buff = new StringBuffer();
				// housenumber
				buff.append((pa.getStreetNumber() != null ? pa
						.getStreetNumber()
						+ ", " : ""));
				// street name
				buff.append((pa.getStreetName() != null ? pa.getStreetName()
						: ""));
				// linebreak
				// buff.append("<br>");
				// buff.append(createPostAddress(pa));
				// ----

				String locality = new String(pa.getZip() + " "
						+ pa.getLocality() + " " + pa.getCountry());

				fillLabel(this.adrStreetLabel, buff.toString());
				fillLabel(this.adrLocalityLabel, locality);
				// fillLabel(this.adrCountryLabel,pa.getCountry());
				/* ------------------------------------------------------- */
			}
		}
		if (CMD_PHONE.equals(cmd)) {
			// ===================================================
			// set the phone number home and mobile
			// ===================================================
			String phoneHome = new String("");
			String phoneMobile = new String("");
			PatientPhone pp;

			pp = patient.getPhone(PhoneType.HOME);
			if (pp != null)
				phoneHome = pp.getNumber();

			pp = patient.getPhone(PhoneType.MOBILE);
			if (pp != null)
				phoneMobile = pp.getNumber();

			fillLabel(adrPhoneHomeLabel, phoneHome);
			fillLabel(adrPhoneMobileLabel, phoneMobile);
		}
		// if (CMD_JOB.equals(cmd)) {//redesign
		// ===================================================
		// set the job information
		// ===================================================
		// fillLabel(adrJobLabel, (patient.getJob() != null ? patient.getJob() :
		// ""));//redesign
		// }//redesign
		if (CMD_BIRTH.equals(cmd)) {
			// ===================================================
			// set the birthday information
			// ===================================================
			fillLabel(adrBirthLabel,
					(patient.getBirthDate() != null ? birthFormat
							.format(patient.getBirthDate())
							+ "  ("
							+ DateUtil.extractAge(patient.getBirthDate()) + ")"
							: ""));
		}
		if (CMD_UCM.equals(cmd)) {
			// ====================================================
			// set the health insurance information
			// ====================================================
			StringBuffer iBuff = new StringBuffer();
			iBuff.append((patient.getInsurance() != null ? patient
					.getInsurance().toString() : ""));
			iBuff.append((patient.getComplementary() != null ? patient
					.getComplementary().toString() : ""));

			fillLabel(adrUcmLabel, iBuff.toString());
			/* ------------------------------------------------------- */
		}

		if (CMD_THIRDPARTY.equals(cmd)) {
			// ==========================================================
			// enable the thirdpartypayer icon
			// ==========================================================
			if (patient.getContacts() != null
					&& patient.getContacts().size() > 0) {
				this.iconThirdPartyButton.setEnabled(true);
				this.iconThirdPartyButton.setToolTipText(Translatrix
						.getTranslationString("pm.simple.thirdParty"));
			} else {
				this.iconThirdPartyButton.setEnabled(false);
				this.iconThirdPartyButton.setToolTipText(Translatrix
						.getTranslationString("pm.simple.noThirdParty"));
			}
			/* ------------------------------------------------------- */
		}
		if (CMD_MEMOWARNING.equals(cmd)) {
			// ==========================================================
			// get the memo count from cache
			// ==========================================================
			Long mCount = (Long) PatientManagerModule.getCache(
					OverviewPanel.this.patient).get(
					PatientManagerModule.CACHE_PATIENT_MEMO_COUNT);
			if (mCount != null && mCount > 0) {
				iconWarningMemoButton.setIcon(GECAMedModule
						.getIcon(GECAMedIconNames.MEMO));
				iconWarningMemoButton.setToolTipText(Translatrix
						.getTranslationString("pm.simple.memo"));
			} else {
				iconWarningMemoButton.setIcon(GECAMedModule
						.getIcon(GECAMedIconNames.MEMO_EMPTY));
				iconWarningMemoButton.setToolTipText(Translatrix
						.getTranslationString("pm.simple.noMemo"));
			}
			// get the warning memos from the patient cache
			Collection<PatientMemo> iMemos = (Collection<PatientMemo>) PatientManagerModule
					.getCache(OverviewPanel.this.patient).get(
							PatientManagerModule.CACHE_PATIENT_WARNING_MEMOS);
			/* ------------------------------------------------------- */
			if (iMemos != null && iMemos.size() > 0) {
				iconWarningMemoButton.setIcon(PatientManagerModule
						.getIcon(PatientManagerModule.WARNING));
				iconWarningMemoButton.setToolTipText(Translatrix
						.getTranslationString("pm.simple.warningMemo"));
			}
			/* ------------------------------------------------------- */
		}

		if (CMD_LASTVISIT.equals(cmd)) {
			// =========================================================
			// obtain the last visit
			// =========================================================
			Incident lastIncident = patientPanel.getHistoryDataAgent()
					.getLatestIncident();
			Date last = null;
			if (lastIncident != null)
				last = lastIncident.getIncidentDate();
			if (last != null) {
				this.infoLastVisitLabel.setText(" "
						+ lastVisitFormat.format(last));
				this.infoLastVisitButton.setEnabled(true);
			} else {
				this.infoLastVisitLabel.setText(" ");
				this.infoLastVisitButton.setEnabled(false);
			}
			/* ------------------------------------------------------- */
		}
		if (CMD_OPEN_INVOICE.equals(cmd)) {
			// ==============================================================
			// 
			PatientAdminInterface pManager = (PatientAdminInterface) ManagerFactory
					.getRemote(PatientAdminBean.class);
			/* ------------------------------------------------------- */
			Long openInvoice = pManager.getOpenInvoices(patient.getId());
			if (openInvoice != null && openInvoice > 0) {
				this.iconOpenInvoiceButton.setEnabled(true);
				this.iconOpenInvoiceButton.setToolTipText(Translatrix
						.getTranslationString("pm.simple.openInvoice"));
			} else {
				this.iconOpenInvoiceButton.setEnabled(false);
				this.iconOpenInvoiceButton.setToolTipText(Translatrix
						.getTranslationString("pm.simple.noOpenInvoice"));
			}
			/* ------------------------------------------------------- */
		}

		/* ================================================== */
	}

	@Override
	public void saveData() {
		/* ================================================== */
		this.patientPanel.getImportantDataModel().saveData();
		/* ================================================== */

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */

	public void actionPerformed(ActionEvent e) {

		if (CMD_ADDRESS.equals(e.getActionCommand())
				|| CMD_PHONE.equals(e.getActionCommand())
				|| CMD_JOB.equals(e.getActionCommand())
				|| CMD_BIRTH.equals(e.getActionCommand())
				|| CMD_UCM.equals(e.getActionCommand())) {
			// show admin panel
			patientPanel.showPanel(PatientAdminPanel.NAME);
		}

		if (CMD_MEMOWARNING.equals(e.getActionCommand())) {
			// -------------------------------------------------------
			SimpleMemoView sm = new SimpleMemoView(this.patientPanel
					.getMemoControler(), this);
			sm.setPatient(patient);
			sm.showDialog(true);
			updateForms();
			// -------------------------------------------------------
		}

		// -------------------------------------------------------
		if (CMD_LASTVISIT.equals(e.getActionCommand())) {
			// -------------------------------------------------------
			// go to consultation
			patientPanel.openConsultation(patientPanel.getHistoryDataAgent()
					.getLatestIncident());
			// -------------------------------------------------------
		}

		if (CMD_OPEN_INVOICE.equals(e.getActionCommand())) {
			// -------------------------------------------------------
			// open billingModule
			MainFrame.getInstance().selectModule(BillingModule.MODULE_NAME);
			// -------------------------------------------------------
		}
		// ======================================================
	}


	/**
	 * @param translatrixString
	 * @return
	 */
	public static JLabel createLabel(String translatrixString) {
		/* ================================================== */
		JLabel l = new JLabel(Translatrix
				.getTranslationString(translatrixString)
				+ ":");
		l.setFont(LABEL_FONT);
		return l;
		/* ================================================== */
	}

	/**
	 * @param pa
	 * @return
	 */
//	public static String createPostAddress(PatientAddress pa) {
//		StringBuffer addrString = new StringBuffer();
//		if (pa != null) {
//			/* ------------------------------------------------------- */
//			// ===============================================================
//			// detect the right country format
//			// if the country of the patient is equal to the office's
//			// then just print zip and locality
//			// if not we will print the country in an extra line
//			// ===============================================================
//			String officeCountry = MainFrame.getCurrentOfficeCountry();
//			// default setting as the software is mainly for luxembourg =)
//			boolean isLuxembourg = false;
//			// default setting for adding the country name in a new line
//			boolean isAbroad = true;
//			/* ------------------------------------------------------- */
//			// check if the patients address is in luxembourg
//			if (Country.LUXEMBOURG.equals(pa.getCountry()))
//				isLuxembourg = true;
//			/* ------------------------------------------------------- */
//			// check if the patient address is in a different country
//			// then the office's address
//			if (officeCountry != null
//					&& pa.getCountry() != null
//					&& officeCountry.toUpperCase().equals(
//							pa.getCountry().toUpperCase()))
//				isAbroad = false;
//
//			// --------------------------------------------
//			// use luxembourg specific layout
//			if (isLuxembourg) { // make an L in front of the zip
//				/* ------------------------------------------------------- */
//				addrString
//						.append("L - " + pa.getZip() + " " + pa.getLocality());
//				/* ------------------------------------------------------- */
//			} else { // just zip plus locality
//				/* ------------------------------------------------------- */
//				addrString.append(pa.getZip() + " " + pa.getLocality());
//				/* ------------------------------------------------------- */
//			}
//			// ---------------------------------------------------------
//			// layout for abroad address
//			if (isAbroad) { // add the country name in english or french
//				/* ------------------------------------------------------- */
//				// get the translation
//				// @SuppressWarnings("unused")
//				// just to initialise the static fields in the entity
////				Country c = new Country();
//				HashMap<String, String> countryCodes = Country.addressTranslation
//						.get(officeCountry);
//				// if no translations has been found, we take the default set
//				if (countryCodes == null)
//					countryCodes = Country.addressTranslation
//							.get(Country.DEFAULT);
//				/* ------------------------------------------------------- */
//				// add the additional line
//				String country = countryCodes.get(pa.getCountry());
//				// if no country translation was found, we use the
//				// data from the PatientAddress object
//				if (country == null)
//					country = pa.getCountry();
//				// add the line
//				addrString.append("<br>" + country);
//				/* ------------------------------------------------------- */
//			}
//		}
//		return addrString.toString();
//	}

	/**
	 * @return
	 */
	public Incident getSelectedIncident() {
		/* ====================================================== */
		return this.historyViewTab.getSelectedIncident();
		/* ====================================================== */
	}

	@Override
	public String getName() {
		return NAME;
	}

}
