/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/**
 *
 */
package lu.tudor.santec.gecamed.prescription.gui.drug;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextArea;
import javax.swing.UIManager;
import javax.swing.border.BevelBorder;
import javax.swing.border.CompoundBorder;
import javax.swing.border.LineBorder;
import javax.swing.border.SoftBevelBorder;
import javax.swing.border.TitledBorder;
import javax.swing.filechooser.FileFilter;

import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.IChangeListener;
import lu.tudor.santec.gecamed.core.gui.utils.WindowToolbox;
import lu.tudor.santec.gecamed.core.gui.widgets.ModuleHeader;
import lu.tudor.santec.gecamed.prescription.ejb.entity.beans.drug.DrugList;
import lu.tudor.santec.gecamed.prescription.gui.PrescriptionIcons;
import lu.tudor.santec.gecamed.prescription.gui.drug.lists.cefip.CefipDrugListModule;
import lu.tudor.santec.gecamed.prescription.gui.drug.lists.cefip.DrugFileModel;
import lu.tudor.santec.gecamed.prescription.gui.util.VTextField;
import lu.tudor.santec.gecamed.prescription.gui.widgets.drugbrowser.DrugBrowser;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.commons.io.FilenameUtils;

import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;


/**
 *
 * Dialog to import the CEFIP list.
 * Only admin can access this dialog.
 * The user can specify a directory that contains the DBF files or define each file. Selected files are tested for
 * integrity. If it is not the expected file, the dialog marks it red and disables the import button.
 * Each time a list is imported, the old one is deleted.
 *
 * @author martin.heinemann@tudor.lu
 *
 *
 * @version
 * <br>$Log: ImportDrugsDialog.java,v $
 * <br>Revision 1.11  2013-02-22 08:46:50  kutscheid
 * <br>remove the tupel classes (please redeploy)
 * <br>
 * <br>Revision 1.10  2010-03-30 10:48:51  troth
 * <br>Redesign of the prescription view - change icon manager class
 * <br>
 * <br>Revision 1.9  2009-05-07 15:56:26  heinemann
 * <br>import of new cefip file format in ms access.
 * <br>using lib jackccess
 * <br>
 * <br>Revision 1.8  2008-09-25 09:43:08  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.7  2008-02-11 16:51:50  heinemann
 * <br>moved IChangeListener to core
 * <br>
 * <br>Revision 1.6  2008-01-15 09:58:55  heinemann
 * <br>better documentation
 * <br>
 * <br>Revision 1.5  2007/08/01 07:09:01  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.4  2007/07/17 13:44:47  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.3  2007/07/03 12:35:29  heinemann
 * <br>system.out removed
 * <br>
 * <br>Revision 1.2  2007/03/02 08:28:42  hermen
 * <br>initial checkin after the merge of PatientModuleRebuild with the main HEAD
 * <br>
 * <br>Revision 1.1.2.3  2007/02/21 10:00:40  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.1.2.2  2006/12/22 12:07:23  heinemann
 * <br>Cefip import dialog enabled
 * <br>
 * <br>Revision 1.1.2.1  2006/12/20 13:47:04  heinemann
 * <br>*** empty log message ***
 * <br>
 */
public class ImportDrugsDialog extends JDialog{

	private static final long serialVersionUID = 1L;


	private final boolean OPAQUE = false;


	private JPanel mainPanel;
	private ModuleHeader titlePanel;

	private JButton closeButton;
	private JPanel centerPanel;
	private JPanel listHeaderPanel;

	private Color borderColor = UIManager.getDefaults().getColor("Button.focus");

	private DateFormat dateFormat = DateFormat.getDateTimeInstance(
			DateFormat.SHORT,
			DateFormat.SHORT,
			Translatrix.getLocale());


	private JLabel listInfoDateLabel;


	private JLabel atcCountLabel;


	private CellConstraints cc;


	private JLabel manuCountLabel;


	private JLabel drugCountLabel;


	private JPanel logPanel;


	private JTextArea logArea;


	private JPanel importPanel;


	private JLabel helpLabel;


	private VTextField cefipDirTextField;


	private JButton cefipDirChooserButton;


	private JButton advancedButton;


	private JButton importButton;


	private JPanel advancedPanel;


	private VTextField atcTextField;


	private VTextField drugTextField;


	private VTextField manuTextField;


	private JButton atcButton;


	private JButton manuButton;


	private JButton drugButton;

	private boolean isAdvancedMode = false;


	private JPanel advancedMetaPanel;


	private JButton advancedImportButton;


	private JPanel folderPanel;


	private JFileChooser fileChooser;


	private DrugFileModel drugFileModel;


	private AbstractAction importAction;


	private JPanel progressPanel;


	private JProgressBar progressBar;


	private JButton importCancelButton;


	private JPanel listInfoPanel;


	private JButton deleteButton;


	private AbstractAction deleteAction;


	public ImportDrugsDialog(Frame parent) {
		/* ====================================================== */
		super(parent, true);
		Translatrix.addBundle("lu.tudor.santec.gecamed.prescription.gui.resources.Translatrix");
		// the model
		this.drugFileModel = new DrugFileModel();

		initComponent();
		initActions();
		initListeners();


		updateListInfo();
		this.add(mainPanel);
		/* ====================================================== */
	}

	/* ******************************************************************
	 * controll methods
	 */
	/* ----------------------------------------------------------------- */


	private void updateListInfo() {
		/* ====================================================== */
		// get the current list
		DrugList drugList = CefipDrugListModule.getManager().getList();

		// get the lists sizes
		ArrayList<Integer> trippel = CefipDrugListModule.getManager().getListCounts();

		this.setListInfo(drugList.getDate(),
				trippel.get(2), trippel.get(0), trippel.get(1));
		/* ====================================================== */
	}


	/**
	 * Sets the info from the current cefip list that is in the database
	 * to the panel
	 *
	 * @param listDate
	 * @param atcCodes
	 * @param manufacurers
	 * @param drugs
	 */
	public void setListInfo(Date listDate, int atcCodes, int manufacurers, int drugs) {
		/* ====================================================== */
		this.listInfoDateLabel.setText("<html><b>"+
							Translatrix.getTranslationString("cefip.importer.listDate") +
							": </b><br>"+(listDate != null ? dateFormat.format(listDate) : ""));
		// the atc codes
		atcCountLabel.setText(""+atcCodes);
		// the manufacturers
		manuCountLabel.setText(""+manufacurers);
		// the drugs
		drugCountLabel.setText(""+drugs);
		// refresh

		listInfoDateLabel.validate();
		listInfoDateLabel.updateUI();

		atcCountLabel.validate();
		atcCountLabel.updateUI();

		manuCountLabel.validate();
		manuCountLabel.updateUI();

		drugCountLabel.validate();
		drugCountLabel.updateUI();

		/* ====================================================== */
	}


	/**
	 * Adds a new row to the log area
	 *
	 * @param log
	 */
	public void addLogRow(String log) {
		/* ====================================================== */
		if (log != null && !log.equals("")) {
			this.logArea.setText(logArea.getText()+"\n"+log);
		}
		/* ====================================================== */
	}

	/**
	 * Clear the log
	 */
	public void clearLog() {
		/* ====================================================== */
		this.logArea.setText("");
		/* ====================================================== */
	}




	/**
	 * Delets the data of a list
	 */
	private void deleteListData() {
		/* ================================================== */
//		 delete ATC classes
		progressBar.setIndeterminate(true);
		importCancelButton.setEnabled(true);
		progressBar.setString(
				Translatrix.getTranslationString("cefip.importer.atc.deleteClasses"));
		drugFileModel.getWorker().deleteAllCefipAtcClasses();
		/* ------------------------------------------------------ */
		updateListInfo();
		/* ------------------------------------------------------ */
		// delete manufacturer
		progressBar.setString(
				Translatrix.getTranslationString("cefip.importer.manufacturer.delete"));
		drugFileModel.getWorker().deleteAllManufacturers();
		/* ------------------------------------------------------ */
		updateListInfo();
		/* ------------------------------------------------------ */
		// delete drugs
		progressBar.setString(
				Translatrix.getTranslationString("cefip.importer.drug.delete"));
		drugFileModel.getWorker().deleteAllCefipDrugs();
		/* ------------------------------------------------------ */
		/* ------------------------------------------------------ */
		progressBar.setString("");
		progressBar.setIndeterminate(false);
		updateListInfo();
		DrugBrowser.getInstance().triggerUpdate();
		/* ================================================== */
	}


	/**
	 * start the import
	 */
	private void startImport() {
		/* ====================================================== */
		deleteListData();
		/* ------------------------------------------------------ */
		// start import
		progressBar.setIndeterminate(false);
		progressBar.setString(Translatrix.getTranslationString("cefip.importer.atc.import"));
		drugFileModel.getWorker().importAtcClasses(progressBar, drugFileModel.getCefipVersion());
		/* ------------------------------------------------------ */
		updateListInfo();
		/* ------------------------------------------------------ */
		progressBar.setString(Translatrix.getTranslationString("cefip.importer.manufacturer.import"));
		drugFileModel.getWorker().importManufacturer(progressBar, drugFileModel.getCefipVersion());
		/* ------------------------------------------------------ */
		updateListInfo();
		/* ------------------------------------------------------ */
		progressBar.setString(Translatrix.getTranslationString("cefip.importer.drug.import"));
		drugFileModel.getWorker().importCefipList(progressBar, drugFileModel.getCefipVersion());
		/* ------------------------------------------------------ */
		updateListInfo();
		/* ------------------------------------------------------ */

//		progressPanel.setVisible(false);
		progressBar.setEnabled(false);
		progressBar.setString(Translatrix.getTranslationString("cefip.importer.finished"));
		progressBar.setIndeterminate(false);
		importCancelButton.setEnabled(false);

		importAction.setEnabled(true);
		enableChooserButtons(true);
		DrugBrowser.getInstance().triggerUpdate();
		/* ====================================================== */
	}

	

	/**
	 * Enable just the filechooser buttons
	 * @param b
	 */
	private void enableChooserButtons(boolean b) {
		/* ====================================================== */
		cefipDirChooserButton.setEnabled(b);
		atcButton.setEnabled(b);
		manuButton.setEnabled(b);
		drugButton.setEnabled(b);
		/* ====================================================== */
	}


	public static void main(String[] args) {
		new ImportDrugsDialog(new JFrame()).setVisible(true);
	}

	/* ----------------------------------------------------------------- */
	/* End controll methods
	 *
	 * ***************************************************************** */

	/* *****************************************************************
	 * Actions
	 */
	/* ----------------------------------------------------------------- */
	private void initActions() {
		/* ====================================================== */
		// first create a fileChooser
		this.fileChooser = new JFileChooser();
		this.fileChooser.setMultiSelectionEnabled(false);

		/* ------------------------------------------------------ */
		// the fileChooserButton for the CEFIP __DIR__
		/* ------------------------------------------------------ */
		cefipDirChooserButton.setAction(new AbstractAction("...") {

			private static final long serialVersionUID = 1L;
			private JFileChooser jf = ImportDrugsDialog.this.fileChooser;

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				// open the filechooser in dir_only mode
				jf.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
				jf.setFileFilter(new CefipFileFilter());
				
				jf.setSelectedFile(null);
				// show
				jf.showOpenDialog(WindowToolbox.getOwnerFrame(cefipDirChooserButton));
				/* ------------------------------------------------------ */
				// set the selected file
				if (jf.getSelectedFile() != null)
					drugFileModel.setCefipDir(jf.getSelectedFile());
				/* ------------------------------------------------------ */
				/* ============================================= */
			}
		});
		/* ---------------------------------------------------------------------- */


		/* ------------------------------------------------------ */
		// the atc file chooser button
		/* ------------------------------------------------------ */
		atcButton.setAction(new AbstractAction("...") {

			private static final long serialVersionUID = 1L;
			private JFileChooser jf = ImportDrugsDialog.this.fileChooser;

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				// open the filechooser in dir_only mode
				jf.setFileSelectionMode(JFileChooser.FILES_ONLY);
				jf.setSelectedFile(null);
				// show
				jf.showOpenDialog(WindowToolbox.getOwnerFrame(atcButton));
				/* ------------------------------------------------------ */
				// set the selected file
				if (jf.getSelectedFile() != null) {
					drugFileModel.setAtcFile(jf.getSelectedFile().getAbsolutePath());
					drugFileModel.fireModelChanged();
				}
				/* ------------------------------------------------------ */
				/* ============================================= */
			}
		});
		/* ---------------------------------------------------------------------- */

		/* ------------------------------------------------------ */
		// the manufacturer file chooser button
		/* ------------------------------------------------------ */
		manuButton.setAction(new AbstractAction("...") {

			private static final long serialVersionUID = 1L;
			private JFileChooser jf = ImportDrugsDialog.this.fileChooser;

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				// open the filechooser in dir_only mode
				jf.setFileSelectionMode(JFileChooser.FILES_ONLY);
				jf.setSelectedFile(null);
				// show
				jf.showOpenDialog(WindowToolbox.getOwnerFrame(manuButton));
				/* ------------------------------------------------------ */
				// set the selected file
				if (jf.getSelectedFile() != null) {
					drugFileModel.setManufacturerFile(jf.getSelectedFile().getAbsolutePath());
					drugFileModel.fireModelChanged();
				}
				/* ------------------------------------------------------ */
				/* ============================================= */
			}
		});
		/* ---------------------------------------------------------------------- */

		/* ------------------------------------------------------ */
		// the drug file chooser button
		/* ------------------------------------------------------ */
		drugButton.setAction(new AbstractAction("...") {

			private static final long serialVersionUID = 1L;
			private JFileChooser jf = ImportDrugsDialog.this.fileChooser;

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				// open the filechooser in dir_only mode
				jf.setFileSelectionMode(JFileChooser.FILES_ONLY);
				jf.setSelectedFile(null);
				// show
				jf.showOpenDialog(WindowToolbox.getOwnerFrame(drugButton));
				/* ------------------------------------------------------ */
				// set the selected file
				if (jf.getSelectedFile() != null) {
					drugFileModel.setDrugFile(jf.getSelectedFile().getAbsolutePath());
					drugFileModel.fireModelChanged();
				}
				/* ------------------------------------------------------ */
				/* ============================================= */
			}
		});
		/* ---------------------------------------------------------------------- */



		/* ------------------------------------------------------ */
		// the import button
		this.importAction = new AbstractAction(
				Translatrix.getTranslationString("cefip.importer.importButton")) {

			private static final long serialVersionUID = 1L;

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				// ask
				// show a popup with really import question.
				// and any furhter instructions
				int result =
					JOptionPane.showConfirmDialog(MainFrame.getInstance(),
						Translatrix.getTranslationString("cefip.drugeditor.reallyImportQuestion"),
						Translatrix.getTranslationString("cefip.drugeditor.import"),
						JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE);

				if (result == JOptionPane.NO_OPTION)
					return;
				else {
					/* ------------------------------------------------------ */
					// start the import
					importAction.setEnabled(false);
					enableChooserButtons(false);
					// show the panel
					progressPanel.setVisible(true);
					/* ------------------------------------------------------ */
					Thread t = new Thread() {
						public void run() {
							/* ------------------------------------------------------ */
							startImport();
							/* ------------------------------------------------------ */
						}
					};
					t.start();
					/* ------------------------------------------------------ */
			}
				/* ============================================= */
			}
		};
		/* ---------------------------------------------------------------------- */
		/* ------------------------------------------------------ */
		// the delete list button action
		/* ------------------------------------------------------ */
		deleteAction = new AbstractAction(
				Translatrix.getTranslationString("cefip.drugeditor.deleteList"),
				PrescriptionIcons.getMediumIcon(PrescriptionIcons.DELETE)) {

			private static final long serialVersionUID = 1L;
//			private JFileChooser jf = ImportDrugsDialog.this.fileChooser;

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				int result =
					JOptionPane.showConfirmDialog(MainFrame.getInstance(),
						Translatrix.getTranslationString("cefip.drugeditor.reallyDeleteQuestion"),
						Translatrix.getTranslationString("prescription.buttons.clear"),
						JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE);

				if (result == JOptionPane.NO_OPTION)
					return;
				else {
					/* ------------------------------------------------------- */
					Thread t = new Thread() {
						public void run() {
							enableChooserButtons(false);
							deleteListData();
							enableChooserButtons(true);
						}
					};
					t.start();
					/* ------------------------------------------------------- */
				}

				/* ------------------------------------------------------ */
				/* ============================================= */
			}
		};
		this.deleteButton.setAction(deleteAction);
		/* ---------------------------------------------------------------------- */




		// set
		importButton.setAction(this.importAction);
		advancedImportButton.setAction(this.importAction);
		importAction.setEnabled(false);

		// cancel import
		importCancelButton.setAction(new AbstractAction(Translatrix.getTranslationString("prescription.core.cancel")) {

			private static final long serialVersionUID = 1L;

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				System.out.println("TODO cancel import action");
				/* ============================================= */
			}

		});
		importCancelButton.setEnabled(false);
		/* ====================================================== */
	}

	/* --------------------------------------------------------------------
	 * End actions
	 *
	 * ********************************************************************/

	/* ******************************************************************
	 * Listener
	 */
	/* ----------------------------------------------------------------- */

	/**
	 * Inits the listener for the communication between model and gui
	 */
	private void initListeners() {
		/* ====================================================== */
		// the cefip dir textfield
		this.drugFileModel.addChangeListener(new IChangeListener() {

			public void fireEvent() {
				/* ============================================= */
				// first get the text
				cefipDirTextField.setText(drugFileModel.getCefipDir());
				// get the state
				if (drugFileModel.isAllOk()) {
					// green
					cefipDirTextField.setState(VTextField.STATE_TRUE);
					// enable importButton action
					importAction.setEnabled(true);
				} else {
					cefipDirTextField.setState(VTextField.STATE_FALSE);
					importAction.setEnabled(false);
				}
				/* ============================================= */
			}
		});
		/* ------------------------------------------------------ */

		/* ------------------------------------------------------ */
		// the atc textfield
		/* ------------------------------------------------------ */
		this.drugFileModel.addChangeListener(new IChangeListener() {

			public void fireEvent() {
				/* ============================================= */
				// first get the text
				atcTextField.setText(drugFileModel.getAtcFile());
				/* ------------------------------------------------------ */
				// check the state
				if (drugFileModel.getState(
						DrugFileModel.ATC).equals(DrugFileModel.FILE_OK)) {
					/* ------------------------------------------------------ */
					atcTextField.setState(VTextField.STATE_TRUE);
					atcTextField.setWarning("");
					/* ------------------------------------------------------ */
				} else if(drugFileModel.getState(DrugFileModel.ATC)
						.equals(DrugFileModel.FILE_NO_STATE)) {
					/* ------------------------------------------------------ */
					atcTextField.setState(VTextField.STATE_NONE);
					atcTextField.setWarning("");
					/* ------------------------------------------------------ */
				}

				else {
					/* ------------------------------------------------------ */
					atcTextField.setState(VTextField.STATE_FALSE);
					if (drugFileModel.getState(
						DrugFileModel.ATC).equals(DrugFileModel.FILE_NOT_READABLE))
						atcTextField.setWarning(Translatrix.getTranslationString("cefip.importer.atc.notReadable"));
					else
						atcTextField.setWarning("");
					/* ------------------------------------------------------ */
				}
				/* ============================================= */
			}

		});

		/* ------------------------------------------------------ */
		// the manufacturer textfield
		/* ------------------------------------------------------ */
		this.drugFileModel.addChangeListener(new IChangeListener() {

			public void fireEvent() {
				/* ============================================= */
				// first get the text
				manuTextField.setText(drugFileModel.getManufacturerFile());
				/* ------------------------------------------------------ */
				// check the state
				if (drugFileModel.getState(
						DrugFileModel.MANUFACTURER).equals(DrugFileModel.FILE_OK)) {
					/* ------------------------------------------------------ */
					manuTextField.setState(VTextField.STATE_TRUE);
					manuTextField.setWarning("");
					/* ------------------------------------------------------ */
				} else
					if (drugFileModel.getState(DrugFileModel.MANUFACTURER)
							.equals(DrugFileModel.FILE_NO_STATE)) {
						/* ------------------------------------------------------ */
						manuTextField.setState(VTextField.STATE_NONE);
						manuTextField.setWarning("");
						/* ------------------------------------------------------ */
					}

				else {
					/* ------------------------------------------------------ */
					manuTextField.setState(VTextField.STATE_FALSE);
					if (drugFileModel.getState(
						DrugFileModel.MANUFACTURER).equals(DrugFileModel.FILE_NOT_READABLE))
						manuTextField.setWarning(Translatrix.getTranslationString("cefip.importer.manufacturer.notReadable"));
					else
						manuTextField.setWarning("");
					/* ------------------------------------------------------ */
				}
				/* ============================================= */
			}

		});
		/* ------------------------------------------------------ */

		/* ------------------------------------------------------ */
		// the drug textfield
		/* ------------------------------------------------------ */
		this.drugFileModel.addChangeListener(new IChangeListener() {

			public void fireEvent() {
				/* ============================================= */
				// first get the text
				drugTextField.setText(drugFileModel.getDrugFile());
				/* ------------------------------------------------------ */
				// check the state
				if (drugFileModel.getState(
						DrugFileModel.DRUG).equals(DrugFileModel.FILE_OK)) {
					/* ------------------------------------------------------ */
					drugTextField.setState(VTextField.STATE_TRUE);
					drugTextField.setWarning("");
					/* ------------------------------------------------------ */
				} else
					if (drugFileModel.getState(DrugFileModel.DRUG)
							.equals(DrugFileModel.FILE_NO_STATE)) {
						/* ------------------------------------------------------ */
						drugTextField.setState(VTextField.STATE_NONE);
						drugTextField.setWarning("");
						/* ------------------------------------------------------ */
					}

				else {
					/* ------------------------------------------------------ */
					drugTextField.setState(VTextField.STATE_FALSE);
					if (drugFileModel.getState(
						DrugFileModel.DRUG).equals(DrugFileModel.FILE_NOT_READABLE))
						drugTextField.setWarning(Translatrix.getTranslationString("cefip.importer.drug.notReadable"));
					else
						drugTextField.setWarning("");
					/* ------------------------------------------------------ */
				}
				/* ============================================= */
			}

		});
		/* ------------------------------------------------------ */
		/* ====================================================== */
	}
	/* ------------------------------------------------------ */






	/* --------------------------------------------------------------------
	 * End Listener
	 *
	 * ********************************************************************/


	/* ******************************************************************
	 * GUI creation methods
	 */
	/* ----------------------------------------------------------------- */

	/**
	 * Init the components
	 */
	private void initComponent() {
		/* ============================================= */
		// the main panel
		mainPanel = new JPanel(new BorderLayout());
		this.cc = new CellConstraints();
		mainPanel.setBackground(Color.WHITE);
		/* ------------------------------------------------------ */
		// the header
		this.titlePanel = new ModuleHeader(Translatrix.getTranslationString("cefip.importer.header"),
				PrescriptionIcons.getIcon(PrescriptionIcons.SUBSTANCE),
				new Color(133, 190, 255));
		mainPanel.add(titlePanel, BorderLayout.NORTH);
		/* ------------------------------------------------------ */
		// the bottom -> two buttons
		JPanel buttonPanel = new JPanel(new FormLayout(
													   // cols
													   "3dlu," +
													   "fill:pref," +
													   "fill:pref:grow," +
													   "fill:pref," +
													   "3dlu",
													   // rows
													   "3dlu," +
													   "fill:pref," +
													   "3dlu"));
		/* ------------------------------------------------------ */
		this.closeButton = new JButton(Translatrix.getTranslationString("cefip.drugeditor.close")
										, PrescriptionIcons.getMediumIcon(PrescriptionIcons.ERROR));
		buttonPanel.add(closeButton, cc.xy(4, 2));
		/* ------------------------------------------------------- */
		this.deleteButton = new JButton();
		buttonPanel.add(deleteButton, cc.xy(2, 2));

		/* ------------------------------------------------------ */
		buttonPanel.setOpaque(OPAQUE);
		mainPanel.add(buttonPanel, BorderLayout.SOUTH);
		/* ------------------------------------------------------ */
		/* ------------------------------------------------------ */
		// the centerPanel
		this.centerPanel = new JPanel(new FormLayout(
												// cols
												"3dlu," +
												"fill:150dlu," +
												"5dlu," +
												"fill:pref:grow," +
												"3dlu",
												// rows
												"3dlu," +
												"fill:pref," +
												"3dlu," +
												"fill:pref:grow," +
												"3dlu"));
		/* ------------------------------------------------------ */
		centerPanel.setOpaque(OPAQUE);
		// left column
		initListInfoPanel();
		initLogPanel();
		this.centerPanel.add(listHeaderPanel, cc.xy(2, 2));
		this.centerPanel.add(logPanel, cc.xy(2, 4));
		/* ------------------------------------------------------ */
		// right column
		initImportPanel();
		/* ------------------------------------------------------ */
		this.centerPanel.add(importPanel, cc.xywh(4, 2, 1, 3));


		// init the davanced panel
		initAdvancedPanel();


		mainPanel.add(centerPanel, BorderLayout.CENTER);
		/* ============================================= */
	}


	/**
	 * Init import panel
	 */
	private void initImportPanel() {
		/* ============================================= */
		this.importPanel = new JPanel(new FormLayout(
												// cols
												"3dlu," +
												"fill:pref:grow," +
												"3dlu",
												// rows
												"3dlu," +			// 1
												"fill:pref," +		// 2
												"3dlu," +			// 3
												"fill:pref," +		// 4
												"3dlu," +			// 5
												"top:pref:grow," +	// 6
												"3dlu" 				// 7
												));
		/* ------------------------------------------------------ */
		// create the helpLabel
		this.helpLabel = new JLabel("<html>"+Translatrix.getTranslationString("cefip.importer.helpText"));
//		helpLabel.setFont(new Font(helpLabel.getFont().getName(),Font.PLAIN, helpLabel.getFont().getSize()));


//		helpLabel.setOpaque(true);
		// a panel for the label
		JPanel helpPanel = new JPanel(new FormLayout("3dlu,fill:pref,3dlu",
													 "3dlu,fill:pref,3dlu"));
		helpPanel.setBorder(new LineBorder(borderColor));
		helpPanel.setBackground(new Color(255,204,204));


		helpPanel.add(helpLabel,cc.xy(2, 2));
		importPanel.add(helpPanel, cc.xy(2, 2));

		/* ------------------------------------------------------ */
		/* ------------------------------------------------------ */
		// the folder panel
		this.folderPanel = new JPanel(new FormLayout(
														// cols
														"3dlu," +
														"fill:pref:grow," +
														"5dlu," +
														"fill:pref," +
														"3dlu",
														// rows
														"3dlu," +
														"fill:pref," +
														"3dlu," +
														"fill:pref," +
														"5dlu," +
														"fill:pref," +
														"3dlu," +
														"fill:pref," +
														"3dlu"));
		folderPanel.setOpaque(OPAQUE);
		/* ------------------------------------------------------ */
		// the label for the folder
		JLabel folderLabel = new JLabel(Translatrix.getTranslationString("cefip.importer.folderSelect"));
		folderPanel.add(folderLabel, cc.xywh(2, 2, 1, 1));
		/* ------------------------------------------------------ */
		// the folder textfield
		this.cefipDirTextField = new VTextField("");
		cefipDirTextField.setEditable(false);
		cefipDirTextField.setOpaque(OPAQUE);
		folderPanel.add(cefipDirTextField, cc.xy(2, 4));
		/* ------------------------------------------------------ */
		// the button to choose the folder
		this.cefipDirChooserButton = new JButton("...");
		folderPanel.add(cefipDirChooserButton, cc.xy(4, 4));

		/* ------------------------------------------------------ */

		/* ------------------------------------------------------ */
		// the advanced button
		this.advancedButton = new JButton(Translatrix.getTranslationString("cefip.importer.advancedButton"));

//		folderPanel.add(advancedButton, cc.xy(2, 6));
		/* ------------------------------------------------------ */
		// import button
		this.importButton = new JButton();
//		folderPanel.add(importButton, cc.xywh(4, 6, 3, 1));

		ButtonBarBuilder bb = new ButtonBarBuilder();
		bb.addGridded(advancedButton);
		bb.addGlue();
		bb.addGridded(importButton);
		bb.getPanel().setOpaque(OPAQUE);

		folderPanel.add(bb.getPanel(), cc.xywh(2, 6, 3, 1));
		/* ------------------------------------------------------ */
		importPanel.add(folderPanel, cc.xy(2, 4));
		/* ------------------------------------------------------ */
		importPanel.setOpaque(OPAQUE);
		/* ------------------------------------------------------ */
		closeButton.addActionListener(new ActionListener() {

			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				ImportDrugsDialog.this.setVisible(false);
				/* ============================================= */
			}

		});

		initAdvancedButtonAction();

		/* ------------------------------------------------------ */
		// the progressPanel
		/* ------------------------------------------------------ */
		this.progressPanel = new JPanel(new FormLayout(
				// cols
				"3dlu," +
				"fill:pref:grow," +
				"fill:pref," +
				"fill:pref:grow," +
				"3dlu",
				// rows
				"5dlu," +
				"fill:pref," +
				"3dlu," +
				"fill:pref," +
				"fill:pref:grow"));
		/* ------------------------------------------------------ */
		// create a Progressbar
		this.progressBar = new JProgressBar();
		progressPanel.setOpaque(OPAQUE);
		// add to metapanel
		progressPanel.add(progressBar, cc.xywh(2, 2, 3, 1));
		progressBar.setBackground(Color.WHITE);
		progressBar.setForeground(Color.RED);
//		progressBar.setIndeterminate(true);
		/* ------------------------------------------------------ */
		progressBar.setStringPainted(true);
		progressBar.setString("");
		/* ------------------------------------------------------ */
		// a cancel button
		this.importCancelButton = new JButton();

		progressPanel.add(importCancelButton, cc.xy(3, 4));


		importPanel.add(progressPanel, cc.xy(2, 6));
//		progressPanel.setVisible(false);

		/* ============================================= */
	}





	/**
	 * Inits the advanced button action
	 */
	private void initAdvancedButtonAction() {
		/* ====================================================== */
		/*
		 * when advanced button is pressed, the advancdPanel will be placed
		 * in the last row of the folderPanel.
		 * The import button is moved to the bottom of the advancedPanel
		 */
		// the state of the advanced mode is stored in isAdvancedMode
		advancedButton.addActionListener(new ActionListener() {
			/* ====================================================== */
			public void actionPerformed(ActionEvent e) {
				/* ============================================= */
				if (isAdvancedMode) {
					/* ------------------------------------------------------ */
					// hide the panel
					isAdvancedMode = false;
					importButton.setVisible(true);
					folderPanel.remove(advancedMetaPanel);
					folderPanel.validate();
					folderPanel.updateUI();
					/* ------------------------------------------------------ */
				} else {
					/* ------------------------------------------------------ */
					// show the panel
					isAdvancedMode = true;
					/* ------------------------------------------------------ */
					// first, hide the import button
					/* ------------------------------------------------------ */
					importButton.setVisible(false);
					/* ------------------------------------------------------ */
					// add the advancedMetaPanel
					folderPanel.add(advancedMetaPanel, cc.xywh(1, 8, 5, 1));
					folderPanel.validate();
					folderPanel.updateUI();
					/* ------------------------------------------------------ */
					/* ------------------------------------------------------ */

					/* ------------------------------------------------------ */
				}
				ImportDrugsDialog.this.pack();
				/* ============================================= */
			}

			/* ====================================================== */
		});


		/* ====================================================== */
	}

	/**
	 * Inits the advanced panel
	 */
	private void initAdvancedPanel() {
		/* ====================================================== */
		this.advancedPanel = new JPanel(new FormLayout(
				// cols
				"3dlu," +
				"fill:pref:grow," +
				"5dlu," +
				"fill:pref," +
				"3dlu",
				// rows
				"3dlu," +		// 1
				"fill:pref," +	// 2
				"3dlu," +		// 3
				"top:pref," +	// 4
				"3dlu," +		// 5
				"top:pref," +	// 6
				"3dlu," +		// 7
				"top:pref," +	// 8
				"3dlu," +		// 9
				"fill:pref," +	// 10
				"3dlu," +		// 11
				"fill:pref," +	// 12
				"3dlu"	 		// 13
				));
		/* ------------------------------------------------------ */
		// make border
		advancedPanel.setBorder(new TitledBorder(
				Translatrix.getTranslationString("cefip.importer.advancedButton")));
		advancedPanel.setOpaque(OPAQUE);
		/* ------------------------------------------------------ */
		// create the textfields etc
		/* ------------------------------------------------------ */
		// labels
		JLabel atcLabel = new JLabel(Translatrix.getTranslationString("cefip.importer.atcCodesSelect"));
		JLabel manuLabel = new JLabel(Translatrix.getTranslationString("cefip.importer.manufacturerSelect"));
		JLabel drugLabel = new JLabel(Translatrix.getTranslationString("cefip.importer.drugSelect"));
		/* ------------------------------------------------------ */
		// textfields
		this.atcTextField = new VTextField("");
		this.manuTextField = new VTextField("");
		this.drugTextField = new VTextField("");
		/* ------------------------------------------------------ */
		// editable false
		atcTextField.setEditable(false);
		manuTextField.setEditable(false);
		drugTextField.setEditable(false);
		/* ------------------------------------------------------ */
		// opaque?
		atcTextField.setOpaque(OPAQUE);
		manuTextField.setOpaque(OPAQUE);
		drugTextField.setOpaque(OPAQUE);
		/* ------------------------------------------------------ */
		// create the filechoser buttons
		/* ------------------------------------------------------ */
		atcButton = new JButton();
		manuButton = new JButton();
		drugButton = new JButton();
		/* ------------------------------------------------------ */
		/* ------------------------------------------------------ */
		// add to the panel

		// atc
		advancedPanel.add(atcLabel, cc.xy(2, 2));
		advancedPanel.add(atcTextField, cc.xy(2, 4));
		advancedPanel.add(atcButton, cc.xy(4, 4));
		/* ------------------------------------------------------ */
		// manufacturer
		advancedPanel.add(manuLabel, cc.xy(2, 6));
		advancedPanel.add(manuTextField, cc.xy(2, 8));
		advancedPanel.add(manuButton, cc.xy(4, 8));
		/* ------------------------------------------------------ */
		// drugs
		advancedPanel.add(drugLabel, cc.xy(2, 10));
		advancedPanel.add(drugTextField, cc.xy(2, 12));
		advancedPanel.add(drugButton, cc.xy(4, 12));
		/* ------------------------------------------------------ */
		// create the metaPanel for the button footer
		this.advancedMetaPanel = new JPanel(new BorderLayout());
		advancedMetaPanel.setOpaque(OPAQUE);
		/* ------------------------------------------------------ */
		// center -> advancedPanel
		advancedMetaPanel.add(advancedPanel, BorderLayout.CENTER);
		/* ------------------------------------------------------ */
		// south -> import button
		this.advancedImportButton = new JButton();
		ButtonBarBuilder bbb = new ButtonBarBuilder();
		bbb.addGlue();
		bbb.addGridded(advancedImportButton);
		bbb.getPanel().setOpaque(OPAQUE);

		this.advancedMetaPanel.add(bbb.getPanel(), BorderLayout.SOUTH);
		/* ------------------------------------------------------ */
		/* ====================================================== */
	}

	/**
	 * Inits the logpanel
	 */
	private void initLogPanel() {
		/* ============================================= */
		this.logPanel = new JPanel(new BorderLayout());
		logPanel.setOpaque(OPAQUE);
		// set a titles border
//		logPanel.setBorder(new TitledBorder(""));
		// add a JTextarea
//		this.logArea = new JTextArea();
//		JScrollPane scroll = new JScrollPane();
//		scroll.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);

//		logPanel.setBorder(scroll.getViewportBorder());

		logPanel.setBorder(new CompoundBorder(new TitledBorder(""),
				new SoftBevelBorder(BevelBorder.LOWERED)));

//		logPanel.add(scroll, BorderLayout.CENTER);
		logPanel.add(listInfoPanel, BorderLayout.CENTER);
		/* ============================================= */
	}



	/**
	 * Init the listinfo panel
	 */
	private void initListInfoPanel() {
		/* ============================================= */
		this.listHeaderPanel = new JPanel(new BorderLayout());
		this.listHeaderPanel.setOpaque(OPAQUE);
		// nice border
		this.listHeaderPanel.setBorder(new LineBorder(borderColor));
		/* ------------------------------------------------------ */

		/* ------------------------------------------------------ */
		/* ------------------------------------------------------ */
		// the message icon
		JLabel listInfoIconLabel = new JLabel(PrescriptionIcons.getIcon(PrescriptionIcons.INFORMATION));
		listInfoIconLabel.setVerticalAlignment(JLabel.TOP);
		listHeaderPanel.add(listInfoIconLabel, BorderLayout.WEST);
		/* ------------------------------------------------------ */
		JLabel msg = new JLabel("Info");
		msg.setFont(new Font(msg.getFont().getName(), Font.BOLD, 15));
		listHeaderPanel.add(msg, BorderLayout.CENTER);
		/* ------------------------------------------------------ */


		this.listInfoPanel = new JPanel(new FormLayout(
				// cols
				"3dlu," +
				"7dlu," +
				"fill:pref," +
				"3dlu," +
				"fill:pref," +
				"fill:pref:grow," +
				"fill:pref," +
				"3dlu",
				// rows
				"3dlu," +
				"fill:pref," +
				"5dlu," +
				"fill:pref," +
				"3dlu," +
				"fill:pref," +
				"3dlu," +
				"fill:pref," +
				"fill:pref:grow"));

		listInfoPanel.setOpaque(OPAQUE);
		// north, text
		JPanel listInfoNorthPanel = new JPanel(new GridLayout(1, 0));
		listInfoNorthPanel.setOpaque(OPAQUE);
		// a label
		this.listInfoDateLabel = new JLabel();
		listInfoNorthPanel.add(this.listInfoDateLabel);

		listInfoPanel.add(listInfoNorthPanel, cc.xywh(2, 2, 6, 1));

		/* ------------------------------------------------------ */
//		// the list stats
//		JPanel listStatPanel = new JPanel(new FormLayout("3dlu," +
//														 "fill:pref," +
//														 "fill:pref:grow," +
//														 "fill:pref," +
//														 "3dlu",
//														 // rows
//														 "3dlu," +
//														 "fill:pref," +
//														 "3dlu," +
//														 "fill:pref," +
//														 "3dlu," +
//														 "fill:pref," +
//														 "3dlu"));
//		/* ------------------------------------------------------ */
		JLabel atcLabel = new JLabel(
						Translatrix.getTranslationString("cefip.importer.atcCodes"));
		atcLabel.setIcon(PrescriptionIcons.getMiniIcon(PrescriptionIcons.NEXT));
		// the number label
		this.atcCountLabel = new JLabel();
		listInfoPanel.add(atcLabel, cc.xy(3, 4));
		listInfoPanel.add(atcCountLabel, cc.xy(5, 4));
		/* ------------------------------------------------------ */
		// the manufacturer
		JLabel manuLabel = new JLabel(
						Translatrix.getTranslationString("cefip.importer.manufacturer"));
		manuLabel.setIcon(PrescriptionIcons.getMiniIcon(PrescriptionIcons.NEXT));
		// the number label
		this.manuCountLabel = new JLabel();
		listInfoPanel.add(manuLabel, cc.xy(3, 6));
		listInfoPanel.add(manuCountLabel, cc.xy(5, 6));
		/* ------------------------------------------------------ */
		// the drugs
		JLabel drugLabel = new JLabel(
						Translatrix.getTranslationString("cefip.importer.drugs"));
		drugLabel.setIcon(PrescriptionIcons.getMiniIcon(PrescriptionIcons.NEXT));
		// the number label
		this.drugCountLabel = new JLabel();
		listInfoPanel.add(drugLabel, cc.xy(3, 8));
		listInfoPanel.add(drugCountLabel, cc.xy(5, 8));


		/* ------------------------------------------------------ */
		/* ============================================= */
	}


	class CefipFileFilter extends FileFilter {

		@Override
		public boolean accept(File f) {
			/* ====================================================== */
			if (f.isDirectory())
				return true;
			
			String extension = FilenameUtils.getExtension(f.getName());
			if (extension != null) {
				if ("MDB".equals(extension.trim().toUpperCase()))
					return true;
			}
			return false;
			/* ====================================================== */
		}

		@Override
		public String getDescription() {
			/* ====================================================== */
			return "Folders and Microsoft Access files";
			/* ====================================================== */
		}
		
	}



}
