/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.usermanagement.ejb.session.interfaces;

import java.util.Collection;
import java.util.List;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Remote;

import lu.tudor.santec.gecamed.core.utils.TimeZoneCheck;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.GecamedInfo;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.GecamedUser;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.Permission;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.Role;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.Setting;
import lu.tudor.santec.settings.SettingReader;

	/**
	 * This Bean-Class Manages the Login and Authentification into JBoss it provides
	 * Methods to retrieve rights, rightinfo and userinfo on the User which is
	 * currently logged in.
	 * 
	 * @author Johannes Hermen
	 * 
	 * @Version
     * <br>$Log: LoginInterface.java,v $
     * <br>Revision 1.23  2013-05-23 14:27:14  ferring
     * <br>Conflicting time zone detection - changing client time zone to server time zone
     * <br>
     * <br>Revision 1.22  2013-01-09 08:29:18  ferring
     * <br>Errors in auto backup caught
     * <br>
     * <br>Revision 1.21  2012-02-07 10:39:23  ferring
     * <br>Warn if server has wrong Java version
     * <br>
     * <br>Revision 1.20  2010-04-27 15:12:58  mack
     * <br>Added getModulePermissions
     * <br>
     * <br>Revision 1.19  2010-04-20 07:24:52  hermen
     * <br>added show server log to tools menu
     * <br>
     * <br>Revision 1.18  2009-02-27 15:25:56  hermen
     * <br>added button to create an empty incident for a patient visit
     * <br>
     * <br>Revision 1.17  2009-02-04 10:15:55  hermen
     * <br>added functionality to get all Users that are logged in on a GECAMed Server
     * <br>
     * <br>Revision 1.16  2008-10-07 09:34:37  heinemann
     * <br>translations of roles and rights;
     * <br>show only  roles and rights of the activated modules in the usermanagement section
     * <br>
     * <br>Revision 1.15  2008-09-25 09:43:08  heinemann
     * <br>fixed copyrights
     * <br>
     * <br>Revision 1.14  2008-01-15 12:58:13  hermen
     * <br>added getAllUsers
     * <br>
     * <br>Revision 1.13  2008-01-15 09:30:14  hermen
     * <br>updated Javadoc and refactured code
     * <br>
	 *
	 */
	
	@Remote
	public interface LoginInterface {

	/**
	 * Returns a GecamedUser Object of the current User
	 *
	 * @return a GecamedUser Object representing the current User
	 * @author Johannes Hermen
	 * @updated 08.12.2005 Martin Heineman
	 * 
	 */
	@RolesAllowed("gecam")
	public abstract GecamedUser getCurrentUser();

	/**
	 * Returns the User ID of the current User
	 *
	 * @return the User ID representing the current User
	 * @author Johannes Hermen
	 * @updated 04.10.2005 Martin Heineman
	 */
//	@RolesAllowed("gecam")
	public abstract Integer getCurrentUserID();

	/**
	 * Returns the Username of the current User
	 *
	 * @return the Username of the current User
	 * @author Johannes Hermen
	 * @updated 04.10.2005 Martin Heineman
	 */
//	@RolesAllowed("gecam")
	public abstract String getCurrentUserName();

	/**
	 * returns a String with all existing user-logins seperated by ","
	 *
	 * @return a String with all existing user-logins seperated by ","
	 * @author Johannes Hermen
	 * @updated 04.10.2005 Martin Heinemann
	 */
	@RolesAllowed("gecam")
	public abstract String getUserLoginList();

	/**
	 * Returns the Setting Object of the current User
	 *
	 * @return the SettingLightValue Object of the current User
	 * @author Johannes Hermen
	 * @updated 04.10.2005 Martin Heinemann
	 */
	@RolesAllowed("gecam")
	public abstract Setting getSettings();

	/**
	 * Returns a GecamedUser Object of the specified User
	 *
	 * @param staffID the Id of the wanted User
	 * @return a StaffValue Object representing the wanted User
	 * @author Johannes Hermen
	 * @updated 04.10.2005 Martin Heinemann
	 */
	@RolesAllowed("gecam")
	public abstract GecamedUser getUser(Integer id);


	/**
	 * Returns a list of all permissions for the given module
	 * 
	 * @param p_ModuleName the name of the module
	 * @return list of all permissions for the given module
	 */
	@RolesAllowed("gecam")
	public List <Permission> getModulePermissions (String p_ModuleName);
	
	/**
	 * Fetches names of all permissions defined for specified module and granted to specified user
	 * @param p_ModuleName : specifies the name of the module to get permissions of
	 * @param p_UserId     : specifies the id of the user to get permissions for
	 * @return a list of permissions defined for the specified module and granted to the specified
	 * user. If no permissions were defined or none were granted to user, <code>null</code> will
	 * be returned.
	 */

	@RolesAllowed("gecam")
	public Collection <String> getModulePermissions (String p_ModuleName, Integer p_UserId);
	
	/**
	 * Returns the roles offered by this module
	 * 
	 * @param moduleName
	 * @return
	 */
	
	@RolesAllowed("gecam")
	public List<Role> getModuleRoles (String moduleName);
	
	/**
	 * Fetches names of all roles defined for specified module and granted to specified user
	 * @param p_ModuleName : specifies the name of the module to get roles of
	 * @param p_UserId     : specifies the id of the user to get roles for
	 * @return a list of roles defined for the specified module and granted to the specified
	 * user. If no roles were defined or none were granted to user, <code>null</code> will
	 * be returned.
	 */
	
	@RolesAllowed("gecam")
	public Collection<String> getModuleRoles (String p_ModuleName, Integer p_UserId);

	/**
	 * checks if the given user has the given permission
	 * @param p_User the GECAMed User
	 * @param p_PermissionId the id of the permission
	 * @return true if user has permission
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public abstract Boolean userHasPermission (GecamedUser p_User, Integer p_PermissionId) throws Exception;

	/**
	 * checks if the current user has the given permission
	 * @param p_PermissionId the id of the permission
	 * @return true if user has permission
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public abstract Boolean userHasPermission (Integer p_PermissionId) throws Exception;
	
	/**
	 * checks if the given user has the given permission
	 * @param p_User the GECAMed User
	 * @param p_PermissionName the name of the permission
	 * @return true if user has permission
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public abstract Boolean userHasPermission (GecamedUser p_User, String p_PermissionName) throws Exception;
	
	/**
	 * checks if the current user has the given permission
	 * @param p_PermissionName the name of the permission
	 * @return true if user has permission
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public abstract Boolean userHasPermission (String p_PermissionName) throws Exception;
	
	/**
	 * checks if the given user has the given role
	 * @param  p_User the GECAMed User
	 * @param p_RoleId p_PermissionName the name of the role
	 * @return true if user has role
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public Boolean userHasRole (GecamedUser p_User, Integer p_RoleId) throws Exception;
	
	/**
	 * checks if the current user has the given role
	 * @param p_RoleId the id of the role
	 * @return true if user has role
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public Boolean userHasRole (Integer p_RoleId) throws Exception;
	
	/**
	 * checks if the given user has the given role
	 * @param p_RoleId  the name of the role
	 * @return true if user has role
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public Boolean userHasRole (GecamedUser p_User, String p_Role) throws Exception;
	
	/**
	 * checks if the current user has the given role
	 * @param p_RoleId  the name of the role
	 * @return true if user has role
	 * @throws Exception
	 */
	@RolesAllowed("gecam")
	public Boolean userHasRole (String p_Role) throws Exception;
	
	/**
	 * saves the Setting Object to the current User
	 *
	 * @param setting the Setting Object for the current User
	 * @author Johannes Hermen
	 * @updated 07.10.2005 Martin Heinemann
	 * 
	 */
	@RolesAllowed("gecam")
	public abstract Setting saveSettings(Setting setting);

	
	/**
	 * returns the global system settings as Setting object
	 * @return  the global system settings as Setting object
	 */
	@RolesAllowed("gecam")
	public abstract Setting getGlobalSettings();

	/**
	 * saves the global setting object
	 * @param setting the global setting object
	 */
	@RolesAllowed("gecam")
	public abstract Setting saveGlobalSettings(Setting setting);
	
//	/**
//	 * returns a SettingReader from which the users settings can be read
//	 * @return a SettingReader from which the users settings can be read
//	 */
//	@RolesAllowed("gecam")
//	public abstract SettingReader getSettingsReader();
	
	/**
	 * returns a SettingReader from which global settings can be read
	 * @return a SettingReader from which global settings can be read
	 */
//	@RolesAllowed("gecam")
	public abstract SettingReader getAdminSettingsReader();
	
	
	/**
	 * returns the user settings value for the given setting and settings plugin
	 * @param plugin the settings plugin to obtain the setting from
	 * @param setting the name of the setting to retrieve
	 * @return the setting value as an object
	 */
	@RolesAllowed("gecam")
	public Object getSettingValue(String plugin, String setting);

	/**
	 * returns the admin settings value for the given setting and settings plugin
	 * @param plugin the admin settings plugin to obtain the setting from
	 * @param setting the name of the setting to retrieve
	 * @return the admin setting value as an object
	 */
	@RolesAllowed("gecam")
	public Object getAdminSettingValue(String plugin, String setting);
	
	@RolesAllowed("gecam")
	public Object getAdminSettingValue(String plugin, String setting, Object defaultVal);
	
	/**
	 * returns a list of all GECAMedInfo entries with the specified key
	 * @param key
	 * @return a list of all GECAMedInfo entries with the specified key
	 */
	@RolesAllowed("gecam")
	public Collection<GecamedInfo> getInfos(String key)  throws Exception;
	
	/**
	 * returns the latest GECAMedInfo entries with the specified key
	 * @param key
	 * @return a list of all
	 */
	@RolesAllowed("gecam")
	public GecamedInfo getInfo(String key) throws Exception;
	
	/**
	 * Stores a GECAMedInfo object to the db
	 * @param info
	 */
	@RolesAllowed("gecam")
	public void setInfo(GecamedInfo info);
	
	/**
	 * gives some information about the database jboss is connected to
	 * @return some information about the database jboss is connected to
	 */
	@RolesAllowed("gecam")
	public String getDatabaseInfo();
	
	/**
	 * gives some information about the database jboss is connected to
	 * @return some information about the database jboss is connected to
	 */
	@RolesAllowed("gecam")
	public String getDatabaseServerVersion();
	
	/**
	 * returns a Collection of all GecamedUsers
	 * @return a Collection of all GecamedUsers
	 */
	@RolesAllowed("gecam")
	public Collection<GecamedUser> getAllUsers();
	
	/**
	 * sends a keepalive that the user is still logged in
	 * @param user the current user
	 * @param site 
	 */
	public void iAmOnline(GecamedUser user);
	
	/**
	 * sends a keepalive that the user is still logged in
	 * @param user the current user
	 */
	public void iAmOffline(GecamedUser user);
	
	/**
	 * returns all users that are currently logged in
	 * @param siteID 
	 * @return list of online users
	 */
	public Collection<GecamedUser> getOnlineUsers(Integer siteID);
	
	public String getServerLog();
	
	public String getJavaVersion();
	
	public TimeZoneCheck getTimeZoneCheck();
}
